/*  MPI_factorDPencil.c  */

#include "../spoolesMPI.h"
#include "../../timings.h"

#define MYDIAGNOSTIC 1

/*--------------------------------------------------------------------*/
typedef struct _FactorMsg FactorMsg ;
struct _FactorMsg {
   int           info[3] ;
   void          *base   ;
   DChv          *chv    ;
   MPI_Request   req     ;
   FactorMsg     *next   ;
} ;
/*--------------------------------------------------------------------*/
static FactorMsg * FactorMsg_new ( void ) ;
static void FactorMsg_setDefaultFields ( FactorMsg *msg ) ;
static void FactorMsg_clearData ( FactorMsg *msg ) ;
static void FactorMsg_free ( FactorMsg *msg ) ;
static FactorMsg * wakeupFront ( DFrontMtx *frontmtx, int J, int myid, 
   int frontOwners[], int firsttag, DChvManager *manager, 
   DChvList *aggList, int stats[], MPI_Comm comm, 
   int msglvl, FILE *msgFile ) ;
static FactorMsg * checkMessages ( DFrontMtx *frontmtx, int J,
   FactorMsg *firstmsg, DChvList *aggList, DChvList *postList,
   DChvManager *manager, int firsttag, int stats[], MPI_Comm comm,
   int msglvl, FILE *msgFile ) ;
static FactorMsg * sendMessages ( int J, int myid, int nfront,
   int par[], int owners[], int firsttag, DChvManager*manager,
   DChvList *aggList, DChvList *postList, FactorMsg *firstmsgsent,
   int stats[], MPI_Comm comm, int msglvl, FILE *msgFile ) ;
static FactorMsg * checkSentMessages ( FactorMsg *firstmsgsent,
   DChvManager *manager, int msglvl, FILE *msgFile ) ;
/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   this is the method that computes the 
   parallel factorization of a sparse matrix A using MPI.

   input ---

      frontmtx      -- front matrix object that holds the factors
      inpmtx        -- matrix object for A
      tau           -- tolerance used in pivoting
      droptol       -- tolerance used in the approximate factorization
      firsttag      -- first tag to use during the factorization,
                       reserved tags are tag, ..., tag + 4*nfront + 2
      frontOwnersIV -- map from fronts to owning processes
      lookahead     -- parameter that governs the ``lookahead''
                       behavior, use lookahead = 0 as a default
      cpus          -- vector to store breakdown of cpu times
         cpus[ 0] -- initialize fronts
         cpus[ 1] -- load original entries
         cpus[ 2] -- update fronts
         cpus[ 3] -- insert aggregate data
         cpus[ 4] -- assemble aggregate data
         cpus[ 5] -- assemble postponed data
         cpus[ 6] -- factor fronts
         cpus[ 7] -- extract postponed data
         cpus[ 8] -- store factor entries
         cpus[ 9] -- post initial receives
         cpus[10] -- check for received messages
         cpus[11] -- post initial sends
         cpus[12] -- check for sent messages
      stats         -- vector to store statistics
         stats[ 0] -- # of aggregate sends
         stats[ 1] -- # of bytes in the aggregate sends
         stats[ 2] -- # of aggregate received
         stats[ 3] -- # of bytes in the aggregate received
         stats[ 4] -- # of postponed data sends
         stats[ 5] -- # of bytes in the postponed data sends
         stats[ 6] -- # of postponed data received
         stats[ 7] -- # of bytes in the postponed data received
         stats[ 8] -- # of active DChv objects (working storage)
         stats[ 9] -- # of active bytes in working storage
         stats[10] -- # of requested bytes in working storage
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator

   return value -- if process id is zero, a pointer to the first
   DChv object in a list that contains postponed rows and columns
   that could not be eliminated.

   created  -- 97nov17, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MPI_factorDInpMtx (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtx,
   double      tau,
   double      droptol,
   int         firsttag,
   IV          *frontOwnersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) {
DChv      *rootchv ;
DPencil   *pencil ;
int   flag, tag_bound ;

MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
if ( firsttag + 4*frontmtx->nfront + 2 > tag_bound ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MPI_factorDInpMtx()"
           "\n tag range is [%d,%d], tag_bound = %d",
           firsttag, firsttag + 4*frontmtx->nfront + 2, tag_bound) ;
   exit(-1) ;
}
pencil = DPencil_new() ;
DPencil_init(pencil, inpmtx, 0.0, NULL) ;
rootchv = DFrontMtx_MPI_factorDPencil(frontmtx, pencil, tau, droptol,
                                   firsttag, frontOwnersIV, lookahead, 
                                   cpus, stats, msglvl, msgFile, comm) ;
DPencil_free(pencil) ;

return(rootchv) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   this is the method that computes the 
   parallel factorization of A + sigma*B using MPI.

   input ---

      frontmtx      -- front matrix object that holds the factors
      pencil        -- matrix pencil object, contains A + sigma*B
      tau           -- tolerance used in pivoting
      droptol       -- tolerance used in the approximate factorization
      firsttag      -- first tag to use during the factorization,
                       reserved tags are tag, ..., tag + 4*nfront + 2
      frontOwnersIV -- map from fronts to owning processes
      lookahead     -- parameter that governs the ``lookahead''
                       behavior, use lookahead = 0 as a default
      cpus          -- vector to store breakdown of cpu times
         cpus[ 0] -- initialize fronts
         cpus[ 1] -- load original entries
         cpus[ 2] -- update fronts
         cpus[ 3] -- insert aggregate data
         cpus[ 4] -- assemble aggregate data
         cpus[ 5] -- assemble postponed data
         cpus[ 6] -- factor fronts
         cpus[ 7] -- extract postponed data
         cpus[ 8] -- store factor entries
         cpus[ 9] -- post initial receives
         cpus[10] -- check for received messages
         cpus[11] -- post initial sends
         cpus[12] -- check for sent messages
      stats         -- vector to store statistics
         stats[ 0] -- # of aggregate sends
         stats[ 1] -- # of bytes in the aggregate sends
         stats[ 2] -- # of aggregate received
         stats[ 3] -- # of bytes in the aggregate received
         stats[ 4] -- # of postponed data sends
         stats[ 5] -- # of bytes in the postponed data sends
         stats[ 6] -- # of postponed data received
         stats[ 7] -- # of bytes in the postponed data received
         stats[ 8] -- # of active DChv objects (working storage)
         stats[ 9] -- # of active bytes in working storage
         stats[10] -- # of requested bytes in working storage
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator

   return value -- if process id is zero, a pointer to the first
   DChv object in a list that contains postponed rows and columns
   that could not be eliminated.

   created  -- 97jul11, cca
   modified -- 97nov13, cca
      blocking communication replaced with nonblocking communication
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MPI_factorDPencil (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   int         firsttag,
   IV          *frontOwnersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) {
char          *status, *willUpdate ;
DChv          *rootchv ;
DChvList      *aggList, *postList ;
DChvManager   *manager ;
DFactorData   *data ;
double        t1, t2 ;
DV            *tmpDV ;
FactorMsg     *firstmsgsent ;
FactorMsg     **p_messages ;
Ideq          *dequeue ;
int           flag, J, K, myid, nfront, nproc, tag, tag_bound ;
int           *head, *link, *nactiveChild, *offsets, *owners, *par ;
int           commstats[4] ;
IV            *markIV, *pivotsizesIV ;
/*
   --------------
   check the data
   --------------
*/
if ( frontmtx == NULL || pencil == NULL || tau < 1.0 || droptol < 0.0 
   || frontOwnersIV == NULL || cpus == NULL || stats == NULL
   || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MPI_factorDPencil()"
           "\n frontmtx = %p, pencil = %p, tau = %f, droptol = %f"
           "\n frontOwnersIV = %p, firsttag = %d"
           "\n cpus = %p, stats = %p, msglvl = %d, msgFile = %p"
           "\n bad input\n",
           frontmtx, pencil, tau, droptol, frontOwnersIV, firsttag,
           cpus, stats, msglvl, msgFile) ;
   exit(-1) ;
}
MPI_Comm_rank(comm, &myid) ;
MPI_Comm_size(comm, &nproc) ;
nfront = frontmtx->nfront ;
par    = ETree_par(frontmtx->frontETree) ;
owners = IV_entries(frontOwnersIV) ;
tag    = firsttag ;
MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
if ( firsttag < 0 || firsttag + 4*nfront + 2 > tag_bound ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MPI_factorDPencil()"
           "\n tag range is [%d,%d], tag_bound = %d",
           firsttag, firsttag + 4*nfront + 2, tag_bound) ;
   exit(-1) ;
}
/*
   ------------------------------------------------------------------
   create a DChvManager object to handle DChv instances. 
   the object does not need a lock since it is local to this process.
   ------------------------------------------------------------------
*/
manager = DChvManager_new() ;
DChvManager_init(manager, 0) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n manager = %p, firsttag = %d", 
           manager, firsttag) ;
   fflush(msgFile) ;
}
if ( frontmtx->pivotingflag == 1 ) {
/*
   --------------------------------------------
   create a DChvList object to handle the 
   postponed data.  it does not require a lock.
   --------------------------------------------
*/
   postList = DFrontMtx_postList(frontmtx, frontOwnersIV, 0) ;
} else {
   postList = NULL ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n postList = %p", postList) ;
   if ( postList != NULL ) {
      if ( postList->counts != NULL ) {
         fprintf(msgFile, "\n counts") ;
         IVfprintf(msgFile, postList->nlist, postList->counts) ;
      }
      fflush(msgFile) ;
   }
}
/*
   --------------------------------------------
   create a DChvList object to handle aggregate 
   fronts.  it does not need a lock either.
   --------------------------------------------
*/
aggList = DFrontMtx_MPI_aggregateList(frontmtx, frontOwnersIV, tag, 
                                     commstats, msglvl, msgFile, comm) ;
tag += 2 ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n aggList = %p", aggList) ;
   fprintf(msgFile, "\n aggregate counts") ;
   IVfprintf(msgFile, nfront, aggList->counts) ;
   fflush(msgFile) ;
}
/*
   -----------------------------
   set up the DFactorData object
   -----------------------------
*/
data = DFactorData_new() ;
DFactorData_init(data, frontmtx, pencil, tau, droptol, frontOwnersIV, 
                 lookahead, manager, aggList, postList) ;
DFactorData_setInfo(data, myid, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n data = %p", data) ;
   fprintf(msgFile, "\n data->fronts = %p", data->fronts) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   willUpdate[J] == 'Y' --> this process owns at 
   least one descendent of J that has support with J
   -------------------------------------------------
*/
willUpdate = DFrontMtx_willUpdate(frontmtx, frontOwnersIV, myid) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n willUpdate = %p", willUpdate) ;
   CVfprintf(msgFile, nfront, willUpdate) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   status[J] == 'F' --> J lies on no active path 
   status[J] == 'W' --> J does lie on an active path 
   -------------------------------------------------
*/
status = DFrontMtx_status(frontmtx, frontOwnersIV, myid) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n willUpdate = %p", willUpdate) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------------------
   nactiveChild[J] = # of children of J that lie on active paths
   -------------------------------------------------------------
*/
nactiveChild = DFrontMtx_nactiveChild(frontmtx, status, myid) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n nactiveChild = %p", nactiveChild) ;
   fflush(msgFile) ;
}
/*
   ---------------------------------------------------------
   load a Ideq structure with the leaves of the active paths
   ---------------------------------------------------------
*/
dequeue = DFrontMtx_setUpDequeue(frontmtx, 
                                 frontOwnersIV, status, myid) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n dequeue = %p", dequeue) ;
   fflush(msgFile) ;
}
/*
   ----------------------------
   allocate the working storage
   ----------------------------
*/
head    = IVinit(nfront, -1) ;
link    = IVinit(nfront, -1) ;
offsets = IVinit(nfront,  0) ;
if ( frontmtx->pivotingflag == 1 ) {
   markIV = IV_new() ;
   if ( frontmtx->symmetryflag == 0 ) {
      pivotsizesIV = IV_new() ;
   } else {
      pivotsizesIV = NULL ;
   }
} else {
   markIV = pivotsizesIV = NULL ;
}
tmpDV = DV_new() ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n head = %p", head) ;
   fprintf(msgFile, "\n link = %p", link) ;
   fprintf(msgFile, "\n offsets = %p", offsets) ;
   fprintf(msgFile, "\n markIV = %p", markIV) ;
   fprintf(msgFile, "\n pivotsizesIV = %p", pivotsizesIV) ;
   fprintf(msgFile, "\n tmpDV = %p", tmpDV) ;
   fflush(msgFile) ;
}
ALLOCATE(p_messages, struct _FactorMsg *, nfront+1) ;
for ( J = 0 ; J <= nfront ; J++ ) {
   p_messages[J] = NULL ;
}
/*
   ---------------------------
   loop while a path is active
   ---------------------------
*/
firstmsgsent = NULL ;
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n ### checking out front %d", J) ;
      fflush(msgFile) ;
   }
   MARKTIME(t1) ;
   if ( status[J] == 'W' ) {
      p_messages[J] = wakeupFront(frontmtx, J, myid, owners, firsttag,
                                  manager, aggList, stats, 
                                  comm, msglvl, msgFile) ;
   }
   MARKTIME(t2) ;
   cpus[10] += t2 - t1 ;
/*
   ---------------------------------------
   check for received messages for front J
   ---------------------------------------
*/
   MARKTIME(t1) ;
   p_messages[J] = checkMessages(frontmtx, J, p_messages[J], aggList, 
                                 postList, manager, firsttag, 
                                 stats, comm, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[11] += t2 - t1 ;
/*
   ----------------------------------
   check for numeric work for front J
   ----------------------------------
*/
   DFrontMtx_checkFront(frontmtx, data, J, lookahead, status,
                        willUpdate, nactiveChild, head, link,
                        offsets, pivotsizesIV, markIV, tmpDV) ;
   if ( status[J] == 'F' ) {
#if MYDIAGNOSTIC > 0
{
      double   t1 ;
      MARKTIME(t1) ;
      fprintf(stdout, 
              "\n proc %d, time %8.3f : front %d is finished", 
              myid, t1, J) ;
      fflush(stdout) ;
      fprintf(msgFile, 
              "\n proc %d, time %8.3f : front %d is finished", 
              myid, t1, J) ;
      fflush(msgFile) ;
}
#endif
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n front %d is finished", J) ;
         fflush(msgFile) ;
      }
      if ( (K = par[J]) != -1 && nactiveChild[K] == 0 ) {
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n\n adding front %d to dequeue", K) ;
            fflush(msgFile) ;
         }
         Ideq_insertAtHead(dequeue, K) ;
      }
/*
      ------------------------------------
      send any messages to other processes
      ------------------------------------
*/
      MARKTIME(t1) ;
      firstmsgsent = sendMessages(J, myid, nfront, par, owners, 
                                  firsttag, manager, aggList, postList,
                                  firstmsgsent, stats,
                                  comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[12] += t2 - t1 ;
   } else {
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n front %d is not finished", J) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtTail(dequeue, J) ;
   }
/*
   -------------------------------
   check the list of sent messages
   -------------------------------
*/
   MARKTIME(t1) ;
   firstmsgsent = checkSentMessages(firstmsgsent, manager,
                                    msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[13] += t2 - t1 ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n done with main loop") ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------------
   if process zero, wait until notification from all 
   the roots that the factorization has finished.
   -------------------------------------------------
*/
rootchv = NULL ;
if ( myid == 0 && postList != NULL ) {
   if ( msglvl > 0 ) {
      fprintf(msgFile, "\n\n receive notification from roots") ;
      fprintf(msgFile, "\n count = %d",
              postList->counts[nfront]) ;
      fflush(msgFile) ;
   }
   p_messages[nfront] = wakeupFront(frontmtx, nfront, myid, owners, 
                                    firsttag, manager, aggList, stats, 
                                    comm, msglvl, msgFile) ;
   while ( 1 ) {
      MARKTIME(t1) ;
      p_messages[nfront] = checkMessages(frontmtx, nfront, 
                                 p_messages[nfront], aggList, 
                                 postList, manager, firsttag, 
                                 stats, comm, msglvl, msgFile) ;
      MARKTIME(t2) ;
      cpus[11] += t2 - t1 ;
      if ( DChvList_isCountZero(postList, nfront) == 1 ) {
         break ;
      }
   }
   rootchv = DChvList_getList(postList, nfront) ;
}
DVcopy(10, cpus, data->cpus) ;
stats[8]  = manager->nactive ;
stats[9]  = manager->nbytesactive ;
stats[10] = manager->nbytesrequested ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
DChvManager_free(manager) ;
if ( postList != NULL ) {
   DChvList_free(postList) ;
}
DChvList_free(aggList) ;
CVfree(status) ;
CVfree(willUpdate) ;
IVfree(nactiveChild) ;
IVfree(head) ;
IVfree(link) ;
IVfree(offsets) ;
Ideq_free(dequeue) ;
if ( markIV != NULL ) {
   IV_free(markIV) ;
}
if ( pivotsizesIV != NULL ) {
   IV_free(pivotsizesIV) ;
}
DV_free(tmpDV) ;
FREE(p_messages) ;

return(rootchv) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------
   initialize a new instance
 
   created -- 97nov13, cca
   -------------------------
*/
static FactorMsg *
FactorMsg_new (
   void
) {
FactorMsg   *msg ;

ALLOCATE(msg, struct _FactorMsg, 1) ;
FactorMsg_setDefaultFields(msg) ;

return(msg) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set the default fields
 
   created -- 97nov13, cca
   -----------------------
*/
static void
FactorMsg_setDefaultFields (
   FactorMsg   *msg
) {
msg->info[0] =   0  ;
msg->info[1] =   0  ;
msg->info[2] =   0  ;
msg->base    = NULL ;
msg->chv     = NULL ;
msg->req     = NULL ;
msg->next    = NULL ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   clear the data
 
   created -- 97nov13, cca
   -----------------------
*/
static void
FactorMsg_clearData (
   FactorMsg   *msg
) {
FactorMsg_setDefaultFields(msg) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   free the object
 
   created -- 97nov13, cca
   -----------------------
*/
static void
FactorMsg_free (
   FactorMsg   *msg
) {
FactorMsg_clearData(msg) ;
FREE(msg) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------
   when a front is initially active, 
   post its necessary receive messages

   created -- 97nov13, cca
   -----------------------------------
*/
static FactorMsg *
wakeupFront ( 
   DFrontMtx     *frontmtx,
   int           J,
   int           myid,
   int           frontOwners[],
   int           firsttag,
   DChvManager   *manager,
   DChvList      *aggList,
   int           stats[],
   MPI_Comm      comm,
   int           msglvl,
   FILE          *msgFile
) {
FactorMsg   *first, *msg ;
int         aggcount, ii, I, nfront, tag, type ;
int         *fch, *sib ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J > (nfront = frontmtx->nfront)
   || manager == NULL || aggList == NULL
   || (msglvl > 1 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in wakeupFront()"
           "\n bad input\n") ;
   exit(-1) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n inside wakeupFront, J = %d", J) ;
   fflush(msgFile) ;
}
first = NULL ;
if ( J < frontmtx->nfront ) {
   if ( (aggcount = aggList->counts[J]) > 0 ) {
/*
      ------------------------------------------------------------
      post notification receives for the incoming aggregate fronts
      ------------------------------------------------------------
*/
      tag  = firsttag + J ;
      type = 1 ;
      for ( ii = 0 ; ii < aggcount ; ii++ ) {
         msg          = FactorMsg_new() ;
         msg->info[0] =   1   ;
         msg->info[1] =   J   ;
         msg->info[2] =   0   ;
         msg->next    = first ;
         first        = msg   ;
         stats[2]++ ;
         stats[3] += 3*sizeof(int) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
                "\n 1. posting Irecv, msg %p, type = 1, J = %d, tag = %d",
                msg, J, tag) ;
            fflush(msgFile) ;
         }
         MPI_Irecv((void *) msg->info, 3, MPI_INT, MPI_ANY_SOURCE, tag, 
                   comm, &msg->req) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, ", return") ;
            fflush(msgFile) ;
         }
      }
   }
   if ( frontmtx->pivotingflag != 0 && frontOwners[J] == myid ) {
/*
      ---------------------------------------------
      post receives for child notification messages
      ---------------------------------------------
*/
      fch = frontmtx->frontETree->tree->fch ;
      if ( fch[J] != -1 ) {
         sib  = frontmtx->frontETree->tree->sib ;
         for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
            if ( frontOwners[I] != myid ) {
               msg        = FactorMsg_new() ;
               msg->info[0] =   3   ;
               msg->info[1] =   I   ;
               msg->info[2] =   0   ;
               msg->next    = first ;
               first        = msg   ;
               stats[6]++ ;
               stats[7] += 3*sizeof(int) ;
               if ( msglvl > 1 ) {
                  fprintf(msgFile, 
                 "\n 1. posting Irecv, msg %p, type = 3, J = %d, tag = %d",
                 msg, J, firsttag + 2*nfront + I) ;
                  fflush(msgFile) ;
               }
               MPI_Irecv((void *) msg->info, 3, MPI_INT, frontOwners[I], 
                         firsttag + 2*nfront + I, comm, &msg->req) ;
               if ( msglvl > 1 ) {
                  fprintf(msgFile, ", return") ;
                  fflush(msgFile) ;
               }
            }
         }
      }
   }
} else {
/*
      --------------------------------------------------
      post receives for notification messages from roots
      --------------------------------------------------
*/
   sib = frontmtx->frontETree->tree->sib ;
   for ( I = frontmtx->frontETree->tree->root ;
         I != -1 ;
         I = sib[I] ) {
      if ( frontOwners[I] != myid ) {
         msg        = FactorMsg_new() ;
         msg->info[0] =   3   ;
         msg->info[1] =   I   ;
         msg->info[2] =   0   ;
         msg->next    = first ;
         first        = msg   ;
         stats[6]++ ;
         stats[7] += 3*sizeof(int) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
           "\n 1. posting Irecv, msg %p, type = 3, J = %d, tag = %d",
           msg, J, firsttag + 2*nfront + I) ;
            fflush(msgFile) ;
         }
         MPI_Irecv((void *) msg->info, 3, MPI_INT, frontOwners[I], 
                   firsttag + 2*nfront + I, comm, &msg->req) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, ", return") ;
            fflush(msgFile) ;
         }
      }
   }
}
return(first) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------
   check for completion of messages for front J

   created -- 97nov13, cca
   --------------------------------------------
*/
static FactorMsg *
checkMessages (
   DFrontMtx     *frontmtx,
   int           J,
   FactorMsg     *firstmsg,
   DChvList      *aggList,
   DChvList      *postList,
   DChvManager   *manager,
   int           firsttag,
   int           stats[],
   MPI_Comm      comm,
   int           msglvl,
   FILE          *msgFile
) {
FactorMsg    *msg, *nextmsg ;
int          flag, nbytes, nfront, source, tag, type ;
MPI_Status   status ;
/*
   ----------------------------------
   loop over the messages in the list
   ----------------------------------
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n inside checkMessages, J = %d", J) ;
   fflush(msgFile) ;
}
nfront = frontmtx->nfront ;
for ( msg = firstmsg, firstmsg = NULL ; 
      msg != NULL ; 
      msg = nextmsg ) {
/*
   ---------------------------
   set next message to look at
   ---------------------------
*/
   nextmsg   = msg->next ;
   msg->next = NULL ;
/*
   --------------------------------------
   extract the type, front id and # bytes
   --------------------------------------
*/
   type   = msg->info[0] ;
   nbytes = msg->info[2] ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n checking msg %p : type %d, front %d, nbytes %d",
              msg, type, msg->info[1], nbytes) ;
      fflush(msgFile) ;
   }
   MPI_Test(&msg->req, &flag, &status) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, ", flag = %d", flag) ;
      fflush(msgFile) ;
   }
   if ( flag == 1 ) {
/*
      -------------------------
      message has been received
      -------------------------
*/
      source = status.MPI_SOURCE ;
      tag    = status.MPI_TAG ;
      nbytes = msg->info[2] ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
                 "\n    message received: source %d, tag %d, nbytes %d",
                 source, tag, nbytes) ;
         fflush(msgFile) ;
      }
      switch ( type ) {
      case 1 :
/*
         -------------------------------------------------
         notification message for incoming aggregate front
         -------------------------------------------------
*/
         if ( nbytes > 0 ) {
/*
            ----------------------------
            post receive for new message
            ----------------------------
*/
            msg->info[0] = 2 ;
            msg->chv     = DChvManager_newObjectOfSizeNbytes(manager,
                                                             nbytes) ;
            msg->base    = (void *) DChv_workspace(msg->chv) ;
            msg->next    = nextmsg ;
            nextmsg      = msg ;
            tag          += nfront ;
            stats[2]++ ;
            stats[3] += nbytes ;
            if ( msglvl > 1 ) {
               fprintf(msgFile,
          "\n    2. posting Irecv, msg %p, type = 2, J = %d, tag = %d",
          msg, J, tag) ;
               fflush(msgFile) ;
            }
            MPI_Irecv(msg->base, nbytes, MPI_BYTE,
                      source, tag, comm, &msg->req) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, ", return") ;
               fflush(msgFile) ;
            }
         } else {
/*
            ------------------------
            aggregate front is empty
            ------------------------
*/
            if ( msglvl > 1 ) {
               fprintf(msgFile,
                       "\n    adding NULL object to aggregate list"
                       "\n    before count is %d", aggList->counts[J]) ;
               fflush(msgFile) ;
            }
            DChvList_addObjectToList(aggList, NULL, J) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile,
                       "\n    count is now %d", aggList->counts[J]) ;
               fflush(msgFile) ;
            }
            FactorMsg_free(msg) ;
         }
         break ;
      case 2 :
/*
         -------------------------------
         put aggregate front on its list
         -------------------------------
*/
         DChv_initFromBuffer(msg->chv) ;
         if ( msglvl > 2 ) {
            fprintf(msgFile, "\n    received aggregate front %d", J) ;
            DChv_writeForHumanEye(msg->chv, msgFile) ;
            fflush(msgFile) ;
         }
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    adding chv to aggregate list"
                    "\n    before count is %d", aggList->counts[J]) ;
            fflush(msgFile) ;
         }
         DChvList_addObjectToList(aggList, msg->chv, J) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile,
                    "\n    count is now %d", aggList->counts[J]) ;
            fflush(msgFile) ;
         }
         FactorMsg_free(msg) ;
         break ;
      case 3 :
         if ( nbytes > 0 ) {
/*
            --------------------------------------------------
            post receive for message to contain postponed data
            --------------------------------------------------
*/
            msg->info[0] = 4 ;
            msg->chv     = DChvManager_newObjectOfSizeNbytes(manager,
                                                             nbytes) ;
            msg->base    = (void *) DChv_workspace(msg->chv) ;
            msg->next    = nextmsg ;
            nextmsg      = msg ;
            tag          += nfront + 1 ;
            stats[6]++ ;
            stats[7] += nbytes ;
            if ( msglvl > 1 ) {
               fprintf(msgFile,
          "\n    2. posting Irecv, msg %p, type = 4, I = %d, tag = %d",
          msg, msg->info[1], tag) ;
               fflush(msgFile) ;
            }
            MPI_Irecv(msg->base, nbytes, MPI_BYTE,
                      source, tag, comm, &msg->req) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile, ", return") ;
               fflush(msgFile) ;
            }
         } else {
/*
            ------------------------
            postponed data is empty
            ------------------------
*/
            if ( msglvl > 1 ) {
               fprintf(msgFile,
                      "\n    adding NULL object to postponed list"
                      "\n    before, count is %d", postList->counts[J]) ;
               fflush(msgFile) ;
            }
            DChvList_addObjectToList(postList, NULL, J) ;
            if ( msglvl > 1 ) {
               fprintf(msgFile,
                       "\n   count is now %d", postList->counts[J]) ;
               fflush(msgFile) ;
            }
            FactorMsg_free(msg) ;
         }
         break ;
      case 4 :
/*
         ------------------------------
         put postponed data on its list
         ------------------------------
*/
         DChv_initFromBuffer(msg->chv) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    received postponed data %d", J) ;
            DChv_writeForHumanEye(msg->chv, msgFile) ;
            fflush(msgFile) ;
         }
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
                     "\n    before, count is %d", postList->counts[J]) ;
            fflush(msgFile) ;
         }
         DChvList_addObjectToList(postList, msg->chv, J) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile,
                    "\n    count is now %d", postList->counts[J]) ;
            fflush(msgFile) ;
         }
         FactorMsg_free(msg) ;
         break ;
      default :
         break ;
      }
   } else {
/*
      -----------------------------------------------
      message has not yet been received, keep in list
      -----------------------------------------------
*/
      msg->next = firstmsg ;
      firstmsg  = msg ;
   }
}
return(firstmsg) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   after front J has completed, remove relevant data from the 
   aggregage and/or postponed lists and send to the owning processor.

   created -- 97jul10,  cca
   ------------------------------------------------------------------
*/
static FactorMsg *
sendMessages (
   int           J,
   int           myid,
   int           nfront,
   int           par[],
   int           owners[],
   int           firsttag,
   DChvManager   *manager,
   DChvList      *aggList,
   DChvList      *postList,
   FactorMsg     *firstmsgsent,
   int           stats[],
   MPI_Comm      comm,
   int           msglvl,
   FILE          *msgFile
) {
DChv        *chv ;
FactorMsg   *msg ;
int         destination, K, nbytes, tag ;

destination = owners[J] ;
if ( msglvl > 1 ) {
   fprintf(msgFile, 
      "\n\n inside sendMessages, J = %d, firsttag = %d, destination %d",
      J, firsttag, destination) ;
   fflush(msgFile) ;
}
if ( destination != myid ) {
/*
   -----------------------------------------
   this process does not own the front, so
   an aggregate front may have been created.
   -----------------------------------------
*/
   if ( DChvList_isListNonempty(aggList, J) == 1 ) {
/*
      -----------------------------------------
      there is an aggregate chevron on the list
      -----------------------------------------
*/
      chv    = DChvList_getList(aggList, J) ;
      nbytes = DChv_nbytesNeeded(chv->nD, chv->nL, chv->nU,
               chv->symflag) ;
   } else {
/*
      ------------------
      no aggregate front
      ------------------
*/
      chv    = NULL ;
      nbytes = 0 ;
   }
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n proc %d does not own front %d, nbytes = %d", 
              myid, J, nbytes) ;
      fflush(msgFile) ;
   }
/*
   --------------------------------------------------------
   first message is a notification with the number of bytes
   --------------------------------------------------------
*/
   msg = FactorMsg_new() ;
   msg->info[0] = 1 ;
   msg->info[1] = J ;
   msg->info[2] = nbytes ;
   tag          = firsttag + J ;
   msg->next    = firstmsgsent ;
   firstmsgsent = msg  ;
   stats[0]++ ;
   stats[1] += 3*sizeof(int) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
  "\n posting Isend, msg %p, type %d, J %d, nbytes %d, dest %d, tag %d",
  msg, msg->info[0], msg->info[1], msg->info[2], destination, tag) ;
      fflush(msgFile) ;
   }
   MPI_Isend((void *) msg->info, 3, MPI_INT, destination, tag, 
             comm, &msg->req) ;
   if ( nbytes > 0 ) {
/*
      -------------------------------------------
      second message contains the aggregate front
      -------------------------------------------
*/
      msg = FactorMsg_new() ;
      msg->info[0] = 2 ;
      msg->info[1] = J ;
      msg->info[2] = nbytes ;
      msg->chv     = chv ;
      msg->base    = (void *) DChv_workspace(chv) ;
      tag          = firsttag + nfront + J ;
      msg->next    = firstmsgsent ;
      firstmsgsent = msg  ;
      stats[0]++ ;
      stats[1] += nbytes ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
  "\n posting Isend, msg %p, type %d, J %d, nbytes %d, dest %d, tag %d",
  msg, msg->info[0], msg->info[1], msg->info[2], destination, tag) ;
         fflush(msgFile) ;
      }
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n aggregate front %d, chv %p", J, chv) ;
         if ( chv != NULL ) {
            DChv_writeForHumanEye(msg->chv, msgFile) ;
         }
         fflush(msgFile) ;
      }
      MPI_Isend(msg->base, nbytes, MPI_BYTE, destination, tag, 
                comm, &msg->req) ;
   }
} else if ( postList != NULL ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n    checking for postponed messages") ;
      fflush(msgFile) ;
   }
/*
   ----------------------------------------------
   pivoting enabled, look for a postponed chevron
   ----------------------------------------------
*/
   if ( (K = par[J]) != -1 ) {
      destination = owners[K] ;
   } else {
      K = nfront ;
      destination = 0 ;
   }
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n    J = %d, K = %d, destination = %d",
              J, K, destination) ;
      fflush(msgFile) ;
   }
   if ( destination != myid ) {
/*
      ---------------------------------------
      owner of the parent is not this process
      ---------------------------------------
*/
      if ( DChvList_isListNonempty(postList, K) == 1 ) {
/*
         ----------------------------------------
         there is a postponed chevron on the list
         ----------------------------------------
*/
         chv    = DChvList_getList(postList, K) ;
         nbytes = DChv_nbytesNeeded(chv->nD, chv->nL, chv->nU,
                  chv->symflag) ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
                    "\n    nD = %d, nL = %d, nU = %d, nbytes = %d",
                    chv->nD, chv->nL, chv->nU, nbytes) ;
            fflush(msgFile) ; 
         }
      } else {
         chv    = NULL ;
         nbytes = 0 ;
      }
/*
      ----------------------------------------
      owner of the parent is not this process,
      send a notification message
      ----------------------------------------
*/
      msg = FactorMsg_new() ;
      msg->info[0] = 3 ;
      msg->info[1] = J ;
      msg->info[2] = nbytes ;
      tag          = firsttag + 2*nfront + J ;
      msg->next    = firstmsgsent ;
      firstmsgsent = msg  ;
      stats[4]++ ;
      stats[5] += 3*sizeof(int) ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
  "\n posting Isend, msg %p, type %d, J %d, nbytes %d, dest %d, tag %d",
  msg, msg->info[0], msg->info[1], msg->info[2], destination, tag) ;
         fflush(msgFile) ;
      }
      MPI_Isend((void *) msg->info, 3, MPI_INT, destination, tag, 
                comm, &msg->req) ;
      if ( nbytes > 0 ) {
/*
         ----------------------------------------------------------
         there is postponed data, send the data in a second message
         ----------------------------------------------------------
*/
         msg = FactorMsg_new() ;
         msg->info[0] = 4 ;
         msg->info[1] = J ;
         msg->info[2] = nbytes ;
         msg->chv     = chv ;
         msg->base    = (void *) DChv_workspace(chv) ;
         tag          = firsttag + 3*nfront + J + 1 ;
         msg->next    = firstmsgsent ;
         firstmsgsent = msg  ;
         stats[4]++ ;
         stats[5] += nbytes ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, 
  "\n posting Isend, msg %p, type %d, J %d, nbytes %d, dest %d, tag %d",
  msg, msg->info[0], msg->info[1], msg->info[2], destination, tag) ;
            fflush(msgFile) ;
         }
         MPI_Isend(msg->base, nbytes, MPI_BYTE, destination, 
                   tag, comm, &msg->req) ;
      }
   }
}
return(firstmsgsent) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   check the list of posted sends.
   if a send has completed, free the message structure 
   and return the DChv object (if present) to its manager.

   created -- 97nov13, cca
   -------------------------------------------------------
*/
static FactorMsg *
checkSentMessages (
   FactorMsg     *firstmsgsent,
   DChvManager   *manager,
   int           msglvl,
   FILE          *msgFile
) {
FactorMsg    *msg, *nextmsg ;
int          flag, J, nbytes, type ;
MPI_Status   status ;

for ( msg = firstmsgsent, firstmsgsent = NULL ;
      msg != NULL ;
      msg = nextmsg ) {
   nextmsg = msg->next ;
   type    = msg->info[0] ;
   J       = msg->info[1] ;
   nbytes  = msg->info[2] ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n checking sent msg %p : type %d, front %d, nbytes %d",
              msg, type, J, nbytes) ;
      fflush(msgFile) ;
   }
   MPI_Test(&msg->req, &flag, &status) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, ", flag = %d", flag) ;
      fflush(msgFile) ;
   }
   if ( flag == 1 ) {
      if ( msg->chv != NULL ) {
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n    chv %p released", msg->chv) ;
            fflush(msgFile) ;
         }
         DChvManager_releaseObject(manager, msg->chv) ;
      }
      FactorMsg_free(msg) ;
   } else {
      msg->next    = firstmsgsent ;
      firstmsgsent = msg ;
   }
}
return(firstmsgsent) ; }

/*--------------------------------------------------------------------*/
