/*  aggList.c  */

#include "../spoolesMPI.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with aggregate chevrons
      
   created  -- 97jul30, cca
   modified -- 97nov12, cca
      made "safe" using MPI_Sendrecv()
   -----------------------------------------------
*/
DChvList *
DFrontMtx_MPI_aggregateList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         tag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) {
char         *mark ;
DChvList     *aggList ;
int          count, destination, ierr, ii, incount, iproc, J, K, 
             left, maxincount, maxoutcount, myid, nfront, nproc,
             offset, outcount, right, size, source ;
int          *aggcounts, *frontOwners, *inbuffer, *incounts, 
             *indices, *outbuffer, *outcounts, *vtxToFront ;
IVL          *symbfacIVL ;
MPI_Status   status ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontOwnersIV == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_MPI_aggregateList(%p,%p,%p)"
           "\n bad input\n", frontmtx, frontOwnersIV, comm) ;
   exit(-1) ;
}
{   int   flag, tag_bound ;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &tag_bound, &flag) ;
    if ( tag < 0 || tag > tag_bound ) {
       fprintf(stderr, "\n fatal error in DFrontMtx_MPI_aggregateList()"
               "\n tag = %d, tag_bound = %d", tag, tag_bound) ;
       exit(-1) ;
    }
}
MPI_Comm_rank(comm, &myid) ;
MPI_Comm_size(comm, &nproc) ;
symbfacIVL = frontmtx->symbfacIVL ;
vtxToFront = ETree_vtxToFront(frontmtx->frontETree) ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n inside DFrontMtx_aggListMPI") ; 
   fflush(msgFile) ;
}
/*
   ----------------------------------------------------
   mark all fronts that are supported by an owned front
   ----------------------------------------------------
*/
mark = CVinit(nfront, 'N') ;
for ( J = 0 ; J < nfront ; J++ ) {
   iproc = frontOwners[J] ;
   if ( iproc == myid ) {
      IVL_listAndSize(symbfacIVL, J, &size, &indices) ;
      for ( ii = 0 ; ii < size ; ii++ ) {
         K = vtxToFront[indices[ii]] ;
         if ( mark[K] == 'N' ) {
            mark[K] = 'Y' ;
            if ( msglvl > 2 ) {
               fprintf(msgFile, "\n front %d supported", K) ;
               fflush(msgFile) ;
            }
         }
      }
   }
}
/*
   ------------------------------
   compute the outcounts[] vector
   ------------------------------
*/
outcounts = IVinit(2*nproc, 0) ;
incounts  = outcounts + nproc ; for ( J = 0 ; J < nfront ; J++ ) {
   if ( (iproc = frontOwners[J]) != myid && mark[J] == 'Y' ) {
      outcounts[iproc]++ ;
   }
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n outcounts :") ;
   IVfp80(msgFile, nproc, outcounts, 20, &ierr) ;
   fflush(msgFile) ;
}
/*
   -------------------------------
   do an all-to-all gather/scatter
   -------------------------------
*/
MPI_Alltoall((void *) outcounts, 1, MPI_INT,
             (void *) incounts,  1, MPI_INT, comm) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n incounts :") ;
   IVfp80(msgFile, nproc, incounts, 20, &ierr) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------
   allocate buffers of sufficient size
   -----------------------------------
*/
for ( iproc = 0, maxoutcount = 0 ; iproc < nproc ; iproc++ ) {
   if ( maxoutcount < outcounts[iproc] ) {
      maxoutcount = outcounts[iproc] ;
   }
}
outbuffer = IVinit(maxoutcount, -1) ;
for ( iproc = 0, maxincount = 0 ; iproc < nproc ; iproc++ ) {
   if ( maxincount < incounts[iproc] ) {
      maxincount = incounts[iproc] ;
   }
}
inbuffer = IVinit(maxincount, -1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n out buffer has size %d", maxoutcount) ;
   fprintf(msgFile, "\n in  buffer has size %d", maxincount) ;
   fflush(msgFile) ;
}
/*
   ----------------------------
   send and receive information
   ----------------------------
*/
aggcounts = IVinit(nfront, 0) ;
for ( offset = 1 ; offset < nproc ; offset++ ) {
/*
   -------------------------------------------------------
   find the left and right processes to exchange data with
   -------------------------------------------------------
*/
   right = (myid + offset) % nproc ;
   if ( offset <= myid ) {
      left = myid - offset ;
   } else {
      left = myid + nproc - offset ;
   }
   if ( (outcount = outcounts[right]) > 0 ) {
/*
      -------------------
      load the out buffer
      -------------------
*/
      for ( ii = J = 0 ; J < nfront ; J++ ) {
         if ( mark[J] == 'Y' && frontOwners[J] == right ) {
            outbuffer[ii++] = J ;
         }
      }
      if ( ii != outcount ) {
         fprintf(stderr, 
                 "\n process %d : fatal error"
                 ", right = %d, outcount = %d, ii = %d",
                 myid, right, outcount, ii) ;
         exit(-1) ;
      }
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n outbuffer for %d", right) ;
         IVfprintf(msgFile, outcount, outbuffer) ;
         fflush(msgFile) ;
      }
      destination = right ;
      stats[0]++ ;
      stats[2] += outcount*sizeof(int) ;
   } else {
      destination = MPI_PROC_NULL ;
   }
   if ( (incount = incounts[left]) > 0 ) {
/*
      -------------------
      set up input buffer
      -------------------
*/
      source = left ;
      stats[1]++ ;
      stats[3] += incount*sizeof(int) ;
   } else {
      source = MPI_PROC_NULL ;
   }
/*
   -------------------------
   send and receive the data
   -------------------------
*/
   MPI_Sendrecv((void *)outbuffer, outcount, MPI_INT, destination, tag,
                (void *)inbuffer,  incount,  MPI_INT, source,      tag,
                comm, &status) ;
   if ( incount > 0 ) {
/*
      ---------------------
      use the incoming data
      ---------------------
*/
      MPI_Get_count(&status, MPI_INT, &count) ;
      if ( count != incount ) {
         fprintf(stderr, 
                 "\n process %d : fatal error"
                 " count from status = %d, incount = %d",
                 myid, count, incount) ;
         exit(-1) ;
      }
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n inbuffer from %d", left) ;
         IVfprintf(msgFile, incount, inbuffer) ;
         fflush(msgFile) ;
      }
      for ( ii = 0 ; ii < incount ; ii++ ) {
         J = inbuffer[ii] ;
         aggcounts[J]++ ;
      }
   }
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n aggcounts") ;
   IVfp80(msgFile, nfront, aggcounts, 80, &ierr) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------
   create and initialize the DChvList object
   -----------------------------------------
*/
aggList = DChvList_new() ;
DChvList_init(aggList, nfront, aggcounts, 0, NULL) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(aggcounts) ;
IVfree(outcounts) ;
IVfree(inbuffer)  ;
IVfree(outbuffer) ;
CVfree(mark)      ;

return(aggList) ; }

/*--------------------------------------------------------------------*/
