/*  spoolesMPI.h  */

#ifndef _SPOOLES_MPI_

#define _SPOOLES_MPI_
#include <mpi.h>
#include "../DFrontMtx.h"
#include "../DLinSystem.h"
#include "../ETree.h"
#include "../IVL.h"
#include "../IV.h"
#include "../Ideq.h"
#include "../DPencil.h"
#include "../Drand.h"

#endif

/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- split methods -----------------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------------------
   purpose -- to split a DDenseMtx object by rows
 
   mtx         -- DDenseMtx object
   rowmapIV    -- map from rows to owning processes
   firsttag    -- first tag to be used in these messages
   stats[4]    -- statistics vector
      stats[0] -- # of messages sent
      stats[1] -- # of messages received
      stats[2] -- # of bytes sent
      stats[3] -- # of bytes received
   msglvl      -- message level
   msgFile     -- message file
   comm        -- MPI communicator
 
   return value -- a new DDenseMtx object filled with the owned rows 
 
   created  -- 97jul13, cca
   modified -- 97oct17, cca
      stats added
   modified -- 97nov08, cca
      send/recv replaces send and receive
   -----------------------------------------------------------------
*/
DDenseMtx *
DDenseMtx_MPI_splitByRows (
   DDenseMtx   *mtx,
   IV          *rowmapIV,
   int         firsttag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   ------------------------------------------------------------------
   purpose -- to split a distributed DPencil object
 
   pencil     -- pointer to the local DPencil object
   mapIV      -- pointer to the map from vertices to processes
   tag        -- first tag value, four might be used, tag through tag+3
   stats[4]    -- statistics vector
      stats[0] -- # of messages sent
      stats[1] -- # of messages received
      stats[2] -- # of bytes sent
      stats[3] -- # of bytes received
   msglvl     -- local message level
   msgFile    -- local message file
   comm       -- MPI communication structure
 
   created  -- 97jul18, cca
   modified -- 97oct17, cca
      stats added
   ------------------------------------------------------------------
*/
void
DPencil_MPI_split (
   DPencil    *pencil,
   IV         *mapIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   ------------------------------------------------------------------
   purpose -- to split a distributed DLinSystem object
 
   linsys     -- pointer to the local DLinSystem object
   mapIV      -- pointer to the map from vertices to processes
   tag        -- first tag value, eight will be used, tag through tag+7
   stats[4]    -- statistics vector
      stats[0] -- # of messages sent
      stats[1] -- # of messages received
      stats[2] -- # of bytes sent
      stats[3] -- # of bytes received
   msglvl     -- local message level
   msgFile    -- local message file
   comm       -- MPI communication structure
 
   created  -- 97jul18, cca
   modified -- 97oct17, cca
      stats added
   ------------------------------------------------------------------
*/
void
DLinSystem_MPI_split (
   DLinSystem   *linsys,
   IV           *mapIV,
   int          firsttag,
   int          stats[],
   int          msglvl,
   FILE         *msgFile,
   MPI_Comm     comm
) ;
/*
   ------------------------------------------------------------
   purpose -- to split a distributed DInpMtx object
 
   comm     -- MPI communication structure
   inpmtx   -- pointer to the local DInpMtx object
   mapIV    -- pointer to the map from vertices to processes
   firsttag -- first tag value, two will be used, tag and tag+1
   stats[4] -- statistics vector
      stats[0] -- # of messages sent
      stats[1] -- # of messages received
      stats[2] -- # of bytes sent
      stats[3] -- # of bytes received
   msglvl   -- local message level
   msgFile  -- local message file
 
   return value -- pointer to the new DInpMtx object 
      that contains the owned entries.
 
   created  -- 97jun20, cca
   modified -- 97oct17, cca
      stats added
   ------------------------------------------------------------
*/
DInpMtx *
DInpMtx_MPI_split (
   DInpMtx    *inpmtx,
   IV         *mapIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- symbolic factorization method -------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------------
   perform the symbolic factorization in parallel
 
   input --
      frontETree    -- front tree for the factorization
      frontOwnersIV -- map from fronts to owning process
      inpmtx        -- matrix object
      firsttag      -- first tag to be used for messages,
                       will use tag, ..., tag + nfront - 1
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator
 
   return value --
      symbfacIVL -- contains front indices for supported fronts
 
   created -- 97dec07, cca
   --------------------------------------------------------------------
*/
IVL *
SymbFac_MPI_initFromDInpMtx (
   ETree      *frontETree,
   DInpMtx    *inpmtx,
   IV         *frontOwnersIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   --------------------------------------------------------------------
   perform the symbolic factorization in parallel
 
   input --
      frontETree    -- front tree for the factorization
      pencil        -- matrix pencil
      frontOwnersIV -- map from fronts to owning process
      firsttag      -- first tag to be used for messages,
                       will use tag, ..., tag + nfront - 1
      stats[4]    -- statistics vector
         stats[0] -- # of messages sent
         stats[1] -- # of messages received
         stats[2] -- # of bytes sent
         stats[3] -- # of bytes received
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator
 
   return value --
      symbfacIVL -- contains front indices for supported fronts
 
   created -- 97jul29, cca
   --------------------------------------------------------------------
*/
IVL *
SymbFac_MPI_initFromDPencil (
   ETree      *frontETree,
   DPencil    *pencil,
   IV         *frontOwnersIV,
   int        firsttag,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- numeric factorization method --------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
  this is the method that computes the 
   parallel factorization of a sparse matrix A using MPI.
 
   input ---
 
      frontmtx      -- front matrix object that holds the factors
      inpmtx        -- matrix object for A
      tau           -- tolerance used in pivoting
      droptol       -- tolerance used in the approximate factorization
      firsttag      -- first tag to use during the factorization,
                       reserved tags are tag, ..., tag + 4*nfront + 2
      frontOwnersIV -- map from fronts to owning processes
      lookahead     -- parameter that governs the ``lookahead''
                       behavior, use lookahead = 0 as a default
      cpus          -- vector to store breakdown of cpu times
         cpus[ 0] -- initialize fronts
         cpus[ 1] -- load original entries
         cpus[ 2] -- update fronts
         cpus[ 3] -- insert aggregate data
         cpus[ 4] -- assemble aggregate data
         cpus[ 5] -- assemble postponed data
         cpus[ 6] -- factor fronts
         cpus[ 7] -- extract postponed data
         cpus[ 8] -- store factor entries
         cpus[ 9] -- post initial receives
         cpus[10] -- check for received messages
         cpus[11] -- post initial sends
         cpus[12] -- check for sent messages
      stats         -- vector to store statistics
         stats[ 0] -- # of aggregate sends
         stats[ 1] -- # of bytes in the aggregate sends
         stats[ 2] -- # of aggregate received
         stats[ 3] -- # of bytes in the aggregate received
         stats[ 4] -- # of postponed data sends
         stats[ 5] -- # of bytes in the postponed data sends
         stats[ 6] -- # of postponed data received
         stats[ 7] -- # of bytes in the postponed data received
         stats[ 8] -- # of active DChv objects (working storage)
         stats[ 9] -- # of active bytes in working storage
         stats[10] -- # of requested bytes in working storage
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator
 
   return value -- if process id is zero, a pointer to the first
   DChv object in a list that contains postponed rows and columns
   that could not be eliminated.
 
   created  -- 97nov17, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MPI_factorDInpMtx (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtx,
   double      tau,
   double      droptol,
   int         firsttag,
   IV          *frontOwnersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   -------------------------------------------------------------------
  this is the method that computes the
   parallel factorization of A + sigma*B using MPI.
 
   input ---
 
      frontmtx      -- front matrix object that holds the factors
      pencil        -- matrix pencil object, contains A + sigma*B
      tau           -- tolerance used in pivoting
      droptol       -- tolerance used in the approximate factorization
      firsttag      -- first tag to use during the factorization,
                       reserved tags are tag, ..., tag + 4*nfront + 2
      frontOwnersIV -- map from fronts to owning processes
      lookahead     -- parameter that governs the ``lookahead''
                       behavior, use lookahead = 0 as a default
      cpus          -- vector to store breakdown of cpu times
         cpus[ 0] -- initialize fronts
         cpus[ 1] -- load original entries
         cpus[ 2] -- update fronts
         cpus[ 3] -- insert aggregate data
         cpus[ 4] -- assemble aggregate data
         cpus[ 5] -- assemble postponed data
         cpus[ 6] -- factor fronts
         cpus[ 7] -- extract postponed data
         cpus[ 8] -- store factor entries
         cpus[ 9] -- post initial receives
         cpus[10] -- check for received messages
         cpus[11] -- post initial sends
         cpus[12] -- check for sent messages
      stats         -- vector to store statistics
         stats[ 0] -- # of aggregate sends
         stats[ 1] -- # of bytes in the aggregate sends
         stats[ 2] -- # of aggregate received
         stats[ 3] -- # of bytes in the aggregate received
         stats[ 4] -- # of postponed data sends
         stats[ 5] -- # of bytes in the postponed data sends
         stats[ 6] -- # of postponed data received
         stats[ 7] -- # of bytes in the postponed data received
         stats[ 8] -- # of active DChv objects (working storage)
         stats[ 9] -- # of active bytes in working storage
         stats[10] -- # of requested bytes in working storage
      msglvl        -- message level
      msgFile       -- message file
      comm          -- MPI communicator
 
   return value -- if process id is zero, a pointer to the first
   DChv object in a list that contains postponed rows and columns
   that could not be eliminated.
 
   created  -- 97jul11, cca
   modified -- 97nov13, cca
      blocking communication replaced with nonblocking communication
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MPI_factorDPencil (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   int         firsttag,
   IV          *frontOwnersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- method to set up the aggregate list -------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with aggregate chevrons
      
   created -- 97jul30, cca
   modified -- 97nov12, cca
      made "safe" using MPI_Sendrecv()
   -----------------------------------------------
*/
DChvList *
DFrontMtx_MPI_aggregateList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         tag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- row map method  ---------------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------------
   purpose -- after pivoting for a nonsymmetric factorization,
              some delayed rows may belong to a process other
              than its original owner. this method returns an
              IV object that maps rows to owning processes.
 
   created -- 97aug14, cca
   -----------------------------------------------------------
*/
IV *
DFrontMtx_MPI_rowmap (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   -------------------------------------------------------------
   purpose -- after pivoting for a nonsymmetric factorization,
              some delayed columns may belong to a process other
              than its original owner. this method returns an
              IV object that maps columns to owning processes.
 
   created -- 97aug14, cca
   -------------------------------------------------------------
*/
IV *
DFrontMtx_MPI_colmap (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- solve method  -----------------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   solve the linear system
 
   created -- 97aug23, cca
   -----------------------
*/
void
DFrontMtx_MPI_solve ( 
   DFrontMtx          *frontmtx,
   DDenseMtx          *solmtx,
   DDenseMtx          *rhsmtx,
   DDenseMtxManager   *manager,
   IV                 *frontOwnersIV,
   int                firsttag,
   int                stats[],
   double             cpus[],
   int                msglvl,
   FILE               *msgFile,
   MPI_Comm           comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- post-process methods  ---------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------
   purpose -- postprocess the DFrontMtx object
 
   permuteflag == 1 --> used when pivoting enabled
     (1) permute the columns of U_{J,bnd{J}} so they are in
         ascending order w.r.t. to global column numbering
     (2) permute the rows of L_{bnd{J},J} so they are in
         ascending order w.r.t. to global row numbering
   localflag == 0 --> do nothing
   localflag == 1 --> global-to-local,
      overwrite row and column indices bnd{J}
      with indices that are local w.r.t. J's parent
   localflag == 2 --> local-to-global,
      overwrite local row and column indices bnd{J}
      with global indices 
 
   return value -- new first tag
 
   created -- 97nov20, cca
   ----------------------------------------------------------
*/
int
DFrontMtx_MPI_postProcess (
   DFrontMtx   *frontmtx,
   int         permuteflag,
   int         localflag,
   IV          *frontOwnersIV,
   int         firsttag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   -------------------------------------------------------
   purpose --- overwrite row and column indices bnd{J}
      with global indices 
 
   firsttag -- first tag for messages, will use tag values
      in [firsttag, firsttag + 2*nfront]
   stats[] -- statistics vector
      stats[0] -- # of sends
      stats[1] -- # of bytes sent
      stats[2] -- # of receives
      stats[3] -- # of bytes received
 
   created -- 97nov22, cca
   -------------------------------------------------------
*/
void
DFrontMtx_MPI_localToGlobalInd (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         firsttag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   -------------------------------------------------------
   purpose --- overwrite row and column indices bnd{J}
      with indices that are local w.r.t. J's parent
 
   firsttag -- first tag for messages, will use tag values
      in [firsttag, firsttag + 2*nfront]
   stats[] -- statistics vector
      stats[0] -- # of sends
      stats[1] -- # of bytes sent
      stats[2] -- # of receives
      stats[3] -- # of bytes received
 
   created -- 97nov22, cca
   -------------------------------------------------------
*/
void
DFrontMtx_MPI_globalToLocalInd (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         firsttag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*
   --------------------------------------------------------
   purpose ---
 
     (1) permute the columns of U_{J,bnd{J}} so they are in
         ascending order w.r.t. to global column numbering
     (2) permute the rows of L_{bnd{J},J} so they are in
         ascending order w.r.t. to global row numbering
 
   firsttag -- first tag for messages, will use tag values
      in [firsttag, firsttag + 2*nfront]
   stats[] -- statistics vector
      stats[0] -- # of sends
      stats[1] -- # of bytes sent
      stats[2] -- # of receives
      stats[3] -- # of bytes received
 
   created -- 97nov22, cca
   --------------------------------------------------------
*/
void
DFrontMtx_MPI_permuteFactor (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         firsttag,
   int         stats[],
   int         msglvl,
   FILE        *msgFile,
   MPI_Comm    comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- full adjacency method  --------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------------
   purpose -- given a distributed DInpMtx object,
      gather all the indices onto each process and create
      an IVL object that contains the full adjacency structure
 
   created -- 97dec17, cca
   -----------------------------------------------------------
*/
IVL *
DInpMtx_MPI_fullAdjacency (
   DInpMtx    *inpmtx,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------
   purpose -- given a distributed DPencil object,
      gather all the indices onto each process and create
      an IVL object that contains the full adjacency structure
 
   created -- 97dec18, cca
   -----------------------------------------------------------
*/
IVL *
DPencil_MPI_fullAdjacency (
   DPencil    *pencil,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
---- ordering methods --------------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMMD() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaMMD (
   DPencil    *pencil,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMMD() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaMMD (
   DInpMtx    *inpmtx,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaND() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaND (
   DInpMtx    *inpmtx,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaND() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaND (
   DPencil    *pencil,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMS() method.
   the difference is that we start from a distributed DInpMtx object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DInpMtx_MPI_orderViaMS (
   DInpMtx    *inpmtx,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*
   -----------------------------------------------------------------
   this is the MPI version of the orderViaMS() method.
   the difference is that we start from a distributed DPencil object
   instead of a global Graph object.
 
   created -- 97dec17, cca
   -----------------------------------------------------------------
*/
ETree *
DPencil_MPI_orderViaMS (
   DPencil    *pencil,
   int        maxdomainsize,
   int        maxzeros,
   int        maxfrontsize,
   int        seed,
   int        stats[],
   int        msglvl,
   FILE       *msgFile,
   MPI_Comm   comm
) ;
/*--------------------------------------------------------------------*/
