/*  testSplitPencil.c  */

#include "../spoolesMPI.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------------
   this program tests splitting a DPencil object

   (1) process 0 reads in a matrix pencil
   (2) using a wrap map, distribute the object
   (3) using a random map, distribute the object again

   created -- 97jul29, cca
   -------------------------------------------------------
*/
{
char         *buffer, *inpmtxAfileName, *inpmtxBfileName ;
DPencil      *pencil ;
double       sigma, t1, t2 ;
Drand        drand ;
int          length, myid, msglvl, neqns, nproc, 
             randomflag, seed, symmetryflag, tag, v ;
int          *map ;
int          stats[4] ;
IV           *mapIV ;
FILE         *msgFile ;
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n process %d of %d, argc = %d", myid, nproc, argc) ;
fflush(stdout) ;
if ( argc != 10 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile neqns inpmtxAfile sigma inpmtxBfile "
"\n         symmetryflag randomflag seed "
"\n    msglvl       -- message level"
"\n    msgFile      -- message file"
"\n    neqns        -- number of equations"
"\n    inpmtxAfile  -- file name for A"
"\n    sigma        -- shift value"
"\n    inpmtxBfile  -- file name for B"
"\n    symmetryflag -- symmetry flag"
"\n       0 --> symmetric structure, symmetric entries"
"\n    randomflag   -- symmetry flag"
"\n       != 0 --> fill with random numbers"
"\n    seed         -- random number seed"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   length = strlen(argv[2]) + 1 + 4 ;
   buffer = CVinit(length, '\0') ;
   sprintf(buffer, "%s.%d", argv[2], myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], argv[2]) ;
      return ;
   }
   CVfree(buffer) ;
}
neqns           = atoi(argv[3]) ;
inpmtxAfileName = argv[4] ;
sigma           = atof(argv[5]) ;
inpmtxBfileName = argv[6] ;
symmetryflag    = atoi(argv[7]) ;
randomflag      = atoi(argv[8]) ;
seed            = atoi(argv[9]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl          -- %d" 
        "\n msgFile         -- %s" 
        "\n neqns           -- %d" 
        "\n inpmtxAfileName -- %s" 
        "\n sigma           -- %f" 
        "\n inpmtxBfileName -- %s" 
        "\n symmetryflag    -- %d" 
        "\n randomflag      -- %d" 
        "\n seed            -- %d" 
        "\n",
        argv[0], msglvl, argv[2], neqns, inpmtxAfileName, sigma,
        inpmtxBfileName, symmetryflag, randomflag, seed) ;
fflush(msgFile) ;
/*
   ----------------------------------
   set up the random number generator
   ----------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, 0.0, 1.0) ;
/*
   -----------------------------------
   initialize the matrix pencil object
   -----------------------------------
*/
MARKTIME(t1) ;
pencil = DPencil_setup(myid, inpmtxAfileName, sigma, inpmtxBfileName,
                    symmetryflag, randomflag, &drand, msglvl, msgFile) ;
DPencil_changeCoordType(pencil, 3) ;
DPencil_changeStorageMode(pencil, 2) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : pencil setup", t2 - t1) ;
if ( msglvl > 2 ) {
   DPencil_writeForHumanEye(pencil, msgFile) ;
} else {
   DPencil_writeStats(pencil, msgFile) ;
}
fflush(msgFile) ;
/*
   ------------------------------------------
   set the initial owners IV to be a wrap map
   ------------------------------------------
*/
mapIV = IV_new() ;
IV_init(mapIV, neqns, NULL) ;
map = IV_entries(mapIV) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = v % nproc ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------------
   split the DPencil object into pieces
   ------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
tag = 1 ;
DPencil_MPI_split(pencil, mapIV, tag, stats,
                 msglvl, msgFile, MPI_COMM_WORLD) ;
tag += 4 ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : pencil split", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
if ( msglvl > 2 ) {
   DPencil_writeForHumanEye(pencil, msgFile) ;
} else {
   DPencil_writeStats(pencil, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------
   generate a random map
   ---------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, 0.0, (double) neqns) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = ((int) Drand_value(&drand)) % nproc ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------
   now split the pencil with the random map
   ----------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
DPencil_MPI_split(pencil, mapIV, tag, stats,
                  msglvl, msgFile, MPI_COMM_WORLD) ;
fprintf(msgFile, 
      "\n\n after splitting the DDenseMtx object with the owners map") ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
if ( msglvl > 2 ) {
   DPencil_writeForHumanEye(pencil, msgFile) ;
} else {
   DPencil_writeStats(pencil, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------
   free the objects
   ----------------
*/
IV_free(mapIV) ;
DPencil_free(pencil) ;

MPI_Finalize() ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
