/*  testSplitDLinsystem.c  */

#include "../spoolesMPI.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------------
   this program tests splitting a DLinSystem object

   (1) process 0 reads in a matrix pencil
       and generates the rhs and solution matrices
       to create a linear system object
   (2) using a wrap map, distribute the object
   (3) using a random map, distribute the object again

   created -- 97jul29, cca
   -------------------------------------------------------
*/
{
char         *buffer, *inpmtxAfileName, *inpmtxBfileName ;
DLinSystem   *linsys ;
double       sigma, t1, t2 ;
Drand        drand ;
int          length, msglvl, myid, neqns, nproc, nrhs, 
             randomflag, seed, symmetryflag, tag, v ;
int          *map ;
int          stats[4] ;
IV           *mapIV ;
FILE         *msgFile ;
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n process %d of %d, argc = %d", myid, nproc, argc) ;
fflush(stdout) ;
if ( argc != 11 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile neqns nrhs "
"\n         inpmtxAfile sigma inpmtxBfile "
"\n         symmetryflag randomflag seed "
"\n    msglvl       -- message level"
"\n    msgFile      -- message file"
"\n    neqns        -- number of equations"
"\n    nrhs         -- number of right hand sides"
"\n    inpmtxAfile  -- file name for A"
"\n    sigma        -- shift value"
"\n    inpmtxBfile  -- file name for B"
"\n    symmetryflag -- symmetry flag"
"\n       0 --> symmetric structure, symmetric entries"
"\n    randomflag   -- symmetry flag"
"\n       != 0 --> fill with random numbers"
"\n    seed         -- random number seed"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   length = strlen(argv[2]) + 1 + 4 ;
   buffer = CVinit(length, '\0') ;
   sprintf(buffer, "%s.%d", argv[2], myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], argv[2]) ;
      return ;
   }
   CVfree(buffer) ;
}
neqns           = atoi(argv[3]) ;
nrhs            = atoi(argv[4]) ;
inpmtxAfileName = argv[5] ;
sigma           = atof(argv[6]) ;
inpmtxBfileName = argv[7] ;
symmetryflag    = atoi(argv[8]) ;
randomflag      = atoi(argv[9]) ;
seed            = atoi(argv[10]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl          -- %d" 
        "\n msgFile         -- %s" 
        "\n neqns           -- %d" 
        "\n nrhs            -- %d" 
        "\n inpmtxAfileName -- %s" 
        "\n sigma           -- %f" 
        "\n inpmtxBfileName -- %s" 
        "\n symmetryflag    -- %d" 
        "\n randomflag      -- %d" 
        "\n seed            -- %d" 
        "\n",
        argv[0], msglvl, argv[2], neqns, nrhs, inpmtxAfileName, sigma,
        inpmtxBfileName, symmetryflag, randomflag, seed) ;
fflush(msgFile) ;
/*
   ----------------------------------
   set up the random number generator
   ----------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, 0.0, 1.0) ;
/*
   -----------------------------------
   initialize the linear system object
   -----------------------------------
*/
MARKTIME(t1) ;
linsys = DLinSystem_new() ;
DLinSystem_setup(linsys, myid, neqns, nrhs, inpmtxAfileName, sigma,
                 inpmtxBfileName, symmetryflag, randomflag, &drand, 
                 msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : generate linear system", t2 - t1) ;
fprintf(msgFile, "\n\n DLinSystem generated") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
fflush(msgFile) ;
/*
   ------------------------------------------
   set the initial owners IV to be a wrap map
   ------------------------------------------
*/
MARKTIME(t1) ;
mapIV = IV_new() ;
IV_init(mapIV, neqns, NULL) ;
map = IV_entries(mapIV) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = v % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set wrap map", t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------------------
   split the linear system object into pieces
   ------------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
tag = 1 ;
MARKTIME(t1) ;
DLinSystem_MPI_split(linsys, mapIV, tag, stats,
                    msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : split system", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
tag += 10 ;
fprintf(msgFile, 
       "\n\n after splitting the DLinSystem object with the wrap map") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------
   generate a random map
   ---------------------
*/
MARKTIME(t1) ;
Drand_setDefaultFields(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, 0.0, (double) neqns) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = ((int) Drand_value(&drand)) % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set random map", t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------
   now split the linear system with the random map
   -----------------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
DLinSystem_MPI_split(linsys, mapIV, tag, stats,
                    msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : split system", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
tag += 10 ;
fprintf(msgFile, 
       "\n\n after splitting the DLinSystem object with the wrap map") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------
   free the objects
   ----------------
*/
IV_free(mapIV) ;
DLinSystem_free(linsys) ;

MPI_Finalize() ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
