/*  testSplit2.c  */

#include "../spoolesMPI.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------------
   this program tests splitting a DDenseMtx object

   (1) process 0 generates a nrow x ncol DDenseMtx object
   (2) using a wrap map, distribute the object
   (3) using a random map, distribute the object again

   created -- 97jul17, cca
   -------------------------------------------------------
*/
{
char         *buffer ;
DDenseMtx    *mtx, *mtxkeep ;
double       t1, t2 ;
double       pchecksums[2], schecksums[2], tchecksums[2] ;
Drand        drand ;
int          inc1, inc2, length, myid, 
             msglvl, ncol, nproc, nrow, seed, tag, v ;
int          *map ;
int          stats[4] ;
IV           *mapIV ;
FILE         *msgFile ;
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n process %d of %d, argc = %d", myid, nproc, argc) ;
fflush(stdout) ;
if ( argc != 8 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile nrow ncol inc1 inc2 seed "
"\n    msglvl  -- message level"
"\n    msgFile -- message file"
"\n    nrow    -- number of rows"
"\n    ncol    -- number of columns"
"\n    inc1    -- row increment"
"\n    inc2    -- column increment"
"\n    seed    -- random number seed"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   length = strlen(argv[2]) + 1 + 4 ;
   buffer = CVinit(length, '\0') ;
   sprintf(buffer, "%s.%d", argv[2], myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], argv[2]) ;
      return ;
   }
   CVfree(buffer) ;
}
nrow = atoi(argv[3]) ;
ncol = atoi(argv[4]) ;
inc1 = atoi(argv[5]) ;
inc2 = atoi(argv[6]) ;
seed = atoi(argv[7]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl     -- %d" 
        "\n msgFile    -- %s" 
        "\n nrow       -- %d" 
        "\n ncol       -- %d" 
        "\n inc1       -- %d" 
        "\n inc2       -- %d" 
        "\n seed       -- %d" 
        "\n",
        argv[0], msglvl, argv[2], nrow, ncol, 
        inc1, inc2, seed) ;
fflush(msgFile) ;
/*
   -----------------------------
   generate the DDenseMtx object
   -----------------------------
*/
mtx = DDenseMtx_new() ;
fprintf(msgFile, "\n mtx = %p", mtx) ;
fflush(msgFile) ;
MARKTIME(t1) ;
if ( myid == 0 ) {
   DDenseMtx_init(mtx, 1, -1, nrow, ncol, inc1, inc2) ;
   Drand_setDefaultFields(&drand) ;
   Drand_setSeed(&drand, seed) ;
   Drand_setUniform(&drand, 0.0, 1.0) ;
   Drand_fillDvector(&drand, nrow*ncol, DDenseMtx_entries(mtx)) ;
} else {
   if ( inc1 == 1 ) {
      DDenseMtx_init(mtx, 1, -1, 0, ncol, inc1, 0) ;
   } else {
      DDenseMtx_init(mtx, 1, -1, 0, ncol, inc1, inc2) ;
   }
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : initialize the DDenseMtx object",
        t2 - t1) ;
fflush(msgFile) ;
fprintf(msgFile, "\n\n DDenseMtx generated") ;
if ( msglvl > 2 ) {
   DDenseMtx_writeForHumanEye(mtx, msgFile) ;
} else {
   DDenseMtx_writeStats(mtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ------------------------------------------
   compute the serial checksums of the matrix
   ------------------------------------------
*/
   {  int      *rowind  = mtx->rowind ;
      double   *entries = mtx->entries ;
      int      nrow     = mtx->nrow ;
      int      ncol     = mtx->ncol ;
      int      ii, nent = nrow*ncol ;
 
      schecksums[0] = schecksums[1] = 0.0 ;
      for ( ii = 0 ; ii < nrow ; ii++ ) {
         schecksums[0] += rowind[ii] ;
      }
      for ( ii = 0 ; ii < nent ; ii++ ) {
         schecksums[1] += fabs(entries[ii])  ;
      }
   }
/*
   ------------------------------------------
   set the initial owners IV to be a wrap map
   ------------------------------------------
*/
MARKTIME(t1) ;
mapIV = IV_new() ;
IV_init(mapIV, nrow, NULL) ;
map = IV_entries(mapIV) ;
for ( v = 0 ; v < nrow ; v++ ) {
   map[v] = v % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set wrap map", t2 - t1) ;
fflush(msgFile) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------
   split the DDenseMtx object into pieces
   --------------------------------------
*/
tag = 1 ;
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
mtxkeep = DDenseMtx_MPI_splitByRows(mtx, mapIV, tag, stats,
                                   msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : split matrix", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
fflush(msgFile) ;
DDenseMtx_free(mtx) ;
mtx = mtxkeep ;
tag += 2 ;
fprintf(msgFile, 
        "\n\n after splitting the DDenseMtx object with the wrap map") ;
if ( msglvl > 2 ) {
   DDenseMtx_writeForHumanEye(mtx, msgFile) ;
} else {
   DDenseMtx_writeStats(mtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------
   generate a random map
   ---------------------
*/
MARKTIME(t1) ;
Drand_setDefaultFields(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, 0.0, (double) nrow) ;
for ( v = 0 ; v < nrow ; v++ ) {
   map[v] = ((int) Drand_value(&drand)) % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : generate random map", t2 - t1) ;
fflush(msgFile) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------
   now split the matrix with the random map
   ----------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
mtxkeep = DDenseMtx_MPI_splitByRows(mtx, mapIV, tag, stats,
                                   msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : split matrix ", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
fflush(msgFile) ;
DDenseMtx_free(mtx) ;
mtx = mtxkeep ;
fprintf(msgFile, 
      "\n\n after splitting the DDenseMtx object with the owners map") ;
if ( msglvl > 2 ) {
   DDenseMtx_writeForHumanEye(mtx, msgFile) ;
} else {
   DDenseMtx_writeStats(mtx, msgFile) ;
}
fflush(msgFile) ;
/*
   -----------------------------------------------
   compute the checksums of the distributed matrix
   -----------------------------------------------
*/
   {  int      *rowind  = mtx->rowind ;
      double   *entries = mtx->entries ;
      int      nrow     = mtx->nrow ;
      int      ncol     = mtx->ncol ;
      int      ii, nent = nrow*ncol ;
 
      pchecksums[0] = pchecksums[1] = 0.0 ;
      for ( ii = 0 ; ii < nrow ; ii++ ) {
         pchecksums[0] += rowind[ii] ;
      }
      for ( ii = 0 ; ii < nent ; ii++ ) {
         pchecksums[1] += fabs(entries[ii])  ;
      }
   }
MPI_Reduce((void *) pchecksums, (void *) tchecksums, 2, MPI_DOUBLE,
          MPI_SUM, 0, MPI_COMM_WORLD) ;
if ( myid == 0 ) {
   fprintf(msgFile, "\n\n checksums for original matrix") ;
   DVfprintf(msgFile, 2, schecksums) ;
   fprintf(msgFile, "\n checksums for distributed matrix") ;
   DVfprintf(msgFile, 2, tchecksums) ;
   tchecksums[0] -= schecksums[0] ;
   tchecksums[1] -= schecksums[1] ;
   fprintf(msgFile, "\n errors ") ;
   DVfprintf(msgFile, 2, tchecksums) ;
}
/*
   ----------------
   free the objects
   ----------------
*/
IV_free(mapIV) ;
DDenseMtx_free(mtx) ;

MPI_Finalize() ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
