/*  testOrder.c  */

#include "../spoolesMPI.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ----------------------------------------------------
   (1) process 0 reads in a DInpMtx object. 
   (2) using a wrap map, distribute the object
   (3) using a random map, distribute the object again
   (4) get an ordering of the global graph

   created -- 97dec17, cca
   ----------------------------------------------------
*/
{
char         *inDInpMtxFileName ;
char         *buffer ;
DInpMtx      *inpmtx, *inpmtxkeep ;
double       ops, t1, t2 ;
double       schecksums[3], pchecksums[3], tchecksums[3] ;
Drand        *drand ;
ETree        *frontETree ;
int          coordType, inputMode, iproc, length, maxfrontsize,
             maxzeros, myid, msglvl, neqns, nproc, nzf, rc, seed, 
             tag, v ;
int          ibuffer[3], stats[4] ;
int          *map ;
IV           *mapIV ;
FILE         *msgFile ;
MPI_Status   status ;
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n process %d of %d, argc = %d", myid, nproc, argc) ;
fflush(stdout) ;
if ( argc != 10 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile neqns seed "
"\n         coordType inputMode inDInpMtxFile "
"\n    msglvl     -- message level"
"\n    msgFile    -- message file"
"\n    neqns      -- number of equations"
"\n    seed       -- random number seed"
"\n    coordType  -- coordinate type"
"\n       1 -- store by rows"
"\n       2 -- store by columns"
"\n       3 -- store by chevrons"
"\n    inputMode -- input mode"
"\n       1 -- indices only"
"\n       2 -- indices and entries"
"\n    inDInpMtxFile -- input file, must be *.dinpmtxf or *.dinpmtxb"
"\n    maxzeros     -- maximum # of zeros in a front"
"\n    maxfrontsize -- maximum front size"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   length = strlen(argv[2]) + 1 + 4 ;
   buffer = CVinit(length, '\0') ;
   sprintf(buffer, "%s.%d", argv[2], myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], argv[2]) ;
      return ;
   }
   CVfree(buffer) ;
}
neqns             = atoi(argv[3]) ;
seed              = atoi(argv[4]) ;
coordType         = atoi(argv[5]) ;
inputMode         = atoi(argv[6]) ;
inDInpMtxFileName = argv[7] ;
maxzeros          = atoi(argv[8]) ;
maxfrontsize      = atoi(argv[9]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl        -- %d" 
        "\n msgFile       -- %s" 
        "\n neqns         -- %d" 
        "\n seed          -- %d" 
        "\n coordType     -- %d" 
        "\n inputMode     -- %d" 
        "\n inDInpMtxFile -- %s" 
        "\n maxzeros      -- %d" 
        "\n maxfrontsize  -- %d" 
        "\n",
        argv[0], msglvl, argv[2], neqns, seed, coordType, 
        inputMode, inDInpMtxFileName, maxzeros, maxfrontsize) ;
fflush(msgFile) ;
/*
   --------------------------
   read in the DInpMtx object
   --------------------------
*/
tag = 1 ;
inpmtx = DInpMtx_new() ;
if ( myid == 0 ) {
   MARKTIME(t1) ;
   rc = DInpMtx_readFromFile(inpmtx, inDInpMtxFileName) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : read in inpmtx from file %s",
           t2 - t1, inDInpMtxFileName) ;
   if ( rc != 1 ) {
      fprintf(msgFile, 
              "\n return value %d from DInpMtx_readFromFile(%p,%s)",
              rc, inpmtx, inDInpMtxFileName) ;
      exit(-1) ;
   }
/*
   --------------------------------------
   change the coordinate type to chevrons
   and the storage mode to sorted triples
   --------------------------------------
*/
   DInpMtx_changeCoordType(inpmtx, coordType) ;
   DInpMtx_changeStorageMode(inpmtx, 2) ;
   inpmtx->inputMode = inputMode ;
   fprintf(msgFile, "\n\n after reading DInpMtx object from file %s",
           inDInpMtxFileName) ;
   if ( msglvl > 2 ) {
      DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   } else {
      DInpMtx_writeStats(inpmtx, msgFile) ;
   }
   fflush(msgFile) ;
/*
   ------------------------------------------
   compute the serial checksums of the matrix
   ------------------------------------------
*/
   {  int      *ivec1 = DInpMtx_ivec1(inpmtx) ;
      int      *ivec2 = DInpMtx_ivec2(inpmtx) ;
      double   *dvec  = DInpMtx_dvec(inpmtx)  ;
      int      ii, nent   = DInpMtx_nent(inpmtx)  ;

      schecksums[0] = schecksums[1] = schecksums[2] = 0.0 ;
      for ( ii = 0 ; ii < nent ; ii++ ) {
         schecksums[0] += ivec1[ii] ;
         schecksums[1] += ivec2[ii] ;
         schecksums[2] += fabs(dvec[ii])  ;
      }
   }
/*
   -------------------------------------------------------
   send the first message identifying the coordinate type,
   storage mode and input mode
   -------------------------------------------------------
*/
   ibuffer[0] = DInpMtx_coordType(inpmtx) ;
   ibuffer[1] = DInpMtx_storageMode(inpmtx) ;
   ibuffer[2] = DInpMtx_inputMode(inpmtx) ;
   MARKTIME(t1) ;
   for ( iproc = 1 ; iproc < nproc ; iproc++ ) {
      MPI_Send((void *) ibuffer, 3, MPI_INT, 
               iproc, tag, MPI_COMM_WORLD);
   }
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : first message sent", t2 - t1) ;
} else {
/*
   -------------------------------------------------------
   send the first message identifying the coordinate type,
   storage mode and input mode
   -------------------------------------------------------
*/
   MARKTIME(t1) ;
   MPI_Recv((void *) ibuffer, 3, MPI_INT, 
            0, tag, MPI_COMM_WORLD, &status) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : first message received", t2 - t1) ;
   DInpMtx_init(inpmtx, ibuffer[0], ibuffer[2], 0, 0) ;
   DInpMtx_changeStorageMode(inpmtx, 1) ;
   fprintf(msgFile, "\n\n after receiving initial information") ;
   if ( msglvl > 2 ) {
      DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   } else {
      DInpMtx_writeStats(inpmtx, msgFile) ;
   }
}
/*
   ------------------------------------------
   set the initial owners IV to be a wrap map
   ------------------------------------------
*/
mapIV = IV_new() ;
IV_init(mapIV, neqns, NULL) ;
map = IV_entries(mapIV) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = v % nproc ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------------
   split the DInpMtx object into pieces
   ------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
tag++ ;
MARKTIME(t1) ;
inpmtxkeep = DInpMtx_MPI_split(inpmtx, mapIV, tag, stats,
                              msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : matrix split", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
DInpMtx_free(inpmtx) ;
inpmtx = inpmtxkeep ;
tag += 2 ;
fprintf(msgFile, 
          "\n\n after splitting the DInpMtx object with the wrap map") ;
if ( msglvl > 2 ) {
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
} else {
   DInpMtx_writeStats(inpmtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------
   generate a random map
   ---------------------
*/
MARKTIME(t1) ;
drand = Drand_new() ;
Drand_setSeed(drand, seed) ;
Drand_setUniform(drand, 0.0, (double) neqns) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = ((int) Drand_value(drand)) % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : random map set", t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n map") ;
   IV_writeForHumanEye(mapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------
   now split the matrix with the random map
   ----------------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
inpmtxkeep = DInpMtx_MPI_split(inpmtx, mapIV, tag, stats,
                               msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : matrix split", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
DInpMtx_free(inpmtx) ;
inpmtx = inpmtxkeep ;
fprintf(msgFile, 
        "\n\n after splitting the DInpMtx object with the owners map") ;
if ( msglvl > 2 ) {
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
} else {
   DInpMtx_writeStats(inpmtx, msgFile) ;
}
fflush(msgFile) ;
/*
   -----------------------------------------------
   compute the checksums of the distributed matrix
   -----------------------------------------------
*/
{  int      *ivec1 = DInpMtx_ivec1(inpmtx) ;
   int      *ivec2 = DInpMtx_ivec2(inpmtx) ;
   double   *dvec  = DInpMtx_dvec(inpmtx)  ;
   int      ii, nent   = DInpMtx_nent(inpmtx)  ;

   pchecksums[0] = pchecksums[1] = pchecksums[2] = 0.0 ;
   for ( ii = 0 ; ii < nent ; ii++ ) {
      pchecksums[0] += ivec1[ii] ;
      pchecksums[1] += ivec2[ii] ;
      pchecksums[2] += fabs(dvec[ii])  ;
   }
}
MPI_Reduce((void *) pchecksums, (void *) tchecksums, 3, MPI_DOUBLE,
           MPI_SUM, 0, MPI_COMM_WORLD) ;
if ( myid == 0 ) {
   fprintf(msgFile, "\n\n checksums for original matrix") ;
   DVfprintf(msgFile, 3, schecksums) ;
   fprintf(msgFile, "\n checksums for distributed matrix") ;
   DVfprintf(msgFile, 3, tchecksums) ;
   tchecksums[0] -= schecksums[0] ;
   tchecksums[1] -= schecksums[1] ;
   tchecksums[2] -= schecksums[2] ;
   fprintf(msgFile, "\n errors ") ;
   DVfprintf(msgFile, 3, tchecksums) ;
}
/*
   ----------------------------
   get an ordering of the graph
   ----------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
MARKTIME(t1) ;
frontETree = DInpMtx_MPI_orderViaMMD(inpmtx, maxzeros, maxfrontsize, 
                seed + myid, stats, msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : order graph", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
nzf = ETree_nFactorEntries(frontETree, 2) ;
ops = ETree_nFactorOps(frontETree, 2) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n front tree %d entries, %.0f operations",
           nzf, ops) ;
   fflush(msgFile) ;
}
if ( msglvl > 0 ) {
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------
   free the objects
   ----------------
*/
IV_free(mapIV) ;
ETree_free(frontETree) ;

MPI_Finalize() ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
