/*  testFactor.c  */

#include "../spoolesMPI.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ----------------------------------------
   (1) process 0 reads in a DInpMtx object. 
   (2) read in an ETree object. (global)
   (3) read in an ownersIV object. (global)
   (4) split the DInpMtx object into pieces
   (5) set up for the factorization
   (6) compute the factorization
   

   created -- 97jun20, cca
   ----------------------------------------
*/
{
char         *buffer, *inpmtxAfileName, *inpmtxBfileName, 
             *inETreeFileName, *inOwnersFileName ;
DChv         *rootchv ;
DDenseMtx    *keepmtx, *solmtx ;
DDenseMtxManager   *manager ;
DFrontMtx    *frontmtx ;
DLinSystem   *linsys ;
Drand        drand ;
double       droptol, misccpu, factorcpu, sigma, solvecpu, tau, t1, t2 ;
double       cpus[20] ;
ETree        *frontETree ;
int          firsttag, J, length, loc, lookahead, msglvl, 
             myid, nent, neqns, nfront, nproc, nrhs, pivotingflag, 
             randomflag, rc, seed, sparsityflag, symmetryflag, v ;
int          *map, *owners, *vtxmap, *vtxToFront ;
int          stats[20] ;
IV           *colmapIV, *mapIV, *oldToNewIV, *ownersIV, 
             *rowmapIV, *vtxmapIV ;
IVL          *symbfacIVL ;
FILE         *msgFile ;
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n process %d of %d, argc = %d", myid, nproc, argc) ;
fflush(stdout) ;
if ( argc != 17 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile inETreeFile inpmtxAfile"
      "\n         sigma inpmtxBfile inOwnersFile seed "
      "\n         symmetryflag sparsityflag pivotingflag "
      "\n         tau droptol randomflag lookahead nrhs"
      "\n    msglvl        -- message level"
      "\n    msgFile       -- message file"
      "\n    inETreeFile   -- front tree file"
      "\n    inpmtxAfile   -- matrix file"
      "\n    sigma         -- matrix file"
      "\n    inpmtxBfile   -- matrix file"
      "\n    inOwnersFile  -- owners map file"
      "\n    seed          -- random number seed"
      "\n    symmetryflag -- symmetry flag"
      "\n       0 --> symmetric structure, symmetric entries"
      "\n       1 --> symmetric structure, nonsymmetric entries"
      "\n       2 --> nonsymmetric structure, nonsymmetric entries"
      "\n       3 --> matrix from QR factorization"
      "\n    sparsityflag -- sparsity flag"
      "\n       0 --> store dense fronts"
      "\n       1 --> store sparse fronts, use droptol to drop entries"
      "\n    pivotingflag -- pivoting flag"
      "\n       0 --> do not pivot"
      "\n       1 --> enable pivoting"
      "\n    tau     -- upper bound on factor entries"
      "\n               used only with pivoting"
      "\n    droptol -- lower bound on factor entries"
      "\n               used only with sparse fronts"
      "\n    randomflag -- random number flag"
      "\n       0 --> use matrix entries from file"
      "\n       1 --> use random matrix entries"
      "\n    lookahead -- parameter to schedule computation"
      "\n       0 --> no lookahead"
      "\n       nonzero --> look up the tree this many steps"
      "\n    nrhs -- number of right hand sides"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   length = strlen(argv[2]) + 1 + 4 ;
   buffer = CVinit(length, '\0') ;
   sprintf(buffer, "%s.%d", argv[2], myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], buffer) ;
      return ;
   }
   CVfree(buffer) ;
}
inETreeFileName   = argv[3] ;
inpmtxAfileName   = argv[4] ;
sigma             = atof(argv[5]) ;
inpmtxBfileName   = argv[6] ;
inOwnersFileName  = argv[7] ;
seed              = atoi(argv[8]) ;
symmetryflag      = atoi(argv[9]) ;
sparsityflag      = atoi(argv[10]) ;
pivotingflag      = atoi(argv[11]) ;
tau               = atof(argv[12]) ;
droptol           = atof(argv[13]) ;
randomflag        = atoi(argv[14]) ;
lookahead         = atoi(argv[15]) ;
nrhs              = atoi(argv[16]) ;
fprintf(msgFile,
        "\n %s "
        "\n msglvl        -- %d"
        "\n msgFile       -- %s"
        "\n inETreeFile   -- %s"
        "\n inpmtxAfile   -- %s"
        "\n sigma         -- %f"
        "\n inpmtxBfile   -- %s"
        "\n inOwnersFile  -- %s"
        "\n seed          -- %d"
        "\n symmetryflag  -- %d"
        "\n sparsityflag  -- %d"
        "\n pivotingflag  -- %d"
        "\n tau           -- %e"
        "\n droptol       -- %e"
        "\n randomflag    -- %d"
        "\n lookahead     -- %d"
        "\n nrhs          -- %d"
        "\n",
        argv[0], msglvl, argv[2], inETreeFileName, inpmtxAfileName,
        sigma, inpmtxBfileName, inOwnersFileName, seed, symmetryflag, 
        sparsityflag, pivotingflag, tau, droptol, randomflag, 
        lookahead, nrhs) ;
fflush(msgFile) ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setNormal(&drand, 0.0, 1.0) ;
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
if ( strcmp(inETreeFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
frontETree = ETree_new() ;
MARKTIME(t1) ;
oldToNewIV = ETree_initFromFile(frontETree, inETreeFileName,
                                msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : read in frontETree from file %s",
        t2 - t1, inETreeFileName) ;
neqns  = frontETree->nvtx   ;
nfront = frontETree->nfront ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after reading ETree object from file %s",
           inETreeFileName) ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
}
fflush(msgFile) ;
/*
   ------------------------
   set up the linear system
   ------------------------
*/
MARKTIME(t1) ;
Drand_setSeed(&drand, seed) ;
linsys = DLinSystem_new() ;
DLinSystem_setup(linsys, myid, neqns, nrhs, inpmtxAfileName, sigma, 
                 inpmtxBfileName, symmetryflag, randomflag, &drand, 
                 msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : linear system set up", t2 - t1) ;
/*
   ------------------------------------------
   set the initial owners IV to be a wrap map
   ------------------------------------------
*/
MARKTIME(t1) ;
mapIV = IV_new() ;
IV_init(mapIV, neqns, NULL) ;
map = IV_entries(mapIV) ;
for ( v = 0 ; v < neqns ; v++ ) {
   map[v] = v % nproc ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : wrap map set ", t2 - t1) ;
/*
   -----------------------------------
   split the linear system into pieces
   -----------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
firsttag = 1 ;
MARKTIME(t1) ;
DLinSystem_MPI_split(linsys, mapIV, firsttag, stats,
                    msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : linear system split ", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
firsttag += 10 ;
fprintf(msgFile, "\n\n after splitting the DLinSystem object") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
/*
   -------------------------
   permute the linear system
   -------------------------
*/
MARKTIME(t1) ;
DLinSystem_permute(linsys, oldToNewIV, msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : linear system permuted ", t2 - t1) ;
fprintf(msgFile, "\n\n after permuting the DLinSystem object") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
/*
   ----------------------------
   read in the owners IV object
   ----------------------------
*/
if ( strcmp(inOwnersFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
ownersIV = IV_new() ;
MARKTIME(t1) ;
rc = IV_readFromFile(ownersIV, inOwnersFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in mapIV from file %s",
        t2 - t1, inOwnersFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from IV_readFromFile(%p,%s)",
           rc, mapIV, inOwnersFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading IV object from file %s",
        inOwnersFileName) ;
if ( msglvl > 2 ) {
   IV_writeForHumanEye(ownersIV, msgFile) ;
} else {
   IV_writeStats(ownersIV, msgFile) ;
}
fflush(msgFile) ;
if ( nproc != 1 + IV_max(ownersIV) ) {
   fprintf(stderr, "\n fatal error in testSymbfac"
           "\n nproc = %d from communicator"
           "\n nproc = %d from ownersIV object\n",
           nproc, 1 + IV_max(ownersIV)) ;
   exit(-1) ;
}
/*
   ------------------------------
   generate the vertex owners map
   ------------------------------
*/
MARKTIME(t1) ;
vtxmapIV = IV_new() ;
IV_init(vtxmapIV, neqns, NULL) ;
owners = IV_entries(ownersIV) ;
vtxmap = IV_entries(vtxmapIV) ;
vtxToFront = ETree_vtxToFront(frontETree) ;
for ( v = 0 ; v < neqns ; v++ ) {
   J = vtxToFront[v] ;
   vtxmap[v] = owners[J] ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : vtx map set up ", t2 - t1) ;
/*
   ------------------------------
   redistribute the linear system 
   ------------------------------
*/
stats[0] = stats[1] = stats[2] = stats[3] = 0 ;
firsttag = 1 ;
MARKTIME(t1) ;
DLinSystem_MPI_split(linsys, vtxmapIV, firsttag, stats,
                     msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : linear system split ", t2 - t1) ;
fprintf(msgFile, 
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
firsttag += 10 ;
fprintf(msgFile, "\n\n after redistributing the DLinSystem object") ;
if ( msglvl > 2 ) {
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
} else {
   DLinSystem_writeStats(linsys, msgFile) ;
}
/*
   ------------------------------
   get the symbolic factorization
   ------------------------------
*/
DPencil_changeStorageMode(linsys->pencil, 3) ;
fprintf(msgFile, 
        "\n\n ###  getting ready to call SymbFac_MPI_initFromDPencil") ;
fflush(msgFile) ;
MARKTIME(t1) ;
stats[0] = stats[1] = stats[2] = stats[3] = 0 ; 
symbfacIVL = SymbFac_MPI_initFromDPencil(frontETree, linsys->pencil, 
                                      ownersIV, firsttag, stats,
                                      msglvl, msgFile, MPI_COMM_WORLD) ;
MPI_Barrier(MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : symbolic factorization done ", t2-t1);
fprintf(msgFile, 
        "\n %d sends, %d bytes sent, %d recv's %d bytes recv'd",
        stats[0], stats[1], stats[2], stats[3]) ;
firsttag += nfront ;
fprintf(msgFile, "\n\n symbfacIVL") ;
if ( msglvl > 1 ) {
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
} else {
   IVL_writeStats(symbfacIVL, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------------------------
   initialize the front matrix object
   ----------------------------------
*/
fprintf(msgFile, "\n nfront = %d, neqns = %d", nfront, neqns) ;
MARKTIME(t1) ;
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL,
               symmetryflag, sparsityflag, pivotingflag, 
               0, myid, ownersIV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : initialize the front matrix",
        t2 - t1) ;
fprintf(msgFile,
        "\n diagDVL->tsize  = %d"
        "\n lowerDVL->tsize = %d"
        "\n upperDVL->tsize = %d",
        (frontmtx->diagDVL  != NULL) ? frontmtx->diagDVL->tsize  : 0,
        (frontmtx->lowerDVL != NULL) ? frontmtx->lowerDVL->tsize : 0,
        (frontmtx->upperDVL != NULL) ? frontmtx->upperDVL->tsize : 0) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n front matrix initialized") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   -------------------------
   compute the factorization
   -------------------------
*/
DVzero(20, cpus) ;
IVzero(20, stats) ;
MARKTIME(t1) ;
rootchv = DFrontMtx_MPI_factorDPencil(frontmtx, linsys->pencil, 
             tau, droptol, firsttag, ownersIV, lookahead, cpus, 
             stats, msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
factorcpu = t2 - t1 ;
fprintf(msgFile, "\n CPU %8.3f : numeric factorization done ", t2-t1);
fflush(msgFile) ;
misccpu = factorcpu - DVsum(9, cpus) - DVsum(4, cpus + 10) ;
if ( factorcpu > 0.0 ) {
   fprintf(msgFile,
   "\n    initialize fronts              %8.3f %6.2f"
   "\n    load original entries          %8.3f %6.2f"
   "\n    update fronts                  %8.3f %6.2f"
   "\n    insert aggregate data          %8.3f %6.2f"
   "\n    assemble aggregate data        %8.3f %6.2f"
   "\n    assemble postponed data        %8.3f %6.2f"
   "\n    factor fronts                  %8.3f %6.2f"
   "\n    extract postponed data         %8.3f %6.2f"
   "\n    store factor entries           %8.3f %6.2f"
   "\n    post initial receives          %8.3f %6.2f"
   "\n    check for received messages    %8.3f %6.2f"
   "\n    post initial sends             %8.3f %6.2f"
   "\n    check for sent messages        %8.3f %6.2f"
   "\n    miscellaneous                  %8.3f %6.2f"
   "\n    total time                     %8.3f",
   cpus[0], 100.*cpus[0]/factorcpu,
   cpus[1], 100.*cpus[1]/factorcpu,
   cpus[2], 100.*cpus[2]/factorcpu,
   cpus[3], 100.*cpus[3]/factorcpu,
   cpus[4], 100.*cpus[4]/factorcpu,
   cpus[5], 100.*cpus[5]/factorcpu,
   cpus[6], 100.*cpus[6]/factorcpu,
   cpus[7], 100.*cpus[7]/factorcpu, 
   cpus[8], 100.*cpus[8]/factorcpu, 
   cpus[10], 100.*cpus[10]/factorcpu, 
   cpus[11], 100.*cpus[11]/factorcpu, 
   cpus[12], 100.*cpus[12]/factorcpu, 
   cpus[13], 100.*cpus[13]/factorcpu, 
   misccpu, 100.*misccpu/factorcpu,
   factorcpu) ;
}
fprintf(msgFile, 
        "\n\n COMMUNICATION STATISTICS"
        "\n aggregate info"
        "\n %8d sends, %10d bytes sent, %8d recv's, %10d bytes recv'd"
        "\n postponed info"
        "\n %8d sends, %10d bytes sent, %8d recv's, %10d bytes recv'd",
        stats[0], stats[1], stats[2], stats[3],
        stats[4], stats[5], stats[6], stats[7]) ;
fprintf(msgFile, 
       "\n\n WORKING STORAGE STATISTICS"
       "\n %d active DChv objects, %d active bytes, %d requested bytes",
        stats[8], stats[9], stats[10]) ;
firsttag += 4*nfront + 2 ;
fprintf(msgFile, "\n\n front matrix") ;
if ( msglvl > 2 ) {
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
} else {
   DFrontMtx_writeStats(frontmtx, msgFile) ;
}
/*
DVL_writeForHumanEye(frontmtx->diagDVL, msgFile) ;
if ( IV_entry(ownersIV, frontmtx->nfront - 1) == myid ) {
   double   *diagent ;
   int      nent ;
   DFrontMtx_diagEntries(frontmtx, frontmtx->nfront - 1,
                         &nent, &diagent) ;
   fprintf(msgFile, "\n\n diag entries for top front") ;
   DVfprintf(msgFile, nent, diagent) ;
}
fflush(msgFile) ;
*/
fflush(msgFile) ;
MPI_Barrier(MPI_COMM_WORLD) ;
/*
   -------------------------------------------------
   sort the rows of L and columns of U in each front
   -------------------------------------------------
*/
MARKTIME(t1) ;
DFrontMtx_MPI_permuteFactor(frontmtx, ownersIV, firsttag, stats,
                            msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute factor ", t2-t1);
firsttag += 2 ;
fprintf(msgFile, "\n\n after permuting factor, front matrix") ;
if ( msglvl > 3 ) {
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
} else {
   DFrontMtx_writeStats(frontmtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------------------------------
   convert the boundary indices to local indices
   ---------------------------------------------
*/
MARKTIME(t1) ;
DFrontMtx_MPI_postProcess(frontmtx, 0, 1, ownersIV, firsttag, stats,
                          msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set local indices ", t2-t1);
firsttag += 2 ;
fprintf(msgFile, "\n\n after setting local indices, front matrix") ;
if ( msglvl > 3 ) {
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
} else {
   DFrontMtx_writeStats(frontmtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------------------------------------------------
   redistribute the right hand side and solution if necessary
   ----------------------------------------------------------
*/
MARKTIME(t1) ;
if ( pivotingflag == 1 ) {
   colmapIV = DFrontMtx_MPI_colmap(frontmtx, ownersIV, 
                                   msglvl, msgFile, MPI_COMM_WORLD) ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n column map IV object") ;
      IV_writeForHumanEye(colmapIV, msgFile) ;
      fflush(msgFile) ;
   }
   IVfill(4, stats, 0) ;
   MARKTIME(t1) ;
   linsys->solmtx = DDenseMtx_MPI_splitByRows(linsys->solmtx, colmapIV, 
                     firsttag, stats, msglvl, msgFile, MPI_COMM_WORLD) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %8.3f : solution matrix split ", t2 - t1) ;
   fprintf(msgFile, 
           "\n send stats : %d messages with %d bytes"
           "\n recv stats : %d messages with %d bytes",
           stats[0], stats[2], stats[1], stats[3]) ;
   if ( symmetryflag == 2 ) {
      rowmapIV = DFrontMtx_MPI_rowmap(frontmtx, ownersIV, 
                                     msglvl, msgFile, MPI_COMM_WORLD) ;
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n row map IV object") ;
         IV_writeForHumanEye(rowmapIV, msgFile) ;
         fflush(msgFile) ;
      }
      IVfill(4, stats, 0) ;
      MARKTIME(t1) ;
      linsys->rhsmtx = DDenseMtx_MPI_splitByRows(linsys->rhsmtx, 
                              rowmapIV, firsttag, stats,
                              msglvl, msgFile, MPI_COMM_WORLD) ;
      MARKTIME(t2) ;
      fprintf(msgFile, "\n CPU %8.3f : rhs matrix split ", t2 - t1) ;
      fprintf(msgFile, 
              "\n send stats : %d messages with %d bytes"
              "\n recv stats : %d messages with %d bytes",
              stats[0], stats[2], stats[1], stats[3]) ;
   } else {
      IVfill(4, stats, 0) ;
      MARKTIME(t1) ;
      linsys->rhsmtx = DDenseMtx_MPI_splitByRows(linsys->rhsmtx, 
                              colmapIV, firsttag, stats,
                              msglvl, msgFile, MPI_COMM_WORLD) ;
      MARKTIME(t2) ;
      fprintf(msgFile, "\n CPU %8.3f : rhs matrix split ", t2 - t1) ;
      fprintf(msgFile, 
              "\n send stats : %d messages with %d bytes"
              "\n recv stats : %d messages with %d bytes",
              stats[0], stats[2], stats[1], stats[3]) ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n solution matrix") ;
      DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
      fprintf(msgFile, "\n\n right hand side matrix") ;
      DDenseMtx_writeForHumanEye(linsys->rhsmtx, msgFile) ;
      fflush(msgFile) ;
   }
}
MPI_Barrier(MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : solution and rhs set up ", t2-t1);
/*
   ----------------------------------
   make a copy of the solution matrix
   ----------------------------------
*/
MARKTIME(t1) ;
keepmtx = DDenseMtx_new() ;
solmtx  = linsys->solmtx ;
DDenseMtx_init(keepmtx, solmtx->rowid, solmtx->colid, solmtx->nrow,
                        solmtx->ncol, solmtx->inc1, solmtx->inc2) ;
nent = solmtx->nrow * solmtx->ncol ;
DVcopy(nent, keepmtx->entries, solmtx->entries) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : copy of solution made ", t2-t1);
MPI_Barrier(MPI_COMM_WORLD) ;
/*
   -----------------------
   solve the linear system
   -----------------------
*/
fprintf(msgFile, "\n maxabs(rhs) = %12.4e",
        DVmax(linsys->rhsmtx->nrow*linsys->rhsmtx->ncol,
              linsys->rhsmtx->entries, &loc)) ;
DVzero(linsys->solmtx->nrow*linsys->solmtx->ncol,
              linsys->solmtx->entries) ;
fprintf(msgFile, "\n before, maxabs(sol) = %12.4e",
        DVmax(linsys->solmtx->nrow*linsys->solmtx->ncol,
              linsys->solmtx->entries, &loc)) ;
IVzero(4,  stats) ;
DVzero(10, cpus) ;
MARKTIME(t1) ;
manager = DDenseMtxManager_new() ;
DFrontMtx_MPI_solve(frontmtx, linsys->solmtx, linsys->rhsmtx, manager,
                    ownersIV, firsttag, stats, cpus, 
                    msglvl, msgFile, MPI_COMM_WORLD);
MARKTIME(t2) ;
solvecpu = t2 - t1 ;
fprintf(msgFile, "\n CPU %8.3f : solve done ", solvecpu) ;
misccpu = solvecpu - DVsum(10, cpus) ;
fprintf(msgFile, "\n\n SOLVE STATISTICS:") ;
fprintf(msgFile, 
        "\n initialize the matrices        %8.3f %6.2f"
        "\n load the rhs and solution      %8.3f %6.2f"
        "\n assemble children and parents  %8.3f %6.2f"
        "\n solve and update               %8.3f %6.2f"
        "\n store rhs and solution         %8.3f %6.2f"
        "\n store updates                  %8.3f %6.2f"
        "\n post initial receives          %8.3f %6.2f"
        "\n check for received messages    %8.3f %6.2f"
        "\n post initial sends             %8.3f %6.2f"
        "\n check for sent messages        %8.3f %6.2f"
        "\n miscellaneous                  %8.3f %6.2f"
        "\n TOTAL                          %8.3f "
        "\n %8d messages sent,     %8d bytes sent"
        "\n %8d messages received, %8d bytes received",
        cpus[0], 100.*cpus[0]/solvecpu,
        cpus[1], 100.*cpus[1]/solvecpu,
        cpus[2], 100.*cpus[2]/solvecpu,
        cpus[3], 100.*cpus[3]/solvecpu,
        cpus[4], 100.*cpus[4]/solvecpu,
        cpus[5], 100.*cpus[5]/solvecpu,
        cpus[6], 100.*cpus[6]/solvecpu,
        cpus[7], 100.*cpus[7]/solvecpu,
        cpus[8], 100.*cpus[8]/solvecpu,
        cpus[9], 100.*cpus[9]/solvecpu,
        misccpu, 100.*misccpu/solvecpu,
        solvecpu,
        stats[0], stats[1], stats[2], stats[3]
) ;
fprintf(msgFile, "\n\n DDenseMtxManager statistics") ;
fprintf(msgFile, 
        "\n %d active objects, %d active bytes, %d requested bytes",
        manager->nactive, manager->nbytesactive,
        manager->nbytesrequested) ;

if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n after solve, solution matrix") ;
   DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
   fflush(msgFile) ;
}
fprintf(msgFile, "\n after, maxabs(sol) = %12.4e",
        DVmax(linsys->solmtx->nrow*linsys->solmtx->ncol,
              linsys->solmtx->entries, &loc)) ;
/*
   -----------------
   compute the error
   -----------------
*/
fprintf(msgFile, "\n maxabs(keepmtx) = %12.4e",
        DVmax(keepmtx->nrow*keepmtx->ncol,
              keepmtx->entries, &loc)) ;
DVsub(nent, solmtx->entries, keepmtx->entries) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n error") ;
   DDenseMtx_writeForHumanEye(solmtx, msgFile) ;
}
fprintf(msgFile, "\n maxabs(error) = %12.4e",
        DVmaxabs(nent, solmtx->entries, &loc)) ;
/*
   ----------------------------------------------
   convert the boundary indices to global indices
   ----------------------------------------------
*/
MARKTIME(t1) ;
DFrontMtx_MPI_postProcess(frontmtx, 0, 2, ownersIV, firsttag, stats,
                          msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set local indices ", t2-t1);
firsttag += 2 ;
fprintf(msgFile, "\n\n after setting global indices, front matrix") ;
if ( msglvl > 2 ) {
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
} else {
   DFrontMtx_writeStats(frontmtx, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------
   free the objects
   ----------------
*/
/*
IV_free(mapIV) ;
IVL_free(symbfacIVL) ;
IV_free(ownersIV) ;
DLinSystem_free(linsys) ;
ETree_free(frontETree) ;
*/

MPI_Finalize() ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
