/*  allInOne.c  */

#include "../spoolesMPI.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
void
main ( int argc, char *argv[] ) {
/*
   ------------------------------------------------------------
   all-in-one MPI program for each process
   ( 1) read in matrix entries and form DInpMtx object
        for a symmetric matrix
   ( 2) read in right hand side entries 
        and form dense matrix DDenseMtx object
   ( 3) for the linear system DLinSystem object
   ( 4) order the system using minimum degree
   ( 5) permute the linear system
   ( 6) create the owners map IV object
   ( 7) redistribute the linear system
   ( 8) compute the symbolic factorization 
   ( 9) compute the numeric factorization without pivoting
   (10) solve the linear system

   created -- 97nov17, cca
   ------------------------------------------------------------
*/
/*--------------------------------------------------------------------*/
char               buffer[20] ;
DChv               *rootchv ;
DDenseMtx          *rhsmtx, *solmtx ;
DDenseMtxManager   *manager ;
DFrontMtx          *frontmtx ;
DInpMtx            *inpmtx ;
DLinSystem         *linsys ;
DPencil            *pencil ;
double             cutoff, factorcpu, misccpu, t1, t2, value ;
double             cpus[20] ;
double             *entries ;
DV                 *cumopsDV ;
ETree              *frontETree ;
FILE               *inputFile, *msgFile ;
int                firsttag, ient, irow, jcol, maxfrontsize, 
                   maxzeros, msglvl, myid, nent, neqns, nfront, 
                   nproc, nrhs, nrow, seed, v ;
int                stats[20] ;
int                *owners, *rowind, *vtxmap, *vtxToFront ;
IV                 *oldToNewIV, *ownersIV, *vtxmapIV ;
IVL                *symbfacIVL ;
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   find out the identity of this process and the number of process
   ---------------------------------------------------------------
*/
MPI_Init(&argc, &argv) ;
MPI_Comm_rank(MPI_COMM_WORLD, &myid) ;
MPI_Comm_size(MPI_COMM_WORLD, &nproc) ;
fprintf(stdout, "\n proc %d, argc = %d", myid, argc) ;
fflush(stdout) ;
/*--------------------------------------------------------------------*/
/*
   --------------------
   get input parameters
   --------------------
*/
if ( argc != 3 ) {
   fprintf(stdout, 
      "\n usage: %s msglvl msgFile "
      "\n    msglvl -- message level"
      "\n    msgFile -- message file"
      "\n   note: matrix entries are read in from matrix.k.input"
      "\n         where k is the process number"
      "\n   note: rhs entries are read in from rhs.k.input"
      "\n         where k is the process number"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else {
   sprintf(buffer, "res.%d", myid) ;
   if ( (msgFile = fopen(buffer, "w")) == NULL ) {
      fprintf(stderr, "\n fatal error in %s"
              "\n unable to open file %s\n",
              argv[0], buffer) ;
      return ;
   }
}
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   read the entries from the input file and create the DInpMtx object
   ------------------------------------------------------------------
*/
sprintf(buffer, "matrix.%d.input", myid) ;
inputFile = fopen(buffer, "r") ;
fscanf(inputFile, "%d %d %d", &neqns, &neqns, &nent) ;
inpmtx = DInpMtx_new() ;
DInpMtx_init(inpmtx, 1, 2, nent, 0) ;
for ( ient = 0 ; ient < nent ; ient++ ) {
   fscanf(inputFile, "%d %d %le", &irow, &jcol, &value) ;
   DInpMtx_inputEntry(inpmtx, irow, jcol, value) ;
}
fclose(inputFile) ;
DInpMtx_changeStorageMode(inpmtx, 3) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n input matrix") ;
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------
   read the entries from the rhs input file 
   and create the DDenseMtx object
   ----------------------------------------
*/
sprintf(buffer, "rhs.%d.input", myid) ;
inputFile = fopen(buffer, "r") ;
fscanf(inputFile, "%d %d", &nrow, &nrhs) ;
rhsmtx = DDenseMtx_new() ;
DDenseMtx_init(rhsmtx, myid, -1, nrow, nrhs, 1, nrow) ;
rowind  = rhsmtx->rowind ;
entries = rhsmtx->entries ;
for ( irow = 0 ; irow < nent ; irow++ ) {
   fscanf(inputFile, "%d", rowind + irow) ;
   for ( jcol = 0 ; jcol < nrhs ; jcol++ ) {
      fscanf(inputFile, "%le", entries + irow + jcol*nrow) ;
   }
}
fclose(inputFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n rhs matrix") ;
   DDenseMtx_writeForHumanEye(rhsmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------------
   form a linear system object
   ---------------------------
*/
pencil = DPencil_new() ;
DPencil_init(pencil, inpmtx, 0.0, NULL) ;
linsys = DLinSystem_new() ;
solmtx = DDenseMtx_new() ;
DDenseMtx_init(solmtx, myid, -1, rhsmtx->nrow, rhsmtx->ncol,
               rhsmtx->inc1, rhsmtx->inc2) ;
IVcopy(solmtx->nrow, solmtx->rowind, rhsmtx->rowind) ;
IVramp(solmtx->nrow, solmtx->colind, 0, 1) ;
DLinSystem_init(linsys, nrow, nrhs, 0, pencil, solmtx, rhsmtx) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n linear system") ;
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------------------------------
   compute the front tree ETree object using the MMD ordering
   ----------------------------------------------------------
*/
maxzeros     = .01 * nrow ;
maxfrontsize = 32 ;
seed         = 131279 * myid + 593651 ;
 
frontETree = DInpMtx_MPI_orderViaMMD(inpmtx, maxzeros, maxfrontsize,
                                     seed, stats, msglvl, msgFile,
                                     MPI_COMM_WORLD ) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   permute the linear system and the front tree
   --------------------------------------------
*/
oldToNewIV = ETree_oldToNewVtxPerm(frontETree) ;
ETree_permuteVertices(frontETree, oldToNewIV) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n permuted front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
   fprintf(msgFile, "\n\n old-to-new vertex permutation") ;
   IV_writeForHumanEye(oldToNewIV, msgFile) ;
   fflush(msgFile) ;
}
DLinSystem_permute(linsys, oldToNewIV, msglvl, msgFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n linear system after permutation") ;
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------------------
   generate the owners map IV object
   ---------------------------------
*/
cutoff   = 1. / ( 2. * nproc ) ;
cumopsDV = DV_new() ;
DV_init(cumopsDV, nproc, NULL) ;
ownersIV = ETree_ddMap(frontETree, cumopsDV, cutoff) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n owners map") ;
   IV_writeForHumanEye(ownersIV, msgFile) ;
   fflush(msgFile) ;
}
IV_sizeAndEntries(ownersIV, &nfront, &owners) ;
/*
   ------------------------------
   generate the vertex owners map 
   redistibute the linear system
   ------------------------------
*/
vtxmapIV = IV_new() ;
IV_init(vtxmapIV, neqns, NULL) ;
vtxmap = IV_entries(vtxmapIV) ;
vtxToFront = ETree_vtxToFront(frontETree) ;
for ( v = 0 ; v < neqns ; v++ ) {
   vtxmap[v] = owners[vtxToFront[v]] ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n vertex owners map") ;
   IV_writeForHumanEye(vtxmapIV, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------------------
   redistibute the linear system
   -----------------------------
*/
IVzero(4, stats) ;
firsttag = 1 ;
DLinSystem_MPI_split(linsys, vtxmapIV, firsttag, stats,
                    msglvl, msgFile, MPI_COMM_WORLD) ;
firsttag += 3 ;
fprintf(msgFile,
        "\n\n redistribute linear system"
        "\n send stats : %d messages with %d bytes"
        "\n recv stats : %d messages with %d bytes",
        stats[0], stats[2], stats[1], stats[3]) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n linear system after redistribution") ;
   DLinSystem_writeForHumanEye(linsys, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------------
   get the symbolic factorization
   ------------------------------
*/
DPencil_changeStorageMode(linsys->pencil, 3) ;
IVzero(4, stats) ;
symbfacIVL = SymbFac_MPI_initFromDPencil(frontETree, pencil, ownersIV, 
                     firsttag, stats, msglvl, msgFile, MPI_COMM_WORLD) ;
firsttag += nfront ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n symbolic factorization") ;
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------------------
   compute the numeric factorization
   ---------------------------------
*/
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL, 0, 0, 0, 
               0, myid, ownersIV) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n front matrix initialized") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
DVzero(20, cpus) ;
IVzero(20, stats) ;
MARKTIME(t1) ;
rootchv = DFrontMtx_MPI_factorDPencil(frontmtx, pencil, 1.0, 0.0,
                                  firsttag, ownersIV, myid, cpus, stats,
                                  msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
firsttag += 4*nfront + 2 ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n numeric factorization") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
factorcpu = t2 - t1 ;
fprintf(msgFile, "\n CPU %8.3f : numeric factorization done ",
t2-t1);
fflush(msgFile) ;
misccpu = factorcpu - DVsum(9, cpus) - DVsum(4, cpus + 10) ;
if ( factorcpu > 0.0 ) {
   fprintf(msgFile,
   "\n    initialize fronts              %8.3f %6.2f"
   "\n    load original entries          %8.3f %6.2f"
   "\n    update fronts                  %8.3f %6.2f"
   "\n    insert aggregate data          %8.3f %6.2f"
   "\n    assemble aggregate data        %8.3f %6.2f"
   "\n    assemble postponed data        %8.3f %6.2f"
   "\n    factor fronts                  %8.3f %6.2f"
   "\n    extract postponed data         %8.3f %6.2f"
   "\n    store factor entries           %8.3f %6.2f"
   "\n    post initial receives          %8.3f %6.2f"
   "\n    check for received messages    %8.3f %6.2f"
   "\n    post initial sends             %8.3f %6.2f"
   "\n    check for sent messages        %8.3f %6.2f"
   "\n    miscellaneous                  %8.3f %6.2f"
   "\n    total time                     %8.3f",
   cpus[0], 100.*cpus[0]/factorcpu,
   cpus[1], 100.*cpus[1]/factorcpu,
   cpus[2], 100.*cpus[2]/factorcpu,
   cpus[3], 100.*cpus[3]/factorcpu,
   cpus[4], 100.*cpus[4]/factorcpu,
   cpus[5], 100.*cpus[5]/factorcpu,
   cpus[6], 100.*cpus[6]/factorcpu,
   cpus[7], 100.*cpus[7]/factorcpu, 
   cpus[8], 100.*cpus[8]/factorcpu, 
   cpus[10], 100.*cpus[10]/factorcpu, 
   cpus[11], 100.*cpus[11]/factorcpu, 
   cpus[12], 100.*cpus[12]/factorcpu, 
   cpus[13], 100.*cpus[13]/factorcpu, 
   misccpu, 100.*misccpu/factorcpu,
   factorcpu) ;
}
fprintf(msgFile,
        "\n\n COMMUNICATION STATISTICS"
        "\n aggregate info"
        "\n %8d sends, %10d bytes sent, %8d recv's, %10d bytes recv'd"
        "\n postponed info"
        "\n %8d sends, %10d bytes sent, %8d recv's, %10d bytes recv'd",
        stats[0], stats[1], stats[2], stats[3],
        stats[4], stats[5], stats[6], stats[7]) ;
fprintf(msgFile,
       "\n\n WORKING STORAGE STATISTICS"
       "\n %d active DChv objects, %d active bytes, %d requested bytes",
        stats[8], stats[9], stats[10]) ;
/*
   -----------------------------
   postprocess the factorization
   -----------------------------
*/
MARKTIME(t1) ;
firsttag = DFrontMtx_MPI_postProcess(frontmtx, 0, 1, ownersIV, firsttag,
                               stats, msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set local indices ", t2-t1);
firsttag += 2 ;
fprintf(msgFile, "\n\n after local indices, front matrix") ;
if ( msglvl > 2 ) {
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
} else {
   DFrontMtx_writeStats(frontmtx, msgFile) ;
}
fflush(msgFile) ;
MPI_Barrier(MPI_COMM_WORLD) ;
/*
   --------------------
   compute the solution
   --------------------
*/
manager = DDenseMtxManager_new() ;
MARKTIME(t1) ;
DFrontMtx_MPI_solve(frontmtx, linsys->solmtx, linsys->rhsmtx, 
                    manager, ownersIV, firsttag, stats, cpus, 
                    msglvl, msgFile, MPI_COMM_WORLD) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n\n CPU %8.3f : solve system\n", t2-t1);
if ( msglvl > 0 ) {
   DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
} else {
   DDenseMtx_writeStats(linsys->solmtx, msgFile) ;
}
fflush(msgFile) ;

/*--------------------------------------------------------------------*/
MPI_Finalize() ;

return ; }
/*--------------------------------------------------------------------*/
