/*  basics.c  */

#include "../IIheap.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   create and return a new instance of the IIheap object

   created -- 95sep30, cca
   -----------------------------------------------------
*/
IIheap *
IIheap_new (
   void
) {
IIheap   *heap ;

ALLOCATE(heap, struct _IIheap, 1) ;

IIheap_setDefaultFields(heap) ;

return(heap) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   set the default fields for an IIheap object

   created -- 95sep30, cca
   -------------------------------------------
*/
void
IIheap_setDefaultFields (
   IIheap   *heap
) {
if ( heap == NULL ) {
   fprintf(stderr, "\n fatal error in IIheap_setDefaultFields(%p)"
           "\n heap is NULL\n", heap) ;
   exit(-1) ;
}
heap->size    =   0  ;
heap->maxsize =   0  ;
heap->heapLoc = NULL ;
heap->objIds  = NULL ;
heap->objVals = NULL ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   clear the data fields

   created -- 95sep30, cca
   -----------------------
*/
void
IIheap_clearData (
   IIheap   *heap
) {
if ( heap == NULL ) {
   fprintf(stderr, "\n fatal error in IIheap_clearData(%p)"
           "\n heap is NULL\n", heap) ;
   exit(-1) ;
}
if ( heap->heapLoc != NULL ) {
   IVfree(heap->heapLoc) ;
}
if ( heap->objIds != NULL ) {
   IVfree(heap->objIds) ;
}
if ( heap->objVals != NULL ) {
   IVfree(heap->objVals) ;
}
IIheap_setDefaultFields(heap) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   free the IIheap object

   created -- 95sep30, cca
   -----------------------
*/
void
IIheap_free (
   IIheap   *heap
) {
if ( heap == NULL ) {
   fprintf(stderr, "\n fatal error in IIheap_free(%p)"
           "\n heap is NULL\n", heap) ;
   exit(-1) ;
}
IIheap_clearData(heap) ;
FREE(heap) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------
   initializer, 
   set heap maximum size to maxsize
   and allocate the arrays

   created -- 95sep30, cca
   --------------------------------
*/
void
IIheap_init ( 
   IIheap   *heap,
   int      maxsize 
) {
if ( heap == NULL || maxsize <= 0 ) {
   fprintf(stderr, "\n\n error in IIheap_init(%p,%d)"
           "\n heap is NULL or size = %d is nonpositive\n",
           heap, maxsize, maxsize) ;
   exit(-1) ;
}
IIheap_clearData(heap) ;
heap->maxsize = maxsize ;
heap->heapLoc = IVinit(maxsize, -1) ;
heap->objIds  = IVinit(maxsize, -1) ;
heap->objVals = IVinit(maxsize, -1) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   fill pid with the id and pkey with the key 
   of the minimum (id, key) pair in the heap

   created -- 95sep30, cca
   ----------------------------------------------
*/
void
IIheap_root ( 
   IIheap   *heap,
   int      *pid, 
   int      *pkey 
) {
if ( heap == NULL || pid == NULL || pkey == NULL ) {
   fprintf(stderr, "\n\n error in IIheap_root(%p,%p,%p)"
           "\n heap is NULL or pid is NULL or pkey is NULL\n",
           heap, pid, pkey) ;
   exit(-1) ;
}

if ( heap->size > 0 ) {
   *pid  = heap->objIds[0]  ;
   *pkey = heap->objVals[0] ;
} else {
   *pid = *pkey = -1 ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   insert the (id, key) pair into the heap

   created -- 95sep30, cca
   -----------------------------------------
*/
void
IIheap_insert ( 
   IIheap   *heap,
   int      id, 
   int      key 
) {
int   loc ;

if ( heap == NULL || id < 0 || id >= heap->maxsize ) {
   fprintf(stderr, "\n error in IIheap_insert(%p,%d,%d)"
           "\n heap is NULL or pair (%d,%d) is out of bounds\n",
           heap, id, key, id, key) ;
   exit(-1) ;
}
if ( heap->heapLoc[id] != -1 ) {
   fprintf(stderr, "\n error in IIheap_insert(%p,%d,%d)"
           "\n object (%d,%d) is already in heap\n",
           heap, id, key, id, key) ;
   exit(-1) ;
}
if ( heap->size == heap->maxsize ) {
   fprintf(stderr, "\n error in IIheap_insert(%p,%d,%d)"
           "\n heap size exceeded\n", heap, id, key) ;
   exit(-1) ;
}
/*
   -----------------------------
   insert the object and sift up
   -----------------------------
*/
heap->heapLoc[id]  = loc = heap->size++ ;
heap->objIds[loc]  = id  ;
heap->objVals[loc] = key ;
IIheap_siftUp(heap, loc) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   remove the value id from the heap

   created -- 95sep30, cca
   ---------------------------------
*/
void
IIheap_remove ( 
   IIheap   *heap,
   int      id 
) {
int   last, loc, newVal, oldVal ;
/*
   ---------------
   check the input
   ---------------
*/
if ( heap == NULL || id < 0 || id >= heap->maxsize ) {
   fprintf(stderr, "\n error in IIheap_remove(%p,%d)"
           "\n heap is NULL or object (%d) is out of bounds\n",
           heap, id, id) ;
   exit(-1) ;
}
if ( (loc = heap->heapLoc[id]) == -1 ) {
   fprintf(stderr, "\n error in IIheap_remove(%p,%d)"
           "\n object %d not in heap", heap, id, id) ;
   exit(-1) ;
}
/*
   -----------------------------------
   save the old value at this location
   and update the three heap vectors
   -----------------------------------
*/
if ( loc == (last = --heap->size) ) {
/*
   -----------------------------------------------------------------
   the element occupied the last position in the heap, simple return
   -----------------------------------------------------------------
*/
   heap->heapLoc[id]  = -1 ;
   heap->objIds[loc]  = -1 ;
   heap->objVals[loc] =  0 ;
} else {
/*
   ---------------------------------------------
   move the last element to the current location
   ---------------------------------------------
*/
   heap->heapLoc[id]   = -1 ;
   id            = heap->objIds[last] ;
   heap->heapLoc[id]   = loc ;
   heap->objIds[loc]   = id ;
   heap->objIds[last]  = -1 ;
   oldVal        = heap->objVals[loc] ;
   heap->objVals[loc]  = newVal = heap->objVals[last] ;
   heap->objVals[last] =  0 ;
   if ( oldVal > newVal ) {
      IIheap_siftUp(heap, loc) ;
   } else if ( oldVal < newVal ) {
      IIheap_siftDown(heap, loc) ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   purpose -- to write the IIheap object to a file

   created -- 95sep30, cca
   -----------------------------------------------
*/
void
IIheap_print ( 
   IIheap   *heap,
   FILE     *fp 
) {
int   ierr, j ;

if ( heap == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in IIheap_print(%p,%p)"
           "\n heap is NULL or file pointer is NULL", heap, fp) ;
   exit(-1) ;
}
fprintf(fp, "\n\n IIheap : present size %d, max size %d",
        heap->size, heap->maxsize) ;
if ( heap->size > 0 ) {
   fprintf(fp, "\n contents of heap : (location id value)") ;
   for ( j = 0 ; j < heap->size ; j++ ) {
      fprintf(fp, "\n %8d %8d %8d", j, 
              heap->objIds[j], heap->objVals[j]) ;
   }
   fprintf(fp, "\n locations of ids") ;
   IVfp80(fp, heap->maxsize, heap->heapLoc, 80, &ierr) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   return the number of bytes taken by the object

   created -- 95sep30, cca
   ----------------------------------------------
*/
int
IIheap_sizeOf ( 
   IIheap   *heap
) {
return(sizeof(IIheap) + 3*heap->maxsize*sizeof(int)) ; }

/*====================================================================*/
/*
   -----------------------------------------------
   purpose -- to sift a heap element down the tree

   created -- 95sep30, cca
   -----------------------------------------------
*/
void
IIheap_siftDown ( 
   IIheap   *heap,
   int      loc 
) {
int   desc, itemp, left, right, size ;
int   *heapLoc, *objIds, *objVals ;
/*
   ---------------
   check the input
   ---------------
*/
if ( heap == NULL || loc < 0 || loc >= heap->size ) {
   fprintf(stderr, "\n fatal error in IIheap_siftDown(%p,%d)"
           "\n heap is NULL or loc = %d out of range\n",
           heap, loc, loc) ;
   exit(-1) ;
}
size    = heap->size    ;
heapLoc = heap->heapLoc ;
objIds  = heap->objIds  ;
objVals = heap->objVals ;

while ( 1 ) {
   left  = 2 * loc + 1 ;
   right = left + 1 ;
   if ( left >= size ) {
      break ;
   } else if ( right >= size ) {
      desc = left ;
   } else {
      if ( objVals[left] <= objVals[right] ) {
         desc = left ;
      } else {
         desc = right ;
      }
   }
/*
   if ( objVals[desc] < objVals[loc] ) {
*/
   if ( objVals[desc] <= objVals[loc] ) {
      itemp         = objVals[desc] ;
      objVals[desc] = objVals[loc]  ;
      objVals[loc]  = itemp         ;
      itemp         = objIds[desc]  ;
      objIds[desc]  = objIds[loc]   ;
      objIds[loc]   = itemp         ;
      heapLoc[objIds[loc]]  = loc   ;
      heapLoc[objIds[desc]] = desc  ;
      loc = desc ;
   } else {
      break ;
   }
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   purpose -- to sift a heap element up the tree

   created -- 95sep30, cca
   ---------------------------------------------
*/
void
IIheap_siftUp ( 
   IIheap   *heap,
   int      loc 
) {
int   itemp, par ;
int   *heapLoc, *objIds, *objVals ;
/*
   ---------------
   check the input
   ---------------
*/
if ( heap == NULL || loc < 0 || loc >= heap->size ) {
   fprintf(stderr, "\n fatal error in IIheap_siftUp(%p,%d)"
           "\n heap is NULL or loc = %d out of range\n",
           heap, loc, loc) ;
   exit(-1) ;
}
heapLoc = heap->heapLoc ;
objIds  = heap->objIds  ;
objVals = heap->objVals ;

while ( loc != 0 ) {
   par = (loc - 1)/2 ;
/*
   if ( objVals[par] > objVals[loc] ) {
*/
   if ( objVals[par] >= objVals[loc] ) {
      itemp        = objVals[par] ;
      objVals[par] = objVals[loc] ;
      objVals[loc] = itemp        ;
      itemp        = objIds[par]  ;
      objIds[par]  = objIds[loc]  ;
      objIds[loc]  = itemp        ;
      heapLoc[objIds[loc]] = loc  ;
      heapLoc[objIds[par]] = par  ;
      loc = par ;
   } else {
      break ;
   }
}

return ; }

/*--------------------------------------------------------------------*/
