/*  schurComplement.c  */

#include "../Graph.h"
#include "../../DInpMtx.h"

#define MYDEBUG 0

#define NTRIPLES 1000

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose --  given a component ids IV object, 
               create and return a Schur complement matrix

   compidsIV -- if compids[v] = 0 then v is a schur complement vertex
      note: on return compids[v] contains the component id for v,
            and may be different on input
   mapIV -- on return, map[v] = -1 --> v is a domain vertex,
      map[v] >= 0 --> v is the map[v] schur complement vertex

   created -- 97apr10, cca
   ------------------------------------------------------------------
*/
Graph *
Graph_makeSchurComplement (
   Graph   *graph,
   IV      *compidsIV,
   IV      *mapIV
) {
DInpMtx   *schurmtx ;
double    *vent ;
Graph     *schurgraph ;
int       count, icomp, ii, last, ncomp, nedges, now, nschur, ntriples,
          nvtx, nvtot, totvwght, u, v, vschur, vsize, w ;
int       *colids, *compids, *head, *link, *list, *map, *mark, 
          *rowids, *schurvwghts, *vadj, *vwghts ;
IVL       *adjIVL ;
/*
   ---------------
   check the input
   ---------------
*/
if ( graph == NULL || compidsIV == NULL || mapIV == NULL ) {
   fprintf(stderr, 
           "\n fatal error in Graph_makeSchurComplement(%p,%p,%p)"
           "\n bad input\n", graph, compidsIV, mapIV) ;
   exit(-1) ;
}
nvtx  = graph->nvtx ;
nvtot = nvtx + graph->nvbnd ;
if ( nvtx != IV_size(compidsIV) ) {
   fprintf(stderr, "\n fatal error in Graph_makeSchurComplement(%p,%p)"
           "\n nvtx = %d, IV_size(compidsIV) = %d\n", 
           graph, compidsIV, nvtx, IV_size(compidsIV)) ;
   exit(-1) ;
}
compids = IV_entries(compidsIV) ;
IV_setSize(mapIV, nvtx) ;
map  = IV_entries(mapIV) ;
list = IVinit(nvtx, -1) ;
/*
   ------------------------------------------
   get component ids : 0 --> schur complement
   ------------------------------------------
*/
nschur = 0 ;
for ( v = 0 ; v < nvtx ; v++ ) {
   if ( compids[v] != 0 ) {
      compids[v] = -1 ;
   } else {
      map[v] = nschur++ ;
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n %d schur complement vertices", nschur) ;
fflush(stdout) ;
#endif
for ( u = 0, ncomp = 0 ; u < nvtx ; u++ ) {
   if ( compids[u] == -1 ) {
      ncomp++ ;
      now = last = 0 ;
      list[0] = u ;
      while ( now <= last ) {
         v = list[now++] ;
         Graph_adjAndSize(graph, v, &vsize, &vadj) ;
         for ( ii = 0 ; ii < vsize ; ii++ ) {
            w = vadj[ii] ;
            if ( w < 0 || w >= nvtot ) {
               fprintf(stderr, 
                    "\n fatal error in Graph_makeSchurComplement(%p,%p)"
                    "\n v = %d, w = %d\n", graph, compidsIV, v, w) ;
               exit(-1) ;
            }
            if ( w < nvtx && compids[w] == -1 ) {
               compids[w] = ncomp ;
               list[++last] = w ;
            }
         }
      }
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n %d components", ncomp) ;
fflush(stdout) ;
#endif
/*
   --------------------------------------------------------------
   initialize a DInpMtx object to form the Schur complement graph
   --------------------------------------------------------------
*/
schurmtx = DInpMtx_new() ;
DInpMtx_init(schurmtx, 1, 1, nschur, 0) ;
/*
   ----------------------------------------------
   get the head/link structure for the components
   ----------------------------------------------
*/
head = IVinit(ncomp+1, -1) ;
link = IVinit(nvtx,    -1) ;
for ( v = 0 ; v < nvtx ; v++ ) {
   icomp = compids[v] ;
   link[v] = head[icomp] ;
   head[icomp] = v ;
}
/*
   ---------------------------------------
   get the boundary of each component and 
   load the clique into the DInpMtx object
   ---------------------------------------
*/
mark = IVinit(nvtx, -1) ;
for ( icomp = 1 ; icomp <= ncomp ; icomp++ ) {
   count = 0 ;
   for ( v = head[icomp] ; v != -1 ; v = link[v] ) {
      Graph_adjAndSize(graph, v, &vsize, &vadj) ;
      for ( ii = 0 ; ii < vsize ; ii++ ) {
         if (  (w = vadj[ii]) < nvtx 
            && compids[w] == 0 
            && mark[w] != icomp ) {
            mark[w] = icomp ;
            list[count++] = map[w] ;
         }
      }
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n component %d has %d boundary vertices", 
           icomp, count) ;
   fflush(stdout) ;
#endif
   DInpMtx_inputMatrix(schurmtx, count, count, 1, count, 
                       list, list, NULL) ;
}
/*
   ------------------------------------
   add the edges between schur vertices
   ------------------------------------
*/
rowids   = IVinit(NTRIPLES, -1) ;
colids   = IVinit(NTRIPLES, -1) ;
ntriples = 0 ;
for ( v = head[0] ; v != -1 ; v = link[v] ) {
   Graph_adjAndSize(graph, v, &vsize, &vadj) ;
#if MYDEBUG > 1
   fprintf(stdout, "\n schur vertex %d, vsize = %d", v, vsize) ;
   fflush(stdout) ;
#endif
   for ( ii = 0 ; ii < vsize ; ii++ ) {
      if (  (w = vadj[ii]) < nvtx 
         && compids[w] == 0 ) {
#if MYDEBUG > 1
         fprintf(stdout, "\n    adding entry (%d,%d)", map[v], map[w]) ;
         fflush(stdout) ;
#endif
         rowids[ntriples]  = map[v] ;
         colids[ntriples]  = map[w] ;
         ntriples++ ;
         if ( ntriples == NTRIPLES ) {
            DInpMtx_inputTriples(schurmtx, ntriples, 
                                 rowids, colids, NULL) ;
            ntriples = 0 ;
         }
      }
   }
}
if ( ntriples > 0 ) {
   DInpMtx_inputTriples(schurmtx, ntriples, rowids, colids, NULL) ;
   ntriples = 0 ;
}
IVfree(rowids) ;
IVfree(colids) ;
/*
   -----------------------------
   sort and compress the entries
   -----------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, 
        "\n before sort and compress, %d entries in schur matrix",
        DInpMtx_nent(schurmtx)) ;
fflush(stdout) ;
#endif
DInpMtx_sortAndCompress(schurmtx) ;
#if MYDEBUG > 0
fprintf(stdout, 
        "\n after sort and compress, %d entries in schur matrix",
        DInpMtx_nent(schurmtx)) ;
fflush(stdout) ;
#endif
DInpMtx_convertToVectors(schurmtx) ;
/*
   -------------------------------------
   initialize the Schur complement graph
   -------------------------------------
*/
schurgraph = Graph_new() ;
Graph_init1(schurgraph, graph->type, nschur, graph->nvbnd, 0,
            IVL_CHUNKED, IVL_CHUNKED) ;
adjIVL = schurgraph->adjIVL ;
for ( v = 0, nedges = 0 ; v < nvtx ; v++ ) {
   if ( compids[v] == 0 ) {
      vschur = map[v] ;
      DInpMtx_vector(schurmtx, vschur, &vsize, &vadj, &vent) ;
      IVqsortUp(vsize, vadj) ;
      IVL_setList(adjIVL, vschur, vsize, vadj) ;
      nedges += vsize ;
   }
}
totvwght = 0 ;
if ( (vwghts = graph->vwghts) != NULL ) {
   schurvwghts = schurgraph->vwghts ;
   for ( v = 0 ; v < nvtx ; v++ ) {
      if ( compids[v] == 0 ) {
         vschur = map[v] ;
         schurvwghts[vschur] = vwghts[v] ;
         totvwght += vwghts[v] ;
      }
   }
} else {
   totvwght = schurgraph->nvtx ;
}
schurgraph->nedges   = nedges   ;
schurgraph->totvwght = totvwght ;
/*
   ---------------------------------
   check that the graph is symmetric
   ---------------------------------
*/
if ( 1 != Graph_isSymmetric(schurgraph) ) {
   fprintf(stderr, "\n fatal error in Graph_schurComplement(%p,%p,%p)"
           "\n schur complement graph is not symmetric\n",
           graph, compidsIV, mapIV) ;
   exit(-1) ;
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
DInpMtx_free(schurmtx) ;
IVfree(head) ;
IVfree(link) ;
IVfree(list) ;
IVfree(mark) ;

return(schurgraph) ; }

/*--------------------------------------------------------------------*/
