/*  testSchurComplement.c  */

#include "../Graph.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------
   read in a Graph and a components id IV object,
   create a Schur complement graph and map IV object

   created -- 97apr10, cca
   -------------------------------------------------
*/
{
char     *inCompidsIVfileName, *inGraphFileName, 
         *outGraphFileName, *outMapIVfileName ;
double   t1, t2 ;
Graph    *graph, *schurgraph ;
int      msglvl, rc ;
IV       *cmapIV, *compidsIV, *mapIV ;
FILE     *msgFile ;

if ( argc != 7 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile inGraphFile inCompIdsFile"
      "\n         outGraphFile outMapFile"
      "\n    msglvl        -- message level"
      "\n    msgFile       -- message file"
      "\n    inGraphFile   -- input file, must be *.graphf or *.graphb"
      "\n    inCompidsFile -- output file, must be *.ivf or *.ivb"
      "\n    outGraphFile  -- output file, must be *.graphf or *.graphb"
      "\n    outMapFile    -- output file, must be *.ivf or *.ivb"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inGraphFileName     = argv[3] ;
inCompidsIVfileName = argv[4] ;
outGraphFileName    = argv[5] ;
outMapIVfileName    = argv[6] ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl        -- %d" 
        "\n msgFile       -- %s" 
        "\n inGraphFile   -- %s" 
        "\n inCompidsFile -- %s" 
        "\n outGraphFile  -- %s" 
        "\n outMapFile    -- %s" 
        "\n",
        argv[0], msglvl, argv[2], inGraphFileName, inCompidsIVfileName,
        outGraphFileName, outMapIVfileName) ;
fflush(msgFile) ;
/*
   ------------------------
   read in the Graph object
   ------------------------
*/
if ( strcmp(inGraphFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
graph = Graph_new() ;
MARKTIME(t1) ;
rc = Graph_readFromFile(graph, inGraphFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
        t2 - t1, inGraphFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from Graph_readFromFile(%p,%s)",
           rc, graph, inGraphFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading Graph object from file %s",
        inGraphFileName) ;
if ( msglvl > 2 ) {
   Graph_writeForHumanEye(graph, msgFile) ;
} else {
   Graph_writeStats(graph, msgFile) ;
}
fflush(msgFile) ;
/*
   ---------------------
   read in the IV object
   ---------------------
*/
if ( strcmp(inCompidsIVfileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
compidsIV = IV_new() ;
MARKTIME(t1) ;
rc = IV_readFromFile(compidsIV, inCompidsIVfileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in compidsIV from file %s",
        t2 - t1, inCompidsIVfileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from IV_readFromFile(%p,%s)",
           rc, compidsIV, inCompidsIVfileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading IV object from file %s",
        inCompidsIVfileName) ;
if ( msglvl > 2 ) {
   IV_writeForHumanEye(compidsIV, msgFile) ;
} else {
   IV_writeStats(compidsIV, msgFile) ;
}
fflush(msgFile) ;
/*
   ------------------------------
   get the schur complement graph
   ------------------------------
*/
mapIV = IV_new() ;
schurgraph = Graph_makeSchurComplement(graph, compidsIV, mapIV) ;
fprintf(msgFile, "\n\n Schur complement graph") ;
if ( msglvl > 2 ) {
   Graph_writeForHumanEye(schurgraph, msgFile) ;
} else {
   Graph_writeStats(schurgraph, msgFile) ;
}
fflush(msgFile) ;
fprintf(msgFile, "\n\n vertex map IV object") ;
if ( msglvl > 2 ) {
   IV_writeForHumanEye(mapIV, msgFile) ;
} else {
   IV_writeStats(mapIV, msgFile) ;
}
fflush(msgFile) ;
/*
   -----------------------
   get the equivalence map
   -----------------------
*/
cmapIV = Graph_equivMap(schurgraph) ;
fprintf(msgFile, "\n\n cmap IV object") ;
if ( msglvl > 2 ) {
   IV_writeForHumanEye(cmapIV, msgFile) ;
} else {
   IV_writeStats(cmapIV, msgFile) ;
}
fflush(msgFile) ;
if ( 1 + IV_max(cmapIV) < schurgraph->nvtx ) {
   Graph   *compressedgraph ;
   int     nvtx, v ;
   int     *cmap, *map ;
/*
   -------------------------------------------
   get the compressed graph and update the map
   -------------------------------------------
*/
   compressedgraph = Graph_compress2(schurgraph, cmapIV, 1) ;
   nvtx = IV_size(mapIV) ;
   map  = IV_entries(mapIV) ;
   cmap = IV_entries(cmapIV) ;
   for ( v = 0 ; v < nvtx ; v++ ) {
      if ( map[v] != -1 ) {
         map[v] = cmap[map[v]] ;
      }
   }
   Graph_free(schurgraph) ;
   schurgraph = compressedgraph ;
   IV_free(cmapIV) ;
   fprintf(msgFile, "\n\n compressed Schur complement graph") ;
   if ( msglvl > 2 ) {
      Graph_writeForHumanEye(schurgraph, msgFile) ;
   } else {
      Graph_writeStats(schurgraph, msgFile) ;
   }
   fflush(msgFile) ;
   fprintf(msgFile, "\n\n compressed vertex map IV object") ;
   if ( msglvl > 2 ) {
      IV_writeForHumanEye(mapIV, msgFile) ;
   } else {
      IV_writeStats(mapIV, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   --------------------------
   write out the Graph object
   --------------------------
*/
if ( strcmp(outGraphFileName, "none") != 0 ) {
   MARKTIME(t1) ;
   rc = Graph_writeToFile(schurgraph, outGraphFileName) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : write schurgraph to file %s",
           t2 - t1, outGraphFileName) ;
   if ( rc != 1 ) {
      fprintf(msgFile, 
              "\n return value %d from Graph_writeToFile(%p,%s)",
              rc, schurgraph, outGraphFileName) ;
   }
}
/*
   ---------------------------
   write out the map IV object
   ---------------------------
*/
if ( strcmp(outMapIVfileName, "none") != 0 ) {
   MARKTIME(t1) ;
   rc = IV_writeToFile(mapIV, outMapIVfileName) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : write mapIV to file %s",
           t2 - t1, outMapIVfileName) ;
   if ( rc != 1 ) {
      fprintf(msgFile, 
              "\n return value %d from IV_writeToFile(%p,%s)",
              rc, mapIV, outMapIVfileName) ;
   }
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
Graph_free(graph) ;
Graph_free(schurgraph) ;
IV_free(compidsIV) ;
IV_free(mapIV) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
