/*  util.c  */

#include "../ETree.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   return the number of bytes taken by the object

   created -- 95nov15, cca
   ----------------------------------------------
*/
int
ETree_sizeOf (
   ETree   *etree
) {
int   nbytes ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL ) {
   fprintf(stderr, "\n fatal error in ETree_sizeOf(%p)"
           "\n bad input\n", etree) ;
   exit(-1) ;
}
nbytes = sizeof(struct _ETree) ;
if ( etree->tree != NULL ) {
   nbytes += Tree_sizeOf(etree->tree) ;
}
if ( etree->nodwghtsIV != NULL ) {
   nbytes += IV_sizeOf(etree->nodwghtsIV) ;
}
if ( etree->nodwghtsIV != NULL ) {
   nbytes += IV_sizeOf(etree->bndwghtsIV) ;
}
if ( etree->vtxToFrontIV != NULL ) {
   nbytes += IV_sizeOf(etree->vtxToFrontIV) ;
}
return(nbytes) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   return the number of factor indices

   created  -- 95nov15, cca
   modified -- 96jan11, cca
   ----------------------------------------
*/
int
ETree_nFactorIndices (
   ETree   *etree
) {
int   nb, nfront, nv, nind, nvtx, v ;
int   *bndwghts, *nodwghts ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0 ) {
   fprintf(stderr, "\n fatal error in ETree_nFactorIndices(%p)"
           "\n bad input\n", etree) ;
   exit(-1) ;
}
nodwghts = IV_entries(etree->nodwghtsIV) ;
bndwghts = IV_entries(etree->bndwghtsIV) ;
nind = 0 ;
for ( v = 0 ; v < nfront ; v++ ) {
   nv = nodwghts[v] ;
   nb = bndwghts[v] ;
   nind += nv + nb ;
}
return(nind) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   return the number of factor entries

   flag == 1 --> LDL^T
   flag == 2 --> LU

   created  -- 95nov15, cca
   ----------------------------------------
*/
int
ETree_nFactorEntries (
   ETree   *etree,
   int     flag
) {
int   nb, nfront, nv, nvtx, nzf, v ;
int   *bndwghts, *nodwghts ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0 
   || flag < 1 || flag > 2 ) {
   fprintf(stderr, "\n fatal error in ETree_nFactorEntries(%p,%d)"
           "\n bad input\n", etree, flag) ;
   exit(-1) ;
}
nodwghts = IV_entries(etree->nodwghtsIV) ;
bndwghts = IV_entries(etree->bndwghtsIV) ;
nzf = 0 ;
if ( flag == 1 ) {
   for ( v = 0 ; v < nfront ; v++ ) {
      nv = nodwghts[v] ;
      nb = bndwghts[v] ;
      nzf += (nv*(nv+1))/2 + nv*nb ;
   }
} else if ( flag == 2 ) {
   for ( v = 0 ; v < nfront ; v++ ) {
      nv = nodwghts[v] ;
      nb = bndwghts[v] ;
      nzf += nv*nv + 2*nv*nb ;
   }
}
return(nzf) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   return the number of factor operations

   flag == 1 --> LDL^T
   flag == 2 --> LU

   created -- 95nov15, cca
   ----------------------------------------
*/
double
ETree_nFactorOps (
   ETree   *etree,
   int     flag
) {
double   nb, nv, ops ;
int      nfront, nvtx, v ;
int      *bndwghts, *nodwghts ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0 
   || flag < 1 || flag > 2 ) {
   fprintf(stderr, "\n fatal error in ETree_nFactorOps(%p,%d)"
           "\n bad input\n", etree, flag) ;
   exit(-1) ;
}
nodwghts = IV_entries(etree->nodwghtsIV) ;
bndwghts = IV_entries(etree->bndwghtsIV) ;
ops = 0 ;
if ( flag == 1 ) {
   for ( v = 0 ; v < nfront ; v++ ) {
      nv = nodwghts[v] ;
      nb = bndwghts[v] ;
      ops += nv*nb*(nv + nb + 1) + (nv*(nv+1)*(2*nv+1))/6 ;
   }
} else if ( flag == 2 ) {
   for ( v = 0 ; v < nfront ; v++ ) {
      nv = nodwghts[v] ;
      nb = bndwghts[v] ;
/*
      ops += ((nv-1)*nv)/2 + ((nv-1)*nv*(2*nv-1))/3
           + nv*nb + 2*nv*nb*nb ;
      ops += ((nv+1)*nv)/2 + ((nv-1)*nv*(2*nv-1))/3
           + nv*nb + 2*(nv-1)*nv*nb + 2*nv*nb*nb ;
*/
      ops += ETree_nExternalLUops(etree, v)
          +  ETree_nInternalLUops(etree, v) ;
   }
}
return(ops) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   return the number of factor operations

   ops[0] -- # of operations in the (1,1) block
   ops[1] -- # of operations in the (2,1) block
   ops[2] -- # of operations in the (1,2) block
   ops[3] -- # of operations in the (2,2) block

   created -- 96dec01, cca
   -----------------------------------------------
*/
void
ETree_nFactorOps2 (
   ETree   *etree,
   double   ops[]
) {
double   b, m ;
int      nfront, nvtx, v ;
int      *bndwghts, *nodwghts ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0 ) {
   fprintf(stderr, "\n fatal error in ETree_nFactorOps(%p,%p)"
           "\n bad input\n", etree, ops) ;
   exit(-1) ;
}
ops[0] = ops[1] = ops[2] = ops[3] = 0.0 ;
nodwghts = IV_entries(etree->nodwghtsIV) ;
bndwghts = IV_entries(etree->bndwghtsIV) ;
for ( v = 0 ; v < nfront ; v++ ) {
   b = nodwghts[v] ;
   m = bndwghts[v] ;
   ops[0] += b*(b+1)/2. + (b-1)*b*(2*b-1)/3 ;
   ops[1] += b*(b-1)*m + m*b ;
   ops[2] += b*(b-1)*m ;
   ops[3] += 2*b*m*m ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   return the number of entries an LU front

   created -- 96dec04, cca
   ----------------------------------------
*/
double
ETree_nLUentries (
   ETree   *etree,
   int     J 
) {
int   b, m, nent ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || etree->nfront <= 0
   || J < 0 || J >= etree->nfront ) {
   fprintf(stderr, "\n fatal error in ETree_nLUentries(%p,%d)"
           "\n bad input\n", etree, J) ;
   exit(-1) ;
}
b = IV_entry(etree->nodwghtsIV, J) ;
m = IV_entry(etree->bndwghtsIV, J) ;
nent = b*(b + 2*m) ;

return(nent) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   return the number of internal LU operations for a front

   created -- 96dec04, cca
   -------------------------------------------------------
*/
double
ETree_nInternalLUops (
   ETree   *etree,
   int     J 
) {
double   b, m, ops ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || etree->nfront <= 0
   || J < 0 || J >= etree->nfront ) {
   fprintf(stderr, "\n fatal error in ETree_nInternalLUops(%p,%d)"
           "\n bad input\n", etree, J) ;
   exit(-1) ;
}
b = IV_entry(etree->nodwghtsIV, J) ;
m = IV_entry(etree->bndwghtsIV, J) ;
ops = b*(b+1)/2. + (b-1)*b*(2*b-1)/3 + 2*b*(b-1)*m + m*b ;

return(ops) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   return the number of external LU operations for a front

   created -- 96dec04, cca
   -------------------------------------------------------
*/
double
ETree_nExternalLUops (
   ETree   *etree,
   int     J 
) {
double   b, m, ops ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL 
   || etree->nfront <= 0
   || J < 0 || J >= etree->nfront ) {
   fprintf(stderr, "\n fatal error in ETree_nExternalLUops(%p,%d)"
           "\n bad input\n", etree, J) ;
   exit(-1) ;
}

b = IV_entry(etree->nodwghtsIV, J) ;
m = IV_entry(etree->bndwghtsIV, J) ;
ops = 2*b*m*m ;

return(ops) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   return a DV object that contains the number of operations 
   for each front using a backward looking algorithm

   created -- 96dec04, cca
   ---------------------------------------------------------
*/
DV *
ETree_backwardOps (
   ETree   *etree,
   int     *vwghts,
   IVL     *symbfacIVL
) {
double   extops ;
double   *ops ;
DV       *opsDV ;
int      bndwght, ii, J, K, nfront, size, v, vwght, wghtJ ;
int      *bndwghts, *indices, *nodwghts, *vtxToFront ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL || symbfacIVL == NULL ) {
   fprintf(stderr, "\n fatal error in ETree_backwardOps(%p,%p,%p)"
           "\n bad input\n", etree, vwghts, symbfacIVL) ;
   exit(-1) ;
}
nfront     = etree->nfront ;
nodwghts   = IV_entries(etree->nodwghtsIV) ;
bndwghts   = IV_entries(etree->bndwghtsIV) ;
vtxToFront = IV_entries(etree->vtxToFrontIV) ;
opsDV = DV_new() ;
DV_init(opsDV, nfront, NULL) ;
ops = DV_entries(opsDV) ;
DV_fill(opsDV, 0.0) ;
extops = 0.0 ;
for ( J = 0 ; J < nfront ; J++ ) {
   ops[J] += ETree_nInternalLUops(etree, J) ;
   wghtJ   = nodwghts[J] ;
   bndwght = bndwghts[J] ;
#if MYDEBUG > 0
   fprintf(stdout, 
"\n front %d, wght = %d, bnd = %d, %.0f internal ops, %.0f external ops",
           J, wghtJ, bndwght, ETree_nInternalLUops(etree, J),
           ETree_nExternalLUops(etree, J)) ;
#endif
   IVL_listAndSize(symbfacIVL, J, &size, &indices) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    size = %d", size) ;
#endif
   for ( ii = 0 ; ii < size ; ii++ ) {
       v = indices[ii] ;
       if ( vtxToFront[v] != J ) {
          break ;
       }
   }
   for (    ; ii < size ; ii++ ) {
      v = indices[ii] ;
      K = vtxToFront[v] ;
      vwght = (vwghts == NULL) ? 1 : vwghts[v] ;
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n    ii = %d, adding %d ops to front %d",
              ii, 2*vwght*wghtJ*(2*bndwght - vwght), K) ;
#endif
      ops[K] += 2*vwght*wghtJ*(2*bndwght - vwght) ;
      extops += 2*vwght*wghtJ*(2*bndwght - vwght) ;
      bndwght -= vwght ;
   }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n    %.0f external ops", extops) ;
#endif
/*
   if ( extops != ETree_nExternalLUops(etree, J)) {
      fprintf(stdout, "\n error, extops = %.0f != %.0f",
              extops, ETree_nExternalLUops(etree, J)) ;
      exit(-1) ;
   }
*/
}
return(opsDV) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   return a DV object that contains the number of operations 
   for each front using a forward-looking algorithm

   created -- 96dec04, cca
   ---------------------------------------------------------
*/
DV *
ETree_forwardOps (
   ETree   *etree
) {
double   *ops ;
DV       *opsDV ;
int      J, nfront ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL ) {
   fprintf(stderr, "\n fatal error in ETree_forwardOps(%p)"
           "\n bad input\n", etree) ;
   exit(-1) ;
}
nfront     = etree->nfront ;
opsDV = DV_new() ;
DV_init(opsDV, nfront, NULL) ;
ops = DV_entries(opsDV) ;
DV_fill(opsDV, 0.0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   ops[J] += ETree_nInternalLUops(etree, J)
          +  ETree_nExternalLUops(etree, J) ;
}
return(opsDV) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   given an IV object that maps uncompressed vertices to vertices,
   create and return an ETree object that is relative to the 
   uncompressed graph.

   created -- 97feb13, cca
   ---------------------------------------------------------------
*/
ETree *
ETree_expand (
   ETree   *etree,
   IV      *eqmapIV
) {
ETree   *etree2 ;
int     ii, ndof, nfront ;
int     *map, *vtxToFront, *vtxToFront2 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL || eqmapIV == NULL ) {
   fprintf(stderr, "\n fatal error in ETree_expand(%p,%p)"
           "\n bad input\n", etree, eqmapIV) ;
   exit(-1) ;
}
nfront = etree->nfront ;
IV_sizeAndEntries(eqmapIV, &ndof, &map) ;
/*
   ---------------------------
   create the new ETree object
   ---------------------------
*/
etree2 = ETree_new() ;
ETree_init1(etree2, nfront, ndof) ;
IV_copy(etree2->nodwghtsIV, etree->nodwghtsIV) ;
IV_copy(etree2->bndwghtsIV, etree->bndwghtsIV) ;
etree2->tree->root = etree->tree->root ;
IVcopy(nfront, etree2->tree->par, etree->tree->par) ;
IVcopy(nfront, etree2->tree->fch, etree->tree->fch) ;
IVcopy(nfront, etree2->tree->sib, etree->tree->sib) ;
vtxToFront  = IV_entries(etree->vtxToFrontIV) ;
vtxToFront2 = IV_entries(etree2->vtxToFrontIV) ;
for ( ii = 0 ; ii < ndof ; ii++ ) {
   vtxToFront2[ii] = vtxToFront[map[ii]] ;
}

return(etree2) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   this method is used to splice together two front trees
   when the domain vertices and schur complement vertices
   have been ordered separately.

   etree0 -- the lower front tree is for vertices in the domains.
   graph0 -- graph for all the vertices
   mapIV  -- IV object that maps vertices to schur complement
             vertices, if IV_entry(mapIV, v) < 0 then v is 
             a domain vertex.
   etree1 -- the upper front tree is for vertices in the schur 
             complement.

   created -- 97feb01, cca
   --------------------------------------------------------------
*/
ETree *
ETree_spliceTwoETrees (
   ETree   *etree0,
   Graph   *graph0,
   IV      *mapIV,
   ETree   *etree1
) {
ETree   *etree2 ;
int     *bndwghts0, *bndwghts1, *bndwghts2, *fch0, *head0, *link0, 
        *map, *mark, *nodwghts0, *nodwghts1, *nodwghts2, *par0, 
        *par1, *par2, *sib0, *vadj, *vtxToFront0, *vtxToFront1, 
        *vtxToFront2 ;
int     ii, J, K, nfront0, nfront1, nfront2, nvtx, phi, v, vsize, w ;
/*
   ---------------
   check the input
   ---------------
*/
if (  etree0 == NULL || graph0 == NULL 
   || mapIV == NULL || etree1 == NULL ) {
   fprintf(stderr, 
           "\n fatal error in ETree_spliceTwoETrees(%p,%p,%p,%p)"
           "\n bad input\n",
           etree0, graph0, mapIV, etree1) ;
   exit(-1) ;
}
nfront0     = etree0->nfront ;
nvtx        = etree0->nvtx    ;
par0        = etree0->tree->par ;
fch0        = etree0->tree->fch ;
sib0        = etree0->tree->sib ;
nodwghts0   = IV_entries(etree0->nodwghtsIV) ;
bndwghts0   = IV_entries(etree0->bndwghtsIV) ;
vtxToFront0 = IV_entries(etree0->vtxToFrontIV) ;
nfront1     = etree1->nfront ;
par1        = etree1->tree->par ;
bndwghts1   = IV_entries(etree1->bndwghtsIV) ;
nodwghts1   = IV_entries(etree1->nodwghtsIV) ;
vtxToFront1 = IV_entries(etree1->vtxToFrontIV) ;
map         = IV_entries(mapIV) ;
/*
   -------------------------
   create the new front tree
   -------------------------
*/
nfront2 = nfront0 + nfront1 ;
etree2 = ETree_new() ;
ETree_init1(etree2, nfront2, etree0->nvtx) ;
par2        = etree2->tree->par ;
nodwghts2   = IV_entries(etree2->nodwghtsIV) ;
bndwghts2   = IV_entries(etree2->bndwghtsIV) ;
vtxToFront2 = IV_entries(etree2->vtxToFrontIV) ;
/*
   --------------------------------------------------
   fill the parent fields for fronts in the same tree
   --------------------------------------------------
*/
for ( J = 0 ; J < nfront0 ; J++ ) {
   par2[J]      = par0[J] ;
   nodwghts2[J] = nodwghts0[J] ;
   bndwghts2[J] = bndwghts0[J] ;
}
for ( J = 0 ; J < nfront1 ; J++ ) {
   par2[J+nfront0]      = nfront0 + par1[J] ;
   nodwghts2[J+nfront0] = nodwghts1[J] ;
   bndwghts2[J+nfront0] = bndwghts1[J] ;
}
/*
   ---------------------------
   set the vertex to front map
   ---------------------------
*/
for ( v = 0 ; v < nvtx ; v++ ) {
   if ( (J = vtxToFront0[v]) >= 0 ) {
      vtxToFront2[v] = J ;
   } else {
      vtxToFront2[v] = vtxToFront1[map[v]] + nfront0 ;
   }
}
/*
   ---------------------------------------------
   link the vertices to fronts in the lower tree
   ---------------------------------------------
*/
head0 = IVinit(nfront0, -1) ;
link0 = IVinit(nvtx,    -1) ;
for ( v = 0 ; v < nvtx ; v++ ) {
   if ( (J = vtxToFront0[v]) >= 0 ) {
      link0[v] = head0[J] ;
      head0[J] = v ;
   }
}
/*
   -------------------------------------------------------
   link roots of the lower tree to nodes in the upper tree
   -------------------------------------------------------
*/
mark = IVinit(nvtx, -1) ;
for ( J = etree0->tree->root ; J != -1 ; J = sib0[J] ) {
/*
   ---------------------------------------
   K is the parent front in the upper tree
   ---------------------------------------
*/
   K = nfront1 ;
/*
   ---------------------------------------
   loop over vertices in the lower front J
   ---------------------------------------
*/
   for ( v = head0[J] ; v != -1 ; v = link0[v] ) {
      Graph_adjAndSize(graph0, v, &vsize, &vadj) ;
      for ( ii = 0 ; ii < vsize ; ii++ ) {
         w = vadj[ii] ;
         if ( vtxToFront0[w] < 0 ) {
            phi = map[w] ;
/*
            ---------------------------------------------------
            w is a vertex that belongs to phi in the upper tree
            ---------------------------------------------------
*/
            if ( mark[phi] != J ) {
               mark[phi] = J ;
               if ( K > vtxToFront1[phi] ) {
/*
                  ------------------------------------
                  least numbered adjacent front so far
                  ------------------------------------
*/
                  K = vtxToFront1[phi] ;
               }
            }
         }
      }
   }
   if ( K < nfront1 ) {
   /*
      --------------------
      set the parent field
      --------------------
   */
      par2[J] = nfront0 + K ;
   }
}
/*
   -----------------------------
   set the remaining tree fields
   -----------------------------
*/
Tree_setFchSibRoot(etree2->tree) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(head0) ;
IVfree(link0) ;
IVfree(mark)  ;

return(etree2) ; }

/*--------------------------------------------------------------------*/
