/*  transform.c  */

#include "../ETree.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   transform an ETree object by 
   (1) merging small fronts into larger fronts
       using the ETree_mergeFronts() method
   (2) split a large front into a chain of smaller fronts
       using the ETree_splitFronts() method

   created  -- 96jun27, cca
   ------------------------------------------------------
*/
ETree *
ETree_transform (
   ETree   *etree,
   int     vwghts[],
   int     maxzeros,
   int     maxfrontsize,
   int     seed
) {
ETree   *etree2, *etree3 ;
int     nfront, nvtx ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0
   || maxfrontsize <= 0 ) {
   fprintf(stderr, "\n fatal error in ETree_transform(%p,%p,%d,%d,%d)"
           "\n bad input\n", etree, vwghts, maxzeros, maxfrontsize, 
           seed) ;
   exit(-1) ;
}
/*
   ----------------------
   first merge the fronts
   ----------------------
*/
etree2 = ETree_mergeFronts(etree, maxzeros) ;
/*
   ---------------------------------
   first split large interior fronts
   ---------------------------------
*/
etree3 = ETree_splitFronts(etree2, vwghts, maxfrontsize, seed) ;
ETree_free(etree2) ;

return(etree3) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------
   static prototype definition
   ---------------------------
*/
static void visit ( int K, int par[], int fch[], int sib[], 
                    int nodwghts[], int bndwghts[], int map[], 
                    int cost[], int nzeros[], int maxzeros) ;
/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   purpose -- merge the front tree allowing at most
              maxzeros zero entries inside a front

   return -- 
      IV object that has the old front to new front map

   created -- 96jun23, cca
   modified -- 97dec18, cca
      bug fixed that incorrectly counted the number of zeros in a front
   --------------------------------------------------------------------
*/
ETree *
ETree_mergeFronts (
   ETree   *etree,
   int     maxzeros
) {
ETree   *etree2 ;
int     J, K, nfront, nvtx, new ;
int     *bndwghts, *cost, *fch, *map, *nodwghts, 
        *nzeros, *par, *place, *rep, *sib ;
IV      *mapIV ;
Tree    *tree ;
/*
   ---------------
   check the input
   ---------------
*/
if (  etree == NULL 
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0 ) {
   fprintf(stderr, "\n fatal error in ETree_mergeFronts(%p,%d)"
           "\n bad input\n", etree, maxzeros) ;
   exit(-1) ;
}
tree     = etree->tree ;
nodwghts = IVinit(nfront, 0) ;
bndwghts = IVinit(nfront, 0) ;
par = IVinit(nfront, -1) ;
fch = IVinit(nfront, -1) ;
sib = IVinit(nfront, -1) ;
IVcopy(nfront, par, tree->par) ;
IVcopy(nfront, fch, tree->fch) ;
IVcopy(nfront, sib, tree->sib) ;
IVcopy(nfront, nodwghts, IV_entries(etree->nodwghtsIV)) ;
IVcopy(nfront, bndwghts, IV_entries(etree->bndwghtsIV)) ;
/*
   ----------------------
   set up working storage
   ----------------------
*/
rep = IVinit(nfront, -1) ;
IVramp(nfront, rep, 0, 1) ;
cost   = IVinit(nfront, 0) ;
nzeros = IVinit(nfront, 0) ;
/*
   ------------------------------------------
   perform a post-order traversal of the tree
   ------------------------------------------
*/
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n\n ##### visiting front %d", J) ;
   fflush(stdout) ;
#endif
   visit(J, par, fch, sib, nodwghts, bndwghts, 
         rep, cost, nzeros, maxzeros) ;
}
#if MYDEBUG > 0
   fprintf(stdout, "\n\n whoa, finished") ;
   fflush(stdout) ;
#endif
/*
   -------------------------------------------------
   take the map from fronts to representative fronts
   and make the map from old fronts to new fronts
   -------------------------------------------------
*/
mapIV = IV_new() ;
IV_init(mapIV, nfront, NULL) ;
map   = IV_entries(mapIV) ;
place = IVinit(nfront, -1) ;
for ( J = 0, new = 0 ; J < nfront ; J++ ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n rep[%d] = %d", J, rep[J]) ;
   fflush(stdout) ;
#endif
   if ( rep[J] != J ) {
      K = J ;
      while ( rep[K] != K ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n    rep[%d] = %d", K, rep[K]) ;
      fflush(stdout) ;
#endif
         K = rep[K] ;
      }
      rep[J] = K ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting rep[%d] = %d", J, rep[J]) ;
      fflush(stdout) ;
#endif
   } else {
      place[J] = new++ ;
   }
}
for ( J = 0 ; J < nfront ; J++ ) {
   K = rep[J] ;
   map[J] = place[K] ;
}
/*
   -------------------------------
   get the compressed ETree object
   -------------------------------
*/
etree2 = ETree_compress(etree, mapIV) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(par)      ;
IVfree(fch)      ;
IVfree(sib)      ;
IVfree(nodwghts) ;
IVfree(bndwghts) ;
IVfree(rep)      ;
IVfree(cost)     ;
IVfree(nzeros)   ;
IVfree(place)    ;
IVfree(map)      ;
IV_free(mapIV)   ;

return(etree2) ; }
   
/*--------------------------------------------------------------------*/
static void
visit ( 
   int    K,
   int    par[],
   int    fch[],
   int    sib[],
   int    nodwghts[],
   int    bndwghts[],
   int    rep[],
   int    cost[],
   int    nzeros[],
   int    maxzeros
) {
int   bestJ, firstI, J, lastI, nextJ, prevJ ;

if ( fch[K] == -1 ) {
   return ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n inside visit(%d), nzeros(%d) = %d", 
        K, K, nzeros[K]) ;
#endif
/*
   ----------------------------------
   find the child with the least cost
   ----------------------------------
*/
#if MYDEBUG > 0
   fprintf(stdout, "\n    nodwght %d, bndwght %d",
           nodwghts[K], bndwghts[K]) ;
#endif
bestJ = -1 ;
for ( J = fch[K] ; J != -1 ; J = sib[J] ) {
   cost[J] = nzeros[J] 
           + nodwghts[J] * (nodwghts[K] + bndwghts[K] - bndwghts[J]) ;
#if MYDEBUG > 0
   fprintf(stdout, "\n    child %d, nodwght %d, bndwght %d, cost %d",
           J, nodwghts[J], bndwghts[J], cost[J]) ;
#endif
   if (  bestJ == -1 
      || cost[J] < cost[bestJ]
      || (cost[J] == cost[bestJ] && nodwghts[J] < nodwghts[bestJ]) ) {
      bestJ = J ;
   }
}
if ( cost[bestJ] + nzeros[K] > maxzeros ) {
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n    no merge: cost[%d] + nzeros[%d] = %d + %d > %d",
           bestJ, K, cost[bestJ], nzeros[K], maxzeros) ;
#endif
/*
   --------------------------------
   no child can be absorbed, return
   --------------------------------
*/
   return ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n    merging child %d into %d", bestJ, K) ;
#endif
/*
   -------------------------
   absorb child bestJ into K
   -------------------------
*/
for ( J = fch[K], prevJ = -1 ; J != bestJ ; J = sib[J] ) {
   prevJ = J ;
}
nextJ = sib[bestJ] ;
#if MYDEBUG > 0
fprintf(stdout, "\n    previous sibling = %d, next sibling = %d",
        prevJ, nextJ) ;
#endif
if ( (firstI = fch[bestJ]) == -1 ) {
   if ( prevJ == -1 ) {
      fch[K] = nextJ ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting fch[%d] = %d", K, fch[K]) ;
#endif
   } else {
      sib[prevJ] = nextJ ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting sib[%d] = %d", prevJ, sib[prevJ]) ;
#endif
   }
} else {
   firstI = fch[bestJ] ;
   par[firstI] = K ;
#if MYDEBUG > 0
   fprintf(stdout, "\n    setting par[%d] = %d", firstI, par[firstI]) ;
#endif
   if ( (lastI = sib[firstI]) != -1 ) {
      while ( sib[lastI] != -1 ) {
         par[lastI] = K ;
#if MYDEBUG > 0
         fprintf(stdout, 
                 "\n    setting par[%d] = %d", lastI, par[lastI]) ;
#endif
         lastI = sib[lastI] ;
      }
      par[lastI] = K ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting par[%d] = %d", lastI, par[lastI]) ;
#endif
   }
   if ( prevJ == -1 ) {
      fch[K] = firstI ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting fch[%d] = %d", K, fch[K]) ;
#endif
   } else {
      sib[prevJ] = firstI ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting sib[%d] = %d", prevJ, sib[prevJ]) ;
#endif
   }
   if ( lastI != -1 ) {
      sib[lastI] = nextJ ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    setting sib[%d] = %d", lastI, sib[lastI]) ;
#endif
   }
}
rep[bestJ]  =  K ;
nodwghts[K] += nodwghts[bestJ] ;
nzeros[K]   += cost[bestJ] ;
#if MYDEBUG > 0
fprintf(stdout, "\n    setting rep[%d] = %d", bestJ, rep[bestJ]) ;
fprintf(stdout, "\n    setting nodwghts[%d] = %d", K, nodwghts[K]) ;
fprintf(stdout, "\n    setting nzeros[%d] = %d", K, nzeros[K]) ;
#endif
/*
   -------------
   visit K again
   -------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n\n ### visiting front %d", K) ;
   fflush(stdout) ;
#endif
visit(K, par, fch, sib, nodwghts, bndwghts, 
      rep, cost, nzeros, maxzeros) ;

return ; }

/*--------------------------------------------------------------------*/

#define MYDEBUG 0
/*
   -------------------------------------------------
   expand an ETree object by splitting a large front 
   into a chain of smaller fronts.

   created -- 96jun27, cca
   -------------------------------------------------
*/
ETree *
ETree_splitFronts (
   ETree   *etree,
   int     vwghts[],
   int     maxfrontsize,
   int     seed
) {
ETree   *etree2 ;
int     count, front, ii, I, Inew, J, Jnew, nbnd, newsize, nint, nfront,
        nfront2, nsplit, nvtx, prev, size, sizeJ, v, vwght ;
int     *bndwghts, *fch, *head, *indices, *link, *newbndwghts, *newmap, 
        *newnodwghts, *newpar, *nodwghts, *roots, *sib, *vtxToFront ;
Tree    *tree ;
/*
   ---------------
   check the input
   ---------------
*/
if ( etree == NULL
   || (nfront = etree->nfront) <= 0
   || (nvtx = etree->nvtx) <= 0
   || maxfrontsize <= 0 ) {
   fprintf(stderr, "\n fatal error in ETree_splitFronts(%p,%p,%d,%d)"
           "\n bad input\n", etree, vwghts, maxfrontsize, seed) ;
   exit(-1) ;
}
tree       = etree->tree ;
fch        = tree->fch ;
sib        = tree->sib ;
nodwghts   = IV_entries(etree->nodwghtsIV) ;
bndwghts   = IV_entries(etree->bndwghtsIV) ;
vtxToFront = IV_entries(etree->vtxToFrontIV) ;
/*
   --------------------------
   set up the working storage
   --------------------------
*/
newpar      = IVinit(nvtx,   -1) ;
roots       = IVinit(nfront, -1) ;
newmap      = IVinit(nvtx,   -1) ;
newnodwghts = IVinit(nvtx,   -1) ;
newbndwghts = IVinit(nvtx,   -1) ;
head        = IVinit(nfront, -1) ;
link        = IVinit(nvtx,   -1) ;
indices     = IVinit(nvtx,   -1) ;
for ( v = 0 ; v < nvtx ; v++ ) {
   front = vtxToFront[v] ;
   link[v] = head[front] ;
   head[front] = v ;
}
/*
   ------------------------------------------------
   execute a post-order traversal of the front tree
   ------------------------------------------------
*/
nfront2 = 0 ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
   sizeJ = 0 ;
   for ( v = head[J], count = 0 ; v != -1 ; v = link[v] ) {
      indices[count++] = v ;
      vwght = (vwghts != NULL) ? vwghts[v] : 1 ;
      sizeJ += vwght ;
   }
   if ( sizeJ != nodwghts[J] ) {
      fprintf(stderr, "\n fatal error in ETree_splitFronts(%p,%p,%d,%d)"
             "\n J = %d, sizeJ = %d, nodwght = %d\n", 
             etree, vwghts, maxfrontsize, seed, J, sizeJ, nodwghts[J]) ;
      exit(-1) ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n\n checking out front %d, size %d", J, sizeJ) ;
#endif
/*
   if ( sizeJ <= maxfrontsize ) {
*/
   if ( sizeJ <= maxfrontsize || fch[J] == -1 ) {
/*
      -------------------------------------------
      this front is small enough (or is a domain)
      -------------------------------------------
*/
      Jnew = nfront2++ ;
      for ( ii = 0 ; ii < count ; ii++ ) {
         v = indices[ii] ;
         newmap[v] = Jnew ;
#if MYDEBUG > 1
            fprintf(stdout, "\n   mapping vertex %d into new front %d",
                    v, Jnew) ;
#endif
      }
      for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
         Inew = roots[I] ;
         newpar[Inew] = Jnew ;
      }
      newnodwghts[Jnew] = nodwghts[J] ;
      newbndwghts[Jnew] = bndwghts[J] ;
      roots[J] = Jnew ;
#if MYDEBUG > 0
      fprintf(stdout, "\n    front is small enough, Jnew = %d", Jnew) ;
#endif
   } else {
/*
      ------------------------------------------
      this front is too large, split into pieces 
      whose size differs by one vertex
      ------------------------------------------
*/
      nsplit  = (sizeJ + maxfrontsize - 1)/maxfrontsize ;
      newsize = sizeJ / nsplit ;
      if ( sizeJ % nsplit != 0 ) {
         newsize++ ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
         "\n    front is too large, %d target fronts, target size = %d",
         nsplit, newsize) ;
#endif
      prev    = -1 ;
      nint    = nodwghts[J] ;
      nbnd    = nint + bndwghts[J] ;
      if ( seed > 0 ) {
         IVshuffle(count, indices, seed) ;
      }
      ii = 0 ;
      while ( ii < count ) {
         Jnew = nfront2++ ;
         size = 0 ;
         while ( ii < count ) {
            v = indices[ii] ;
            vwght = (vwghts != NULL) ? vwghts[v] : 1 ;
#if MYDEBUG > 0
            fprintf(stdout, 
                "\n   ii = %d, v = %d, vwght = %d, size = %d",
                ii, v, vwght, size) ;
#endif
/*
   -----------------------------------------------
   97aug28, cca
   bug fix. front is created even if it is too big
   -----------------------------------------------
*/
            if ( newsize >= size + vwght || size == 0 ) {
               newmap[v] = Jnew ;
               size += vwght ;
#if MYDEBUG > 0
               fprintf(stdout, 
                "\n   mapping vertex %d into new front %d, size = %d",
                v, Jnew, size) ;
#endif
               ii++ ;
            } else {
               break ;
            }
         }
         if ( prev == -1 ) {
            for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
               Inew = roots[I] ;
               newpar[Inew] = Jnew ;
            }
         } else {
            newpar[prev] = Jnew ;
         }
         prev = Jnew ;
         newnodwghts[Jnew] = size ;
         nbnd = nbnd - size ;
         newbndwghts[Jnew] = nbnd ;
#if MYDEBUG > 0
         fprintf(stdout, "\n    new front %d, size %d, bnd %d",
                 Jnew, newnodwghts[Jnew], newbndwghts[Jnew]) ;
#endif
      }
      roots[J] = Jnew ;
   }
}
/*
   ---------------------------
   create the new ETree object
   ---------------------------
*/
etree2 = ETree_new() ;
ETree_init1(etree2, nfront2, nvtx) ;
IVcopy(nfront2, etree2->tree->par, newpar) ;
Tree_setFchSibRoot(etree2->tree) ;
IVcopy(nvtx, IV_entries(etree2->vtxToFrontIV), newmap) ;
IVcopy(nfront2, IV_entries(etree2->nodwghtsIV), newnodwghts) ;
IVcopy(nfront2, IV_entries(etree2->bndwghtsIV), newbndwghts) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(newpar) ;
IVfree(roots)  ;
IVfree(newmap) ;

return(etree2) ; }

/*--------------------------------------------------------------------*/
