/*  testTransform.c  */

#include "../../ETree.h"
#include "../../SymbFac.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------------------
   read in an ETree object and testTransform it by 
   (1) merge a subtree of small fronts into a larger front
   (2) split a large front into a chain of smaller fronts

   created -- 96jun27, cca
   -------------------------------------------------------
*/
{
char     *inETreeFileName, *inGraphFileName, *outETreeFileName ;
double   nfops1, nfops2, nfops3, t1, t2 ;
Graph    *graph ;
int      ierr, maxsize, maxzeros, msglvl, nfent1, nfent2, nfent3,
         nfind1, nfind2, nfind3, 
         nfront, nleaves1, nleaves2, nleaves3, nnode1, nnode2, nnode3,
         rc, seed ;
int      *temp ;
IVL      *symbfacIVL ;
ETree    *etree, *etree2 ;
FILE     *msgFile ;

if ( argc != 9 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile inETreeFile inGraphFile outETreeFile "
"\n         maxzeros maxsize seed"
      "\n    msglvl       -- message level"
      "\n    msgFile      -- message file"
      "\n    inETreeFile  -- input file, must be *.etreef or *.etreeb"
      "\n    inGraphFile  -- input file, must be *.graphf or *.graphb"
      "\n    outETreeFile -- output file, must be *.etreef or *.etreeb"
      "\n    maxzeros     -- maximum number of zeros in a front"
      "\n    maxsize      -- maximum number of vertices in a front"
      "\n    seed         -- random number seed"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inETreeFileName  = argv[3] ;
inGraphFileName  = argv[4] ;
outETreeFileName = argv[5] ;
maxzeros         = atoi(argv[6]) ;
maxsize          = atoi(argv[7]) ;
seed             = atoi(argv[8]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl       -- %d" 
        "\n msgFile      -- %s" 
        "\n inETreeFile  -- %s" 
        "\n inGraphFile  -- %s" 
        "\n outETreeFile -- %s" 
        "\n maxzeros     -- %d" 
        "\n maxsize      -- %d" 
        "\n seed         -- %d" 
        "\n",
        argv[0], msglvl, argv[2], inETreeFileName, inGraphFileName, 
        outETreeFileName, maxzeros, maxsize, seed) ;
fflush(msgFile) ;
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
if ( strcmp(inETreeFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
etree = ETree_new() ;
MARKTIME(t1) ;
rc = ETree_readFromFile(etree, inETreeFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in etree from file %s",
        t2 - t1, inETreeFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from ETree_readFromFile(%p,%s)",
           rc, etree, inETreeFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading ETree object from file %s",
        inETreeFileName) ;
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(etree, msgFile) ;
} else {
   ETree_writeStats(etree, msgFile) ;
}
fflush(msgFile) ;
/*
   ------------------------
   read in the Graph object
   ------------------------
*/
if ( strcmp(inGraphFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
graph = Graph_new() ;
MARKTIME(t1) ;
rc = Graph_readFromFile(graph, inGraphFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
        t2 - t1, inGraphFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from Graph_readFromFile(%p,%s)",
           rc, graph, inGraphFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading Graph object from file %s",
        inGraphFileName) ;
if ( msglvl > 2 ) {
   Graph_writeForHumanEye(graph, msgFile) ;
} else {
   Graph_writeStats(graph, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------------
   compute the statistics
   ----------------------
*/
nnode1 = etree->tree->n ;
nfind1 = ETree_nFactorIndices(etree) ;
nfent1 = ETree_nFactorEntries(etree, 1) ;
nfops1 = ETree_nFactorOps(etree, 1) ;
nleaves1 = Tree_nleaves(etree->tree) ;
/*
fprintf(stdout, "\n root front %d has %d vertices",
        etree->tree->root,
        etree->nodwghtsIV->vec[etree->tree->root]) ;
*/
/*
   --------------------------
   transform the ETree object
   --------------------------
*/
MARKTIME(t1) ;
etree2 = ETree_transform(etree, graph->vwghts, 
                         maxzeros, maxsize, seed) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : transform the supernode partition",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n new ETree object") ;
   ETree_writeForHumanEye(etree2, msgFile) ;
}
nnode2 = etree2->tree->n ;
nfind2 = ETree_nFactorIndices(etree2) ;
nfent2 = ETree_nFactorEntries(etree2, 1) ;
nfops2 = ETree_nFactorOps(etree2, 1) ;
nleaves2 = Tree_nleaves(etree2->tree) ;
if ( msglvl > 0 ) {
   fprintf(msgFile,
           "\n   old tree  %5d %8d %10d %10.3f %10d"
           "\n   new tree  %5d %8d %10d %10.3f %10d",
           nnode1, nfind1, nfent1, nfops1*1.e-6, nleaves1,
           nnode2, nfind2, nfent2, nfops2*1.e-6, nleaves2) ;
   fprintf(msgFile,
           "\n   old tree  %5d %8d %10d %10.0f %10d"
           "\n   new tree  %5d %8d %10d %10.0f %10d",
           nnode1, nfind1, nfent1, nfops1, nleaves1,
           nnode2, nfind2, nfent2, nfops2, nleaves2) ;
   fprintf(msgFile, 
           "\n maxsize = %d, ratios = %10.3f %10.3f %10.3f %10.3f", 
           maxsize,
           ((double) nnode2)/nnode1, ((double) nfind2)/nfind1,
           ((double) nfent2)/nfent1, ((double) nfops2)/nfops1) ;
}
nfront = etree2->nfront ;
temp = IVinit(nfront, 0) ;
IVcopy(nfront, temp, IV_entries(etree2->nodwghtsIV)) ;
IVqsortUp(nfront, temp) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n nodwghts") ;
   IVfp80(msgFile, nfront, temp, 80, &ierr) ;
}
/*
   ----------------------------------------
   create the symbolic factorization object
   ----------------------------------------
*/
symbfacIVL = SymbFac_initFromGraph(etree2, graph) ;
nnode3 = etree2->tree->n ;
nfind3 = ETree_nFactorIndices(etree2) ;
nfent3 = ETree_nFactorEntries(etree2, 1) ;
nfops3 = ETree_nFactorOps(etree2, 1) ;
nleaves3 = Tree_nleaves(etree2->tree) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n after symbolic factorization") ;
   fprintf(msgFile,
           "\n   last tree %5d %8d %10d %10.3f %10d",
           nnode3, nfind3, nfent3, nfops3*1.e-6, nleaves3) ;
   fprintf(msgFile,
           "\n   last tree %5d %8d %10d %10.0f %10d",
           nnode3, nfind3, nfent3, nfops3, nleaves3) ;
   fprintf(msgFile, 
           "\n maxsize = %d, ratios = %10.3f %10.3f %10.3f %10.3f", 
           maxsize,
           ((double) nnode3)/nnode1, ((double) nfind3)/nfind1,
           ((double) nfent3)/nfent1, ((double) nfops3)/nfops1) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n after symbolic factorization") ;
   ETree_writeForHumanEye(etree2, msgFile) ;
   fprintf(msgFile, "\n\n after symbolic factorization") ;
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
}
/*
   --------------------------
   write out the ETree object
   --------------------------
*/
if ( strcmp(outETreeFileName, "none") != 0 ) {
   MARKTIME(t1) ;
   rc = ETree_writeToFile(etree2, outETreeFileName) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %9.5f : write etree to file %s",
           t2 - t1, outETreeFileName) ;
}
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from ETree_writeToFile(%p,%s)",
           rc, etree, outETreeFileName) ;
}
/*
   ---------------------
   free the ETree object
   ---------------------
*/
ETree_free(etree) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
