/*  testStorage.c  */

#include "../../ETree.h"
#include "../../SymbFac.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------------------
   read in an ETree object.
   read in a Graph object.
   get the symbolic factorization IVL object.
   compute the storage profiles for the general sparse and
   multifrontal methods.
   
   created -- 96oct03, cca
   ---------------------------------------------------------------
*/
{
char     *inETreeFileName, *inGraphFileName ;
double   elapsed, nfops1, t1, t2 ;
double   *GSvec, *MFvec, *backwardops, *forwardops, *vmetric ;
DV       *vmetricDV ;
Graph    *graph ;
int      J, msglvl, nfent1, nfind1, nfront, nleaves1, nnode1, rc ;
IVL      *symbfacIVL ;
ETree    *etree ;
FILE     *msgFile ;
Tree     *tree ;

if ( argc != 5 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile inETreeFile inGraphFile "
"\n    msglvl       -- message level"
"\n    msgFile      -- message file"
"\n    inETreeFile  -- input file, must be *.etreef or *.etreeb"
"\n    inGraphFile  -- input file, must be *.graphf or *.graphb"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inETreeFileName  = argv[3] ;
inGraphFileName  = argv[4] ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl        -- %d" 
        "\n msgFile       -- %s" 
        "\n inETreeFile   -- %s" 
        "\n inGraphFile   -- %s" 
        "\n",
        argv[0], msglvl, argv[2], 
        inETreeFileName, inGraphFileName) ;
fflush(msgFile) ;
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
if ( strcmp(inETreeFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
etree = ETree_new() ;
MARKTIME(t1) ;
rc = ETree_readFromFile(etree, inETreeFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in etree from file %s",
        t2 - t1, inETreeFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from ETree_readFromFile(%p,%s)",
           rc, etree, inETreeFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading ETree object from file %s",
        inETreeFileName) ;
if ( msglvl > 2 ) {
   ETree_writeForHumanEye(etree, msgFile) ;
} else {
   ETree_writeStats(etree, msgFile) ;
}
ETree_leftJustify(etree) ;
fprintf(msgFile, "\n\n %d LU entries", ETree_nFactorEntries(etree, 2)) ;
fflush(msgFile) ;
/*
   ------------------------
   read in the Graph object
   ------------------------
*/
if ( strcmp(inGraphFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
graph = Graph_new() ;
MARKTIME(t1) ;
rc = Graph_readFromFile(graph, inGraphFileName) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : read in graph from file %s",
        t2 - t1, inGraphFileName) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from Graph_readFromFile(%p,%s)",
           rc, graph, inGraphFileName) ;
   exit(-1) ;
}
fprintf(msgFile, "\n\n after reading Graph object from file %s",
        inGraphFileName) ;
if ( msglvl > 2 ) {
   Graph_writeForHumanEye(graph, msgFile) ;
} else {
   Graph_writeStats(graph, msgFile) ;
}
fflush(msgFile) ;
/*
   ----------------------
   compute the statistics
   ----------------------
*/
tree   = etree->tree   ;
nfront = etree->nfront ;
nnode1 = etree->tree->n ;
nfind1 = ETree_nFactorIndices(etree) ;
nfent1 = ETree_nFactorEntries(etree, 1) ;
nfops1 = ETree_nFactorOps(etree, 1) ;
nleaves1 = Tree_nleaves(etree->tree) ;
fprintf(stdout, "\n root front %d has %d vertices",
        etree->tree->root,
        etree->nodwghtsIV->vec[etree->tree->root]) ;
/*
   --------------------------------------------
   create the symbolic factorization IVL object
   --------------------------------------------
*/
symbfacIVL = SymbFac_initFromGraph(etree, graph) ;
fprintf(msgFile, 
        "\n\n symbolic factorization IVL object in old ordering") ;
if ( msglvl > 2 ) {
   IVL_writeForHumanEye(symbfacIVL, msgFile) ;
} else {
   IVL_writeStats(symbfacIVL, msgFile) ;
}
fflush(msgFile) ;
/*
   --------------------------
   get the operations profile
   --------------------------
*/
vmetricDV = ETree_backwardOps(etree, graph->vwghts, symbfacIVL) ;
vmetric = DV_entries(vmetricDV) ;
backwardops = DVinit(nfront, 0.0) ;
elapsed = 0.0 ;
for ( J = Tree_postOTfirst(etree->tree) ;
      J != -1 ;
      J = Tree_postOTnext(etree->tree, J) ) {
   elapsed += vmetric[J] ;
   backwardops[J] = elapsed ;
}
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n sum of backward ops = %.0f",
           DV_sum(vmetricDV)) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n backward ops") ;
   DVfprintf(msgFile, nfront, backwardops) ;
}
DV_free(vmetricDV) ;
vmetricDV = ETree_forwardOps(etree)   ;
vmetric   = DV_entries(vmetricDV) ;
forwardops = DVinit(nfront, 0.0) ;
elapsed = 0.0 ;
for ( J = Tree_postOTfirst(etree->tree) ;
      J != -1 ;
      J = Tree_postOTnext(etree->tree, J) ) {
   elapsed += vmetric[J] ;
   forwardops[J] = elapsed ;
}
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n\n sum of forward ops = %.0f",
           DV_sum(vmetricDV)) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n forward ops") ;
   DVfprintf(msgFile, nfront, forwardops) ;
}
/*
   --------------------------------------
   get the general sparse storage profile
   --------------------------------------
*/
GSvec = DVinit(nfront, 0.0) ;
ETree_GSstorageProfile(etree, symbfacIVL, graph->vwghts, GSvec) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n GSvec storage") ;
   DVfprintf(msgFile, nfront, GSvec) ;
}
/*
   ------------------------------------
   get the multifrontal storage profile
   ------------------------------------
*/
MFvec = DVinit(nfront, 0.0) ;
ETree_MFstackProfile(etree, MFvec) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n MFvec storage") ;
   DVfprintf(msgFile, nfront, MFvec) ;
}
if ( msglvl > 0 ) {
   fprintf(msgFile, 
           "\n %% backward-ops GS-storage forward-ops MF-storage") ;
   fprintf(msgFile, "\n data = [ ...") ;
   for ( J = Tree_postOTfirst(tree) ;
         J != -1 ;
         J = Tree_postOTnext(tree, J) ) {
      fprintf(msgFile, "\n %12.0f %12.0f %12.0f %12.0f", 
              backwardops[J], GSvec[J], forwardops[J], MFvec[J]) ;
   }
   fprintf(msgFile, " ]") ;
}
/*
   ----------------
   free the objects
   ----------------
*/
ETree_free(etree) ;
Graph_free(graph) ;
IVL_free(symbfacIVL) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
