/*  listmanip.c  */

#include "../DVL.h"

#define   MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   fills size of and pointer to the entries of a list

   set *psize = size of list ilist
   set *pvec = base address of list ilist

   use as follows :

   DVL_listAndSize(dvl, ilist, &size, &vec) ;
   for ( i = 0 ; i < size ; i++ ) {
      do something with vec[i] ;
   }

   created -- 95sep22, cca
   ----------------------------------------------------------------
*/
void
DVL_listAndSize (
   DVL      *dvl,
   int      ilist,
   int      *psize,
   double   **pvec
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || ilist < 0 || ilist >= dvl->nlist
     || psize == NULL || pvec == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_listAndSize(%p,%d,%p,%p)"
           "\n bad input\n", dvl, ilist, psize, pvec) ;
   exit(-1) ;
}
/*
   --------------------------
   set the two pointer fields
   --------------------------
*/
*psize = dvl->sizes[ilist] ;
*pvec  = dvl->p_vec[ilist] ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------
   returns a pointer to the first element in list ilist

   to be used as a general iterator, e.g.,

   for ( pi = DVL_firstInList(dvl, ilist) ;
         pi != NULL ;
         pi = DVL_nextInList(dvl, ilist, pi) ) {
      do something ;
   }

   created -- 95sep27, cca
   ----------------------------------------------------
*/
double *
DVL_firstInList (
   DVL   *dvl,
   int   ilist
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_firstInList(%p,%d)"
           "\n bad input, dvl is NULL\n", dvl, ilist) ;
   exit(-1) ;
}
if ( ilist < 0 || ilist >= dvl->nlist ) {
   fprintf(stderr, "\n fatal error in DVL_firstInList(%p,%d)"
           "\n bad input, ilist = %d, must be in [0,%d) \n", 
           dvl, ilist, ilist, dvl->nlist) ;
   exit(-1) ;
}
if ( dvl->sizes[ilist] == 0 ) {
   return(NULL) ;
} else if ( dvl->p_vec[ilist] != NULL ) {
   return(dvl->p_vec[ilist]) ;
} else {
   fprintf(stderr, "\n fatal error in DVL_firstInList(%p,%d)"
           "\n size > 0 but list is NULL\n", dvl, ilist) ;
   exit(-1) ;
}
}
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------
   returns a pointer to the next element in list ilist

   to be used as a general iterator, e.g.,

   for ( pd = DVL_firstInList(dvl, ilist) ;
         pd != NULL ;
         pd = DVL_nextInList(dvl, ilist, pd) ) {
      do something ;
   }

   created -- 95sep27, cca
   ----------------------------------------------------
*/
double *
DVL_nextInList (
   DVL      *dvl,
   int      ilist,
   double   *pd
) {
int   offset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_nextInList(%p,%d,%p)"
           "\n bad input, dvl is NULL\n", dvl, ilist, pd) ;
   exit(-1) ;
}
if ( ilist < 0 || ilist >= dvl->nlist ) {
   fprintf(stderr, "\n fatal error in DVL_nextInList(%p,%d,%p)"
           "\n bad input, ilist = %d, must be in [0,%d) \n", 
           dvl, ilist, pd, ilist, dvl->nlist) ;
   exit(-1) ;
}
if (  (pd == NULL) 
   || ((offset = pd - dvl->p_vec[ilist]) < 0)
   || offset >= dvl->sizes[ilist] ) {
   fprintf(stderr, "\n fatal error in DVL_nextInList(%p,%d,%p)"
           "\n bad pointer\n", dvl, ilist, pd) ;
   exit(-1) ;
} else if ( offset == dvl->sizes[ilist] - 1 ) {
   return(NULL) ;
} else {
   return(pd+1) ;
}
}
/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   purpose -- to set or reset a list.

   ilist -- list id to set or reset
   size  -- size of the list
      if the present size of list ilist is smaller than size,
         the old list is free'd (if dvl->type = DVL_SOLO) 
         or lost (if dvl->type = DVL_CHUNKED)
         or un-set (if dvl->type = DVL_UNKNOWN)
         and new storage is allocated (for DVL_SOLO and DVL_CHUNKED)
   vec  -- list vector
      if dvl->type is DVL_UNKNOWN then
         if vec != NULL then
            we set the dvl list pointer to be vec
         endif
      else if vec != NULL
         we copy vec[] into dvl's storage for the list
      endif

   created   -- 95sep27, cca
   last mods -- 95oct06, cca
      type = DVL_UNKNOWN, p_vec[ilist] set to vec 
      only when vec is not NULL
      bug fixed, dvl->sizes[ilist] = size ;
   -----------------------------------------------------------------
*/
void
DVL_setList ( 
   DVL      *dvl, 
   int      ilist, 
   int      size,
   double   vec[]
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_setList(%p,%d,%d,%p)"
           "\n bad input, dvl is NULL\n", dvl, ilist, size, vec) ;
   exit(-1) ;
}
if ( ilist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_setList(%p,%d,%d,%p)"
           "\n bad input, ilist < 0", dvl, ilist, size, vec) ;
   exit(-1) ;
}
if ( ilist >= dvl->maxnlist ) {
   int   newmaxnlist = (int) 1.25*dvl->maxnlist ;
   if ( newmaxnlist < 10 ) {
      newmaxnlist = 10 ;
   } 
   if ( ilist >= newmaxnlist ) {
      newmaxnlist = ilist + 1 ;
   } 
   DVL_setMaxnlist(dvl, newmaxnlist) ;
}
if ( ilist >= dvl->nlist ) {
   dvl->nlist = ilist + 1 ;
}
if ( size == 0 ) {
/*
   ------------------------------------------
   new list is empty, free storage if present
   ------------------------------------------
*/
   if ( dvl->type == DVL_SOLO ) {
      if ( dvl->p_vec[ilist] != NULL ) {
         DVfree(dvl->p_vec[ilist]) ;
      }
   }
   dvl->tsize -= dvl->sizes[ilist] ;
   dvl->sizes[ilist] =   0  ;
   dvl->p_vec[ilist] = NULL ;
} else if ( dvl->type == DVL_UNKNOWN ) {
/*
   -------------------------------
   simply set the size and pointer
   -------------------------------
*/
   dvl->tsize += size - dvl->sizes[ilist] ;
   dvl->sizes[ilist] = size ;
   if ( vec != NULL ) {
      dvl->p_vec[ilist] = vec ;
   }
} else {
/*
   --------------------------------------------------
   the list entries will be copied into dvl's storage
   --------------------------------------------------
*/
/*
fprintf(stdout, "\n dvl->sizes[%d] = %d, size = %d",
        ilist, dvl->sizes[ilist], size) ;
*/
   if ( dvl->sizes[ilist] < size ) {
/*
      --------------------------------------------
      old size (might be zero) is not large enough
      switch over the storage types
      --------------------------------------------
*/
      switch ( dvl->type ) {
      case DVL_SOLO :
         if ( dvl->p_vec[ilist] != NULL ) {
/*
            ---------------------------------------
            free old storage before allocating more
            and decrement the total list size
            ---------------------------------------
*/
            DVfree(dvl->p_vec[ilist]) ;
         }
         dvl->p_vec[ilist] = DVinit(size, -1) ;
         break ;
      case DVL_CHUNKED : {
         Dchunk   *chunk ;
         if (  (chunk = dvl->chunk) == NULL
            || (chunk->size - chunk->inuse) < size ) {
/*
            -------------------------------
            allocate a new chunk of storage
            -------------------------------
*/
            ALLOCATE(chunk, struct _Dchunk, 1) ;
            if ( size < dvl->incr ) {
               chunk->size = dvl->incr ;
            } else {
               chunk->size = size ;
            }
/*
fprintf(stdout, "\n    getting new chunk, size %d", chunk->size) ;
*/
            chunk->inuse = 0 ;
            chunk->base  = DVinit(chunk->size, 0) ;
            chunk->next  = dvl->chunk ;
            dvl->chunk   = chunk ;
         }
/*
         --------------------------
         set pointer for list ilist
         --------------------------
*/
         dvl->p_vec[ilist] = chunk->base + chunk->inuse ;
         chunk->inuse += size ;
         } break ;
      default :
         fprintf(stderr, "\n fatal error in DVL_setList(%p,%d,%d,%p)"
               "\n you are trying to grow a list but type = %d"
               "\n type must be DVL_CHUNKED = 1 or DVL_SOLO = 2\n",
               dvl, ilist, size, vec, dvl->type) ;
         exit(-1) ;
      }
   }
   dvl->tsize += size - dvl->sizes[ilist] ;
   dvl->sizes[ilist] = size ;
   if ( vec != NULL ) {
/*
      --------------------------------
      copy the list into dvl's storage
      --------------------------------
*/
      DVcopy(size, dvl->p_vec[ilist], vec) ;
   }
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   set a pointer to a list but don't allocate storage for the list.
   this method was needed when we form a subgraph with a boundary.
   lists for the interior vertices point into the parent graph,
   but lists for the boundary vertices must be allocated and owned.
   used only for type = DVL_CHUNKED. at some point in the future we
   should rethink the storage semantics for the DVL object.

   created -- 95nov11, cca
   ----------------------------------------------------------------
*/
void
DVL_setPointerToList (
   DVL      *dvl, 
   int      ilist, 
   int      size,
   double   vec[]
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_setPointerToList(%p,%d,%d,%p)"
           "\n bad input, dvl is NULL\n", dvl, ilist, size, vec) ;
   exit(-1) ;
}
if ( dvl->type != DVL_CHUNKED ) {
   fprintf(stderr, "\n fatal error in DVL_setPointerToList(%p,%d,%d,%p)"
           "\n this method is only used with type DVL_CHUNKED\n", 
           dvl, ilist, size, vec) ;
   exit(-1) ;
}
if ( ilist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_setPointerToList(%p,%d,%d,%p)"
           "\n bad input, ilist < 0", dvl, ilist, size, vec) ;
   exit(-1) ;
}
if ( ilist >= dvl->maxnlist ) {
   int   newmaxnlist = (int) 1.25*dvl->maxnlist ;
   if ( newmaxnlist < 10 ) {
      newmaxnlist = 10 ;
   } 
   if ( ilist >= newmaxnlist ) {
      newmaxnlist = ilist + 1 ;
   } 
   DVL_setMaxnlist(dvl, newmaxnlist) ;
}
if ( ilist >= dvl->nlist ) {
   dvl->nlist = ilist + 1 ;
}
if ( dvl->type == DVL_SOLO && dvl->p_vec[ilist] != NULL ) {
   DVfree(dvl->p_vec[ilist]) ;
}
dvl->tsize += size - dvl->sizes[ilist] ;
dvl->sizes[ilist] = size ;
dvl->p_vec[ilist] = vec  ;

return ; }

/*--------------------------------------------------------------------*/
