/*  init.c  */

#include "../DVL.h"

#define   MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   initialize given the type and maximum number of lists.
   used for type DVL_CHUNKED, DVL_SOLO or DVL_UNKNOWN

   created -- 95sep22, cca
   ------------------------------------------------------
*/
void
DVL_init1 ( 
   DVL   *dvl, 
   int   type, 
   int   maxnlist
) {
/*
   -------------------
   check for bad input
   -------------------
*/
if ( dvl == NULL 
  || (type != DVL_CHUNKED && type != DVL_SOLO && type != DVL_UNKNOWN)
  || maxnlist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_init1(%p,%d,%d)"
           "\n bad input", dvl, type, maxnlist) ;
   exit(-1) ;
}
/*
   --------------------------
   clear the data, if present
   --------------------------
*/
DVL_clearData(dvl) ;
/*
   ----------
   initialize
   ----------
*/
dvl->type     = type  ;
dvl->maxnlist = maxnlist ;
dvl->nlist    = maxnlist ;
if ( maxnlist > 0 ) {
   dvl->sizes = IVinit(maxnlist, 0) ;
   dvl->p_vec = PDVinit(maxnlist) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   initialize given the type, number of lists and their total size
   only used when type is DVL_CHUNKED.

   created -- 95sep22, cca
   ---------------------------------------------------------------
*/
void
DVL_init2 ( 
   DVL   *dvl, 
   int   type, 
   int   maxnlist, 
   int   tsize 
) {
/*
   -------------------
   check for bad input
   -------------------
*/
if ( dvl == NULL || type != DVL_CHUNKED || maxnlist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_init2(%p,%d,%d,%d)"
           "\n bad input", dvl, type, maxnlist, tsize) ;
   exit(-1) ;
}
/*
   ---------------------------------
   initialize via DVL_init1() method
   ---------------------------------
*/
DVL_init1(dvl, type, maxnlist) ;
/*
   ----------------------------------
   create chunk to hold tsize entries
   ----------------------------------
*/
if ( tsize > 0 ) {
   ALLOCATE(dvl->chunk, struct _Dchunk, 1) ;
   dvl->chunk->size  = tsize ;
   dvl->chunk->inuse = 0 ;
/*
   dvl->chunk->base  = DVinit(tsize, 0.0) ;
*/
   dvl->chunk->base  = DVinit2(tsize) ;
   dvl->chunk->next  = NULL ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------
   initialize from a vector of list sizes.
   used with DVL_SOLO or DVL_CHUNKED.

   created -- 95sep22, cca
   --------------------------------------
*/
void
DVL_init3 ( 
   DVL   *dvl, 
   int   type, 
   int   maxnlist, 
   int   sizes[] 
) {
int   ilist ;
/*
   -------------------
   check for bad input
   -------------------
*/
if ( dvl == NULL || (type != DVL_CHUNKED && type != DVL_SOLO)
  || maxnlist < 0 || sizes == NULL ) {
   fprintf(stderr,
           "\n fatal error in DVL_init3(%p,%d,%d,%p)"
           "\n bad input", dvl, type, maxnlist, sizes) ;
   exit(-1) ;
}
switch ( type ) {
case DVL_SOLO :
/*
   ---------------------------------
   initialize via DVL_init1() method
   ---------------------------------
*/
   DVL_init1(dvl, type, maxnlist) ;
   break ;
case DVL_CHUNKED :
/*
   ---------------------------------
   initialize via DVL_init2() method
   ---------------------------------
*/
   DVL_init2(dvl, type, maxnlist, IVsum(maxnlist, sizes)) ;
   break ;
}
/*
   -------------------------
   set the size of each list
   -------------------------
*/
for ( ilist = 0 ; ilist < maxnlist ; ilist++ ) {
   DVL_setList(dvl, ilist, sizes[ilist], NULL) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   this method resizes the maximum number of lists,
   replacing the old sizes[] and p_vec[] vectors
   as necessary. the nlist value is NOT reset.

   created -- 96dec05, cca
   ------------------------------------------------
*/
void
DVL_setMaxnlist (
   DVL   *dvl,
   int   newmaxnlist
) {
if ( dvl == NULL || newmaxnlist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_setMaxnlist(%p,%d)"
           "\n bad input\n", dvl, newmaxnlist) ;
   exit(-1) ;
}
if ( newmaxnlist != dvl->maxnlist ) {
   double   **pdvec ;
   int      *ivec ;
/*
   --------------------------------------------
   allocate (and fill) the larger sizes[] array
   --------------------------------------------
*/
   ivec = IVinit(newmaxnlist, 0) ;
   if ( dvl->sizes != NULL ) {
      if ( dvl->nlist > newmaxnlist ) {
         IVcopy(newmaxnlist, ivec, dvl->sizes) ;
      } else if ( dvl->nlist > 0 ) {
         IVcopy(dvl->nlist, ivec, dvl->sizes) ;
      }
      IVfree(dvl->sizes) ;
   }
   dvl->sizes = ivec ;
/*
   --------------------------------------------
   allocate (and fill) the larger p_vec[] array
   --------------------------------------------
*/
   pdvec = PDVinit(newmaxnlist) ;
   if ( dvl->p_vec != NULL ) {
      if ( dvl->nlist > newmaxnlist ) {
         PDVcopy(newmaxnlist, pdvec, dvl->p_vec) ;
      } else if ( dvl->nlist > 0 ) {
         PDVcopy(dvl->nlist, pdvec, dvl->p_vec) ;
      }
      PDVfree(dvl->p_vec) ;
   }
   dvl->p_vec = pdvec ;
/*
   -----------------------------------
   set the new maximum number of lists
   -----------------------------------
*/
   dvl->maxnlist = newmaxnlist ;
   if ( dvl->nlist > newmaxnlist ) {
      dvl->nlist = newmaxnlist ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   this method resizes the number of lists,
   replacing the old sizes[] and p_vec[] vectors
   as necessary.

   created -- 96dec05, cca
   ---------------------------------------------
*/
void
DVL_setNlist (
   DVL   *dvl,
   int   newnlist
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || newnlist < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_setNlist(%p,%d)"
           "\n bad input\n", dvl, newnlist) ;
   exit(-1) ;
}
if ( newnlist > dvl->maxnlist ) {
/*
   ------------------------------------
   increase the maximum number of lists
   ------------------------------------
*/
   DVL_setMaxnlist(dvl, newnlist) ;
}
/*
   -------------------
   set the nlist field
   -------------------
*/
dvl->nlist = newnlist ;

return ; }

/*--------------------------------------------------------------------*/
