/*  IO.c  */

#include "../DVL.h"

static const char *suffixb = ".dvlb" ;
static const char *suffixf = ".dvlf" ;

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   purpose -- to read in an DVL object from a file

   input --

      fn -- filename, must be *.dvlb or *.dvlf

   return value -- 1 if success, 0 if failure

   created -- 95sep29, cca
   -----------------------------------------------
*/
int
DVL_readFromFile ( 
   DVL    *dvl, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fn == NULL ) {
   fprintf(stderr, 
    "\n error in DVL_readFromFile(%p,%s), file %s, line %d"
    "\n bad input\n", dvl, fn, __FILE__, __LINE__) ;
   return(0) ;
}
switch ( dvl->type ) {
case DVL_CHUNKED :
case DVL_SOLO    :
case DVL_UNKNOWN :
   break ;
default :
   fprintf(stderr, "\n error in DVL_readFromFile(%p,%s)"
    "\n bad type = %d", dvl, fn, dvl->type) ;
   return(0) ;
}
/*
   -------------
   read the file
   -------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "rb")) == NULL ) {
         fprintf(stderr, "\n error in DVL_readFromFile(%p,%s)"
                 "\n unable to open file %s", dvl, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DVL_readFromBinaryFile(dvl, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "r")) == NULL ) {
         fprintf(stderr, "\n error in DVL_readFromFile(%p,%s)"
                 "\n unable to open file %s", dvl, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DVL_readFromFormattedFile(dvl, fp) ;
         fclose(fp) ;
      }
   } else {
      fprintf(stderr, "\n error in DVL_readFromFile(%p,%s)"
              "\n bad DVL file name %s,"
              "\n must end in %s (binary) or %s (formatted)\n",
              dvl, fn, fn, suffixb, suffixf) ;
      rc = 0 ;
   }
} else {
   fprintf(stderr, "\n error in DVL_readFromFile(%p,%s)"
       "\n bad DVL file name %s,"
       "\n must end in %s (binary) or %s (formatted)\n",
       dvl, fn, fn, suffixb, suffixf) ;
   rc = 0 ;
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   purpose -- to read an DVL object from a formatted file

   return value -- 1 if success, 0 if failure

   created -- 95sep29, cca
   ------------------------------------------------------
*/
int
DVL_readFromFormattedFile ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int   nlist, rc, type ;
int   itemp[3] ;
int   *sizes ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DVL_readFromFormattedFile(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   return(0) ;
}
switch ( dvl->type ) {
case DVL_CHUNKED :
case DVL_SOLO    :
   break ;
default :
   fprintf(stderr, "\n error in DVL_readFormattedFile(%p,%p)"
    "\n bad type = %d", dvl, fp, dvl->type) ;
   return(0) ;
}
/*
   -------------------------------------------
   save the dvl type and clear the data fields
   -------------------------------------------
*/
type = dvl->type ;
DVL_clearData(dvl) ;
/*
   -----------------------------------
   read in the three scalar parameters
   type, # of lists, # of entries
   -----------------------------------
*/
if ( (rc = IVfscanf(fp, 3, itemp)) != 3 ) {
   fprintf(stderr, "\n error in DVL_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", dvl, fp, rc, 3) ;
   return(0) ;
}
nlist = itemp[1] ;
/*
   --------------------------
   read in the sizes[] vector
   --------------------------
*/
sizes = IVinit(nlist, 0) ;
if ( (rc = IVfscanf(fp, nlist, sizes)) != nlist ) {
   fprintf(stderr, "\n error in DVL_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", dvl, fp, rc, nlist) ;
   return(0) ;
}
/*
fprintf(stdout, "\n %d lists, %d sum ", 
        nlist, IVsum(nlist, sizes)) ;
fflush(stdout) ;
*/
/*
   ---------------------
   initialize the object
   ---------------------
*/
DVL_init3(dvl, type, nlist, sizes) ;
IVfree(sizes) ;
/*
   -----------------------
   now read in the entries
   -----------------------
*/
switch ( type ) {
case DVL_SOLO : {
   int      ilist, size ;
   double   *dvec ;

   for ( ilist = 0 ; ilist < nlist ; ilist++ ) {
      DVL_listAndSize(dvl, ilist, &size, &dvec) ;
      if ( size > 0 ) {
         if ( (rc = DVfscanf(fp, size, dvec)) != size ) {
            fprintf(stderr, 
                    "\n error in DVL_readFromFormattedFile(%p,%p)"
                    "\n list %d, %d items of %d read\n", 
                    dvl, fp, ilist, rc, size) ;
            return(0) ;
         }
      }
   }
   } break ;
case DVL_CHUNKED : {
/*
   --------------------------------------------------------
   read in the entries into the contiguous block of storage
   --------------------------------------------------------
*/
   if ( (rc = DVfscanf(fp, dvl->tsize, dvl->chunk->base)) 
        != dvl->tsize ) {
      fprintf(stderr, "\n error in DVL_readFromFormattedFile(%p,%p)"
              "\n %d items of %d read\n", dvl, fp, rc, dvl->tsize) ;
      return(0) ;
   }
   } break ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------
   purpose -- to read an DVL object from a binary file

   return value -- 1 if success, 0  if failure

   created -- 95sep29, cca
   ---------------------------------------------------
*/
int
DVL_readFromBinaryFile ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int   nlist, rc, type ;
int   itemp[3] ;
int   *sizes ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_readFromBinaryFile(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   return(0) ;
}
switch ( dvl->type ) {
case DVL_CHUNKED :
case DVL_SOLO    :
   break ;
default :
   fprintf(stderr, "\n error in DVL_readBinaryFile(%p,%p)"
    "\n bad type = %d", dvl, fp, dvl->type) ;
   return(0) ;
}
/*
   -------------------------------------------
   save the dvl type and clear the data fields
   -------------------------------------------
*/
type = dvl->type ;
DVL_clearData(dvl) ;
/*
   -----------------------------------
   read in the three scalar parameters
   type, # of lists, # of entries
   -----------------------------------
*/
if ( (rc = fread((void *) itemp, sizeof(int), 3, fp)) != 3 ) {
   fprintf(stderr, "\n error in DVL_readFromBinaryFile(%p,%p)"
           "\n itemp(3) : %d items of %d read\n", dvl, fp, rc, 3) ;
   return(0) ;
}
nlist = itemp[1] ;
/*
   --------------------------
   read in the sizes[] vector
   --------------------------
*/
sizes = IVinit(nlist, 0) ;
if ( (rc = fread((void *) sizes, sizeof(int), nlist, fp)) != nlist ) {
   fprintf(stderr, "\n error in DVL_readFromBinaryFile(%p,%p)"
           "\n sizes(%d) : %d items of %d read\n", 
           dvl, fp, nlist, rc, nlist) ;
   return(0) ;
}
/*
   ---------------------
   initialize the object
   ---------------------
*/
DVL_init3(dvl, type, nlist, sizes) ;
IVfree(sizes) ;
/*
   -------------------
   read in the entries
   -------------------
*/
switch ( type ) {
case DVL_SOLO : {
   int      ilist, size ;
   double   *dvec ;

   for ( ilist = 0 ; ilist < nlist ; ilist++ ) {
      DVL_listAndSize(dvl, ilist, &size, &dvec) ;
      if ( (rc = fread((void *) dvec, sizeof(double), size, fp)) 
               != size ) {
         fprintf(stderr, "\n error in DVL_readFromBinaryFile(%p,%p)"
                 "\n list %d, %d items of %d read\n", 
                 dvl, fp, ilist, rc, size) ;
         return(0) ;
      }
   }
   } break ;
case DVL_CHUNKED : {
/*
   --------------------------------------------------------
   read in the entries into the contiguous block of storage
   --------------------------------------------------------
*/
   if ( (rc = fread((void *) dvl->chunk->base, 
                    sizeof(double), dvl->tsize, fp)) != dvl->tsize ) {
      fprintf(stderr, "\n error in DVL_readFromBinaryFile(%p,%p)"
              "\n entries(%d) : %d items of %d read\n", 
              dvl, fp, dvl->tsize, rc, dvl->tsize) ;
      return(0) ;
   }
   } break ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   purpose -- to write an DVL object to a file

   input --

      fn -- filename
        *.dvlb -- binary
        *.dvlf -- formatted
        anything else -- for human eye

   return value -- 1 if success, 0 otherwise

   created -- 95sep29, cca
   -------------------------------------------
*/
int
DVL_writeToFile ( 
   DVL    *dvl, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fn == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_writeToFile(%p,%s)"
    "\n bad input\n", dvl, fn) ; 
}
switch ( dvl->type ) {
case DVL_CHUNKED :
case DVL_SOLO    :
case DVL_UNKNOWN :
   break ;
default :
   fprintf(stderr, "\n fatal error in DVL_writeToFile(%p,%s)"
           "\n bad type = %d", dvl, fn, dvl->type) ;
   return(0) ;
}
/*
   ------------------
   write out the file
   ------------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "wb")) == NULL ) {
         fprintf(stderr, "\n error in DVL_writeToFile(%p,%s)"
                 "\n unable to open file %s", dvl, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DVL_writeToBinaryFile(dvl, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "w")) == NULL ) {
         fprintf(stderr, "\n error in DVL_writeToFile(%p,%s)"
                 "\n unable to open file %s", dvl, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DVL_writeToFormattedFile(dvl, fp) ;
         fclose(fp) ;
      }
   } else {
      if ( (fp = fopen(fn, "a")) == NULL ) {
         fprintf(stderr, "\n error in DVL_writeToFile(%p,%s)"
                 "\n unable to open file %s", dvl, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DVL_writeForHumanEye(dvl, fp) ;
         fclose(fp) ;
      }
   }
} else {
   if ( (fp = fopen(fn, "a")) == NULL ) {
      fprintf(stderr, "\n error in DVL_writeToFile(%p,%s)"
              "\n unable to open file %s", dvl, fn, fn) ;
      rc = 0 ;
   } else {
      rc = DVL_writeForHumanEye(dvl, fp) ;
      fclose(fp) ;
   }
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   purpose -- to write an DVL object to a formatted file

   return value -- 1 if success, 0 otherwise

   created -- 95sep29, cca
   -----------------------------------------------------
*/
int
DVL_writeToFormattedFile ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int      count, ierr, j, jsize, nlist, rc ;
double   *dvec ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fp == NULL || (nlist = dvl->nlist) <= 0 ) {
   fprintf(stderr, "\n fatal error in DVL_writeToFormattedFile(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   exit(-1) ;
}
/*
   -------------------------------------
   write out the three scalar parameters
   -------------------------------------
*/
rc = fprintf(fp, "\n %d %d %d", dvl->type, dvl->nlist, dvl->tsize) ;
if ( rc < 0 ) {
   fprintf(stderr, "\n fatal error in DVL_writeToFormattedFile(%p,%p)"
           "\n rc = %d, return from first fprintf\n", dvl, fp, rc) ;
   return(0) ;
}
if ( dvl->nlist > 0 ) {
   IVfp80(fp, dvl->nlist, dvl->sizes, 80, &ierr) ;
   if ( ierr < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DVL_writeToFormattedFile(%p,%p)"
              "\n ierr = %d, return from sizes[] IVfp80\n", 
              dvl, fp, ierr) ;
      return(0) ;
   }
}
switch ( dvl->type ) {
case DVL_NOTYPE :
   break ;
case DVL_UNKNOWN :
case DVL_CHUNKED :
case DVL_SOLO    :
   for ( j = 0, count = 80 ; j < dvl->nlist ; j++ ) {
      DVL_listAndSize(dvl, j, &jsize, &dvec) ;
      if ( jsize > 0 ) {
         DVfprintf(fp, jsize, dvec) ;
      }
   }
   break ;
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   purpose -- to write an DVL object to a binary file

   return value -- 1 if success, 0 otherwise

   created -- 95sep29, cca
   --------------------------------------------------
*/
int
DVL_writeToBinaryFile ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int      j, jsize, nlist, rc ;
double   *dvec ;
int      itemp[3] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fp == NULL || (nlist = dvl->nlist) <= 0 ) {
   fprintf(stderr, "\n fatal error in DVL_writeToBinaryFile(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   exit(-1) ;
}
itemp[0] = dvl->type ;
itemp[1] = dvl->nlist ;
itemp[2] = dvl->tsize ;
rc = fwrite((void *) itemp, sizeof(int), 3, fp) ;
if ( rc != 3 ) {
   fprintf(stderr, "\n error in DVL_writeToBinaryFile(%p,%p)"
           "\n %d of %d scalar items written\n", dvl, fp, rc, 3) ;
   return(0) ;
}
rc = fwrite((void *) dvl->sizes, sizeof(int), dvl->nlist, fp) ;
if ( rc != dvl->nlist ) {
   fprintf(stderr, "\n error in DVL_writeToBinaryFile(%p,%p)"
           "\n dvl->sizes, %d of %d items written\n",
           dvl, fp, rc, dvl->nlist) ;
   return(0) ;
}
switch ( dvl->type ) {
case DVL_NOTYPE :
   break ;
case DVL_CHUNKED :
case DVL_SOLO    :
case DVL_UNKNOWN :
   for ( j = 0 ; j < dvl->nlist ; j++ ) {
      DVL_listAndSize(dvl, j, &jsize, &dvec) ;
      if ( jsize > 0 ) {
         rc = fwrite((void *) dvec, sizeof(double), jsize, fp) ;
         if ( rc != jsize ) {
            fprintf(stderr, "\n error in DVL_writeToBinaryFile(%p,%p)"
                    "\n list %d, %d of %d items written\n",
                    dvl, fp, j, rc, jsize) ;
            return(0) ;
         }
      }
   }
   break ;
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   purpose -- to write an DVL object for a human eye

   return value -- 1 if success, 0 otherwise

   created -- 95sep29, cca
   -------------------------------------------------
*/
int
DVL_writeForHumanEye ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int      j, size, rc ;
double   *dvec ;

if ( dvl == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DVL_writeForHumanEye(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   exit(-1) ;
}
if ( (rc = DVL_writeStats(dvl, fp)) == 0 ) {
   fprintf(stderr, "\n fatal error in DVL_writeForHumanEye(%p,%p)"
           "\n rc = %d, return from DVL_writeStats(%p,%p)\n",
           dvl, fp, rc, dvl, fp) ;
   return(0) ;
}
for ( j = 0 ; j < dvl->nlist ; j++ ) {
   DVL_listAndSize(dvl, j, &size, &dvec) ;
   if ( size > 0 ) {
      fprintf(fp, "\n %5d :", j) ;
      DVfprintf(fp, size, dvec) ;
   }
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- to write out the statistics for the DVL object

   return value -- 1 if success, 0 otherwise

   created -- 95sep29, cca
   ---------------------------------------------------------
*/
int
DVL_writeStats ( 
   DVL    *dvl, 
   FILE   *fp 
) {
int   nactive, rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( dvl == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DVL_writeStats(%p,%p)"
           "\n bad input\n", dvl, fp) ;
   exit(-1) ;
}
nactive = 0 ;
if ( dvl->nlist > 0 ) {
   nactive = IVsum(dvl->nlist, dvl->sizes) ;
}
rc = fprintf(fp, "\n DVL : integer vector list object :") ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, "\n type %d", dvl->type) ;
if ( rc < 0 ) { goto IO_error ; }
switch ( dvl->type ) {
case DVL_CHUNKED : rc = fprintf(fp, ", chunked storage") ; break ;
case DVL_SOLO    : rc = fprintf(fp, ", solo storage")    ; break ;
case DVL_UNKNOWN : rc = fprintf(fp, ", unknown storage") ; break ;
}
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, 
             "\n %d lists, %d maximum lists, %d tsize, %d total bytes",
             dvl->nlist, dvl->maxnlist, dvl->tsize, DVL_sizeOf(dvl)) ;
if ( rc < 0 ) { goto IO_error ; }
switch ( dvl->type ) {
case DVL_CHUNKED : {
   Dchunk   *chunk ;
   int      nalloc, nchunk ;

   nalloc = nchunk = 0 ;
   for ( chunk = dvl->chunk ; chunk != NULL ; chunk = chunk->next){
      nchunk++ ;
      nalloc += chunk->size ;
   }
   rc = fprintf(fp, "\n %d chunks, %d active entries, %d allocated",
                nchunk, nactive, nalloc) ;
   if ( rc < 0 ) { goto IO_error ; }
   if ( nalloc > 0 ) {
      rc = fprintf(fp, ", %.2f %% used", (100.*nactive)/nalloc) ;
      if ( rc < 0 ) { goto IO_error ; }
   }
   } break ;
case DVL_SOLO :
   rc = fprintf(fp, 
                "\n %d lists separately allocated, %d active entries",
                dvl->nlist, nactive) ;
  if ( rc < 0 ) { goto IO_error ; }
   break ;
}
return(1) ;

IO_error :
   fprintf(stderr, "\n fatal error in DVL_writeStats(%p,%p)"
           "\n rc = %d, return from fprintf\n", dvl, fp, rc) ;
   return(0) ;
}

/*--------------------------------------------------------------------*/
