/*  DVL.h  */

#include "../DV.h"
#include "../IV.h"
#include "../cfiles.h"

/*--------------------------------------------------------------------*/
/*
   The DVL object stores and manipulates Lists of Double Vectors,
   so the acronym for Double Vector Lists.

   The most common use of an DVL object is to contain
   the entries of a sparse matrix.

   The DVL object supports the following functionality.
      1. initialization, # of lists must be known.
      2. clearing data and free'ing the object
      3. set the size of a list and/or its entries
      4. un-setting a list or releasing its data 
      6. returning the maximum size of a list

   The number of lists is fixed on initialization, but the number
   of entries in any list or the total size of the lists need not
   be known at initialization. Storage for the lists is handled in
   one of three ways.
      1. chunked (DVL_CHUNKED = 1)
         A chunk of data is allocated by the object and lists point
         to data in the chunk. When the free space is not large
         enough to contain a new list, another chunk is allocated.
         The object keeps track of the chunks and free's all the
         storage when requested.
      2. solo (DVL_SOLO = 2)
         Each list is allocated separately using the DVinit() or
         DVinit2() function. When requested, each list is free'd
         using the DVfree() function.
      3. unknown (DVL_UNKNOWN = 3)
         Each list points to storage somewhere but the object does
         not keep track of any storage to be free'd. The application
         that gave rise to this storage mode has a subgraph "share"
         list storage with the larger graph that contains it. it made
         no sense to replicate the storage for a very large graph
         just to instantiate a subgraph. when the subgraph was free'd,
         it did not release any storage owned by the parent graph.

   created -- 96apr28, cca
*/
/*--------------------------------------------------------------------*/
/*
   --------------------------------------
   typedef for the DVL and Dchunk objects
   --------------------------------------
*/
typedef struct _DVL      DVL    ;
typedef struct _Dchunk   Dchunk ;
/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   type -- type of double vector list
      DVL_NOTYPE  -- no type
      DVL_CHUNKED -- list are in possibly many chunks of data
      DVL_SOLO    -- each list is created using DVinit()
                     and free'd using DVfree()
      DVL_UNKNOWN -- storage for lists unknown, 
                     it is the user's responsibility
   maxnlist -- maximum number of lists
   nlist    -- number of lists
   tsize    -- total size of the lists
   sizes    -- vector of list sizes, size nlist
   p_vec    -- vector of list pointers, size nlist
   incr     -- increment for chunks, number of entries allocated
               when a new chunk is necessary, used only when
               type = DVL_CHUNKED
   chunk    -- first Dchunk structure, 
               NULL unless type = DVL_CHUNKED
   -------------------------------------------------------------
*/
struct _DVL {
   int      type     ;
   int      maxnlist ;
   int      nlist    ;
   int      tsize    ;
   int      *sizes   ;
   double   **p_vec  ;
   int      incr     ;
   Dchunk   *chunk   ;
} ;
/*
   -------------------------------------------------------
   the Dchunk object is the data structure 
   that handles chunks of storage for mode DVL_CHUNKED

   size  -- number of entries in the chunk,
            also the dimension of the array base[]
   inuse -- the number of entries in use,
            size - inuse is the number of free entries
   base  -- base address for storage used for list entries
   next  -- pointer to the next Dchunk object
   -------------------------------------------------------
*/
struct _Dchunk {
   int      size  ;
   int      inuse ;
   double   *base ;
   Dchunk   *next ;
} ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------
   definitions for storage type
   ----------------------------
*/
#define DVL_NOTYPE     (-1)
#define DVL_CHUNKED      1
#define DVL_SOLO         2
#define DVL_UNKNOWN      3
#define DVL_INCR         0
/*
#define DVL_INCR      1024
#define DVL_INCR    262144
*/

/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 96apr28, cca
   -----------------------
*/
DVL *
DVL_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields
 
   created -- 96apr28, cca
   -----------------------
*/
void
DVL_setDefaultFields (
   DVL   *dvl
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 96apr28, cca
   --------------------------------------------------
*/
void
DVL_clearData (
   DVL   *dvl
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 96apr28, cca
   ------------------------------------------
*/
DVL *
DVL_free (
   DVL   *dvl
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------
   initialize given the type and maximum number of lists.
   used for type DVL_CHUNKED, DVL_SOLO or DVL_UNKNOWN
 
   created -- 95sep22, cca
   ------------------------------------------------------
*/
void
DVL_init1 ( 
   DVL   *dvl, 
   int   type, 
   int   maxnlist
) ;
/*
   ---------------------------------------------------------------
   initialize given the type, number of lists and their total size
   only used when type is DVL_CHUNKED.
 
   created -- 95sep22, cca
   ---------------------------------------------------------------
*/
void
DVL_init2 (
   DVL   *dvl,
   int   type,
   int   maxnlist,
   int   tsize
) ;
/*
   --------------------------------------
   initialize from a vector of list sizes.
   used with DVL_SOLO or DVL_CHUNKED.
 
   created -- 95sep22, cca
   --------------------------------------
*/
void
DVL_init3 (
   DVL   *dvl,
   int   type,
   int   maxnlist,
   int   sizes[]
) ;
/*
   ------------------------------------------------
   this method resizes the maximum number of lists,
   replacing the old sizes[] and p_vec[] vectors
   as necessary. the nlist value is NOT reset.
 
   created -- 96dec05, cca
   ------------------------------------------------
*/
void
DVL_setMaxnlist (
   DVL   *dvl,
   int   newmaxnlist
) ;
/*
   ---------------------------------------------
   this method resizes the number of lists,
   replacing the old sizes[] and p_vec[] vectors
   as necessary.
 
   created -- 96dec05, cca
   ---------------------------------------------
*/
void
DVL_setNlist (
   DVL   *dvl,
   int   newnlist
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in instance.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------
   return the storage type of the object
 
   created -- 96dec06, cca
   -------------------------------------
*/
int
DVL_type (
   DVL   *dvl
) ;
/*
   ----------------------------------
   return the maximum number of lists
 
   created -- 96dec06, cca
   ----------------------------------
*/
int
DVL_maxnlist (
   DVL   *dvl
) ;
/*
   --------------------------
   return the number of lists
 
   created -- 96dec06, cca
   --------------------------
*/
int
DVL_nlist (
   DVL   *dvl
) ;
/*
   ----------------------------------
   return the total size of the lists
 
   created -- 96dec06, cca
   ----------------------------------
*/
int
DVL_tsize (
   DVL   *dvl
) ;
/*
   ----------------------------
   return the storage increment
 
   created -- 96dec06, cca
   ----------------------------
*/
int
DVL_incr (
   DVL   *dvl
) ;
/*
   -------------------------
   set the storage increment
 
   created -- 96dec06, cca
   -------------------------
*/
void
DVL_setincr (
   DVL   *dvl,
   int   incr
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in listmanip.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------------
  fills size of and pointer to the entries of a list
 
   set *psize = size of list ilist
   set *pvec = base address of list ilist
 
   use as follows :
 
   DVL_listAndSize(dvl, ilist, &size, &vec) ;
   for ( i = 0 ; i < size ; i++ ) {
      do something with vec[i] ;
   }
 
   created -- 95sep22, cca
   ----------------------------------------------------------------
*/
void
DVL_listAndSize (
   DVL      *dvl,
   int      ilist,
   int      *psize,
   double   **pvec
) ;
/*
   ----------------------------------------------------
   returns a pointer to the first element in list ilist
 
   to be used as a general iterator, e.g.,
 
   for ( pd = DVL_firstInList(dvl, ilist) ;
         pd != NULL ;
         pd = DVL_nextInList(dvl, ilist, pd) ) {
      do something ;
   }
 
   created -- 95sep27, cca
   ----------------------------------------------------
*/
double *
DVL_firstInList (
   DVL   *dvl,
   int   ilist
) ;
/*
   ----------------------------------------------------
   returns a pointer to the next element in list ilist
 
   to be used as a general iterator, e.g.,
 
   for ( pd = DVL_firstInList(dvl, ilist) ;
         pd != NULL ;
         pd = DVL_nextInList(dvl, ilist, pd) ) {
      do something ;
   }
 
   created -- 95sep27, cca
   ----------------------------------------------------
*/
double *
DVL_nextInList (
   DVL      *dvl,
   int      ilist,
   double   *pd
) ;
/*
   -----------------------------------------------------------------
   purpose -- to set or reset a list.
 
   ilist -- list id to set or reset
   size  -- size of the list
      if the present size of list ilist is smaller than size,
         the old list is free'd (if dvl->type = DVL_SOLO)
         or lost (if dvl->type = DVL_CHUNKED)
         or un-set (if dvl->type = DVL_UNKNOWN)
         and new storage is allocated (for DVL_SOLO and DVL_CHUNKED)
   vec  -- list vector
      if dvl->type is DVL_UNKNOWN then
         if vec != NULL then
            we set the dvl list pointer to be vec
         endif
      else if vec != NULL
         we copy vec[] into dvl's storage for the list
      endif
 
   created   -- 95sep27, cca
   last mods -- 95oct06, cca
      type = DVL_UNKNOWN, p_vec[ilist] set to vec
      only when vec is not NULL
      bug fixed, dvl->sizes[ilist] = size ;
   -----------------------------------------------------------------
*/
void
DVL_setList (
   DVL      *dvl,
   int      ilist,
   int      size,
   double   vec[]
) ;
/*
   ----------------------------------------------------------------
  set a pointer to a list but don't allocate storage for the list.
   this method was needed when we form a subgraph with a boundary.
   lists for the interior vertices point into the parent graph,
   but lists for the boundary vertices must be allocated and owned.
  used only for type = DVL_CHUNKED. at some point in the future we
   should rethink the storage semantics for the DVL object.
 
   created -- 95nov11, cca
   ----------------------------------------------------------------
*/
void
DVL_setPointerToList (
   DVL      *dvl,
   int      ilist,
   int      size,
   double   vec[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------
   purpose -- to read in an DVL object from a file
 
   input --
 
      fn -- filename, must be *.dvlb or *.dvlf
 
   return value -- 1 if success, 0 if failure
 
   created -- 95sep29, cca
   -----------------------------------------------
*/
int
DVL_readFromFile ( 
   DVL    *dvl, 
   char   *fn 
) ;
/*
   ------------------------------------------------------
   purpose -- to read an DVL object from a formatted file
 
   return value -- 1 if success, 0 if failure
 
   created -- 95sep29, cca
   ------------------------------------------------------
*/
int
DVL_readFromFormattedFile (
   DVL    *dvl,
   FILE   *fp
) ;
/*
   ---------------------------------------------------
   purpose -- to read an DVL object from a binary file
 
   return value -- 1 if success, 0  if failure
 
   created -- 95sep29, cca
   ---------------------------------------------------
*/
int
DVL_readFromBinaryFile (
   DVL    *dvl,
   FILE   *fp
) ;
/*
   -------------------------------------------
   purpose -- to write an DVL object to a file
 
   input --
 
      fn -- filename
        *.dvlb -- binary
        *.dvlf -- formatted
        anything else -- for human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95sep29, cca
   -------------------------------------------
*/
int
DVL_writeToFile (
   DVL    *dvl,
   char   *fn
) ;
/*
   -----------------------------------------------------
   purpose -- to write an DVL object to a formatted file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95sep29, cca
   -----------------------------------------------------
*/
int
DVL_writeToFormattedFile (
   DVL    *dvl,
   FILE   *fp
) ;
/*
   --------------------------------------------------
   purpose -- to write an DVL object to a binary file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95sep29, cca
   --------------------------------------------------
*/
int
DVL_writeToBinaryFile (
   DVL    *dvl,
   FILE   *fp
) ;
/*
   -------------------------------------------------
   purpose -- to write an DVL object for a human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95sep29, cca
   -------------------------------------------------
*/
int
DVL_writeForHumanEye (
   DVL    *dvl,
   FILE   *fp
) ;
/*
   ---------------------------------------------------------
   purpose -- to write out the statistics for the DVL object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 95sep29, cca
   ---------------------------------------------------------
*/
int
DVL_writeStats (
   DVL    *dvl,
   FILE   *fp
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in profile.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   to fill xDV and yDV with a log10 profile of the magnitudes of
   the entries in the DVL object. tausmall and tau big provide
   cutoffs within which to examine the entries. pnsmall, pnbig
   and pntotal are address to hold the number of entries smaller
   than tausmall, larger than taubig and total entries, respectively.
 
   created -- 96sep09, cca
   ------------------------------------------------------------------
*/
void
DVL_log10profile (
   DVL      *dvl,
   int      npts,
   DV       *xDV,
   DV       *yDV,
   double   tausmall,
   double   taubig,
   int      *pnsmall,
   int      *pnbig,
   int      *pntotal
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------
   return the number of bytes taken by the object
 
   created -- 96apr28, cca
   ----------------------------------------------
*/
int
DVL_sizeOf (
   DVL   *dvl
) ;
/*
   ----------------------------
   return the maximum list size
 
   created -- 96apr28, cca
   ----------------------------
*/
int
DVL_maxListSize (
   DVL   *dvl
) ;
/*
   ----------------------------------------------------------------
   DVL object dvl1 absorbs the lists and data from DVL object dvl2.
   the lists in dvl2 are mapped into lists in dvl1 using the mapIV
   IV object.
 
   created -- 96dec06, cca
   ----------------------------------------------------------------
*/
void
DVL_absorbDVL (
   DVL   *dvl1,
   DVL   *dvl2,
   IV    *mapIV
) ;
/*--------------------------------------------------------------------*/
