/*  QRfactor.c  */

#include "../DStairMtx.h"

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   purpose ---
      given a matrix A, convert to an upper triangular or
      upper trapezoidal R using householder transformations.
      The transformations and their coeffecients are not saved.

      this code follows closely page 210 from golub and van loan,
      "matrix computations", 3rd edition.
 
      note: presently the matrix must be row major

   created -- 97mar29, dkw
   --------------------------------------------------------------
*/
void
DStairMtx_QRfactor (
  DA2    *mtx,
  int    *lastrowindex,
  double *facops
) {
double   beta, fac, mu, sigma, v0 ;
double   *colj, *colk ;
int      i, inc1, irow, jcol, kcol, lhhv, ncol, nrow, nstep, width ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || lastrowindex == NULL ) {
   fprintf(stderr, "\n fatal error in DStairmtx_QRfactor(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if (  (nrow = DA2_nrow(mtx)) <= 0
   || (ncol = DA2_ncol(mtx)) <= 0
   || DA2_inc2(mtx) != 1
   || DA2_entries(mtx) == NULL ) {
   fprintf(stderr, "\n fatal error in DStairmtx_QRfactor(%p)"
           "\n bad object\n", mtx) ;
   DA2_writeStats(mtx, stderr) ;
   exit(-1) ;
}
/*
   ----------------------------------------------
   determine the number of steps = min(nrow,ncol)
   ----------------------------------------------
*/
*facops   = 0 ;
inc1  = DA2_inc1(mtx) ;
nstep = (ncol < nrow) ? ncol : nrow - 1 ;
/*
   -------------------
   loop over the steps
   -------------------
*/
for ( jcol = 0 ; jcol < nstep ; jcol++ ) {
/*
   -----------------------------------------
   compute the pointer to the present column
   -----------------------------------------
*/
   colj  = DA2_column(mtx, jcol) ;
/*
   -----------------------------------
   compute ||colj(jcol+1:nrow-1)||_2^2
   -----------------------------------
*/
   sigma = 0.0 ;
   nrow  = lastrowindex[jcol] + 1 ;
   lhhv  = nrow - jcol ;
   width = ncol - jcol - 1 ;
   *facops += 4*lhhv*width + 6*lhhv - width ;
   for ( i=jcol+1,irow = (jcol + 1)*inc1 ; i < nrow ; i++, irow+=inc1 ) {
      sigma += colj[irow]*colj[irow] ;
   }
   if ( sigma != 0.0 ) {
/*
      --------------------------------------------
      there are zero entries below the diagonal
      compute mu = compute ||colj(jcol:nrow-1)||_2
      --------------------------------------------
*/
      mu = sqrt(sigma + colj[jcol*inc1]*colj[jcol*inc1]) ;
/*
      -----------------------------------------------
      set the first element of the householder vector
      -----------------------------------------------
*/
      if ( colj[jcol*inc1] <= 0 ) {
         v0 = colj[jcol*inc1] - mu ;
      } else {
         v0 = -sigma/(colj[jcol*inc1] + mu) ;
      }
/*
      -------------------
      get the coefficient
      -------------------
*/
      beta = 2*v0*v0/(sigma + v0*v0) ;
/*
      ---------------------------------------------
      set the first entry of the transformed column
      ---------------------------------------------
*/
      colj[jcol*inc1] = mu ;
/*
      -----------------------------------------------------------
      scale the householder vector so that its first entry is 1.0
      -----------------------------------------------------------
*/
      for ( i=jcol+1,irow = (jcol + 1)*inc1 ; i < nrow ; i++, irow+=inc1 ) {
         colj[irow] = colj[irow] / v0 ;
      }
/*
      ------------------------------------------------------------
      loop over the following columns and apply the transformation
      ------------------------------------------------------------
*/
      for ( kcol = jcol + 1 ; kcol < ncol ; kcol++ ) {
         colk = DA2_column(mtx, kcol) ;
/*
         ---------------------------------------------
         compute the dot product between colj and colk
         ---------------------------------------------
*/
         for ( i=jcol+1,fac=colk[jcol*inc1],irow = (jcol + 1)*inc1 ;
               i < nrow ;
               i++, irow+=inc1 ) {
            fac += colk[irow]*colj[irow] ;
         }
/*
         --------------------------
         compute the scaling factor
         --------------------------
*/
         fac = -beta*fac ;
/*
         ---------------------------------------
         apply the transformation to column colk
         ---------------------------------------
*/
         colk[jcol*inc1] += fac ;
         for ( i=jcol+1,irow = (jcol + 1)*inc1 ; i < nrow ; i++, irow+=inc1 ) {
            colk[irow] += fac * colj[irow] ;
         }
      }
   } else {
      beta = 0.0 ;
   }
}
return ; }
/*--------------------------------------------------------------------*/
