/*  setup.c  */

#include "../DPencil.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   initialize the matrix pencil A + sigma*B

   myid -- id of process, used in MPI implementation
      if myid = 0 then 
         the pencil is loaded with the matrices read from the files
      else
         the pencil is loaded with the empty matrices 
      endif
   inpmtxAfile  -- filename for A
   sigma        -- scaling factor
   inpmtxBfile  -- filename for B
   symmetryflag -- symmetry flag, 
      if zero, drop entries in lower triangle 
   randomflag   -- random flag, 
      if 1 then fill with random numbers
   drand        -- random number generator
   msglvl       -- message level
   msgFile      -- message file

   return value -- pointer to a DPencil object

   created -- 97jul18, cca
   ----------------------------------------------------------------
*/
DPencil *
DPencil_setup (
   int        myid,
   char       *inpmtxAfile,
   double     sigma,
   char       *inpmtxBfile,
   int        symmetryflag,
   int        randomflag,
   Drand      *drand,
   int        msglvl,
   FILE       *msgFile
) {
DInpMtx   *inpmtxA, *inpmtxB ;
double    t1, t2 ;
DPencil   *pencil ;
int       rc ;

if ( strcmp(inpmtxAfile, "none") != 0 ) {
/*
   --------------------------
   read in the DInpMtx object
   --------------------------
*/
   inpmtxA = DInpMtx_new() ;
   if ( myid == 0 ) {
      MARKTIME(t1) ;
      rc = DInpMtx_readFromFile(inpmtxA, inpmtxAfile) ;
      MARKTIME(t2) ;
      fprintf(msgFile,"\n CPU %8.3f : read in inpmtxA", t2 - t1) ;
      if ( rc != 1 ) {
         fprintf(msgFile, 
                 "\n return value %d from DInpMtx_readFromFile(%p,%s)",
                 rc, inpmtxA, inpmtxAfile) ;
         exit(-1) ;
      }
      if ( symmetryflag == 0 ) {
/*
         ----------------------------------------------------
         symmetric matrix, drop entries in the lower triangle
         ----------------------------------------------------
*/
         MARKTIME(t1) ;
         DInpMtx_dropLowerTriangle(inpmtxA) ;
         DInpMtx_changeStorageMode(inpmtxA, 1) ;
/*
         ------------------------------------
         sort and compress the matrix entries
         ------------------------------------
*/
         DInpMtx_sortAndCompress(inpmtxA) ;
         MARKTIME(t2) ;
         fprintf(msgFile,"\n CPU %8.3f : initialize inpmtxA", t2 - t1) ;
      }
      if ( randomflag == 1 ) {
/*
         -----------------------------------
         fill the matrix with random numbers
         -----------------------------------
*/
         MARKTIME(t1) ;
         Drand_fillDvector(drand, inpmtxA->nent, 
                           DV_entries(&inpmtxA->dvecDV)) ;
         MARKTIME(t2) ;
         fprintf(msgFile, 
          "\n CPU %8.3f : fill inpmtxA with random numbers", t2 - t1) ;
      } else if ( randomflag == -1 ) {
/*
         -----------------------------
         double one entry in magnitude
         -----------------------------
*/
         double   *dvec = DV_entries(&inpmtxA->dvecDV) ;
         dvec[0] *= 2 ;
      }
   }
/*
   --------------------------------------
   change the coordinate type to chevrons
   and the storage mode to sorted triples
   --------------------------------------
*/
   MARKTIME(t1) ;
   DInpMtx_changeCoordType(inpmtxA, 3) ;
   DInpMtx_changeStorageMode(inpmtxA, 2) ;
   MARKTIME(t2) ;
   fprintf(msgFile, 
           "\n CPU %8.3f : change inpmtxA to chevrons", t2 - t1) ;
} else {
   inpmtxA = NULL ;
}
if ( strcmp(inpmtxBfile, "none") != 0 ) {
/*
   --------------------------
   read in the DInpMtx object
   --------------------------
*/
   inpmtxB = DInpMtx_new() ;
   if ( myid == 0 ) {
      MARKTIME(t1) ;
      rc = DInpMtx_readFromFile(inpmtxB, inpmtxBfile) ;
      MARKTIME(t2) ;
      fprintf(msgFile,"\n CPU %8.3f : read in inpmtxB", t2 - t1) ;
      if ( rc != 1 ) {
         fprintf(msgFile, 
                 "\n return value %d from DInpMtx_readFromFile(%p,%s)",
                 rc, inpmtxB, inpmtxBfile) ;
         exit(-1) ;
      }
      if ( symmetryflag == 0 ) {
/*
         ----------------------------------------------------
         symmetric matrix, drop entries in the lower triangle
         ----------------------------------------------------
*/
         MARKTIME(t1) ;
         DInpMtx_dropLowerTriangle(inpmtxB) ;
         DInpMtx_changeStorageMode(inpmtxB, 1) ;
/*
         ------------------------------------
         sort and compress the matrix entries
         ------------------------------------
*/
         DInpMtx_sortAndCompress(inpmtxB) ;
         MARKTIME(t2) ;
         fprintf(msgFile,"\n CPU %8.3f : initialize inpmtxB", t2 - t1) ;
      }
      if ( randomflag == 1 ) {
/*
         -----------------------------------
         fill the matrix with random numbers
         -----------------------------------
*/
         MARKTIME(t1) ;
         Drand_fillDvector(drand, inpmtxB->nent, 
                           DV_entries(&inpmtxB->dvecDV)) ;
         MARKTIME(t2) ;
         fprintf(msgFile, 
          "\n CPU %8.3f : fill inpmtxB with random numbers", t2 - t1) ;
      } else if ( randomflag == -1 ) {
/*
         -----------------------------
         double one entry in magnitude
         -----------------------------
*/
         double   *dvec = DV_entries(&inpmtxB->dvecDV) ;
         dvec[0] *= 2 ;
      }
   }
/*
   --------------------------------------
   change the coordinate type to chevrons
   and the storage mode to sorted triples
   --------------------------------------
*/
   MARKTIME(t1) ;
   DInpMtx_changeCoordType(inpmtxB, 3) ;
   DInpMtx_changeStorageMode(inpmtxB, 2) ;
   MARKTIME(t2) ;
   fprintf(msgFile, 
           "\n CPU %8.3f : change inpmtxB to chevrons", t2 - t1) ;
} else {
   inpmtxB = NULL ;
}
/*
   -----------------------------
   initialize the DPencil object
   -----------------------------
*/
pencil = DPencil_new() ;
DPencil_init(pencil, inpmtxA, sigma, inpmtxB) ;

return(pencil) ; }

/*--------------------------------------------------------------------*/
