/*  DPencil.h  */

#include "../DInpMtx.h"
#include "../Drand.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   this object stores a matrix pencil
      A + sigma * B

   created -- 97jul13, cca
   ------------------------------------------------------------------
*/
typedef struct _DPencil   DPencil ;
struct _DPencil {
   DInpMtx   *inpmtxA ;
   DInpMtx   *inpmtxB ;
   double    sigma    ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97jul13, cca
   -----------------------
*/
DPencil *
DPencil_new (
   void
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97jul13, cca
   -----------------------
*/
void
DPencil_setDefaultFields (
   DPencil   *pencil
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97jul13, cca
   --------------------------------------------------
*/
void
DPencil_clearData (
   DPencil   *pencil
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97jul13, cca
   ------------------------------------------
*/
DPencil *
DPencil_free (
   DPencil   *pencil
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   initialize the object
 
   created -- 97jul13, cca
   -----------------------
*/
void
DPencil_init (
  DPencil   *pencil,
  DInpMtx   *inpmtx,
  double    sigma,
  DInpMtx   *inpmtxB
) ;
/*
   --------------------------
   change the coordinate type
 
   created -- 97jul18, cca
   --------------------------
*/
void
DPencil_changeCoordType (
   DPencil   *pencil,
   int       newType
) ;
/*
   -----------------------
   change the storage mode
 
   created -- 97jul18, cca
   -----------------------
*/
void
DPencil_changeStorageMode (
   DPencil   *pencil,
   int       newMode
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in permute.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------
   permute the matrix pencil
 
   created -- 97jul18, cca
   -------------------------
*/
void
DPencil_permute (
   DPencil   *pencil,
   IV        *rowOldToNewIV,
   IV        *colOldToNewIV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in mvm.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------
   compute y = (A + sigma*B)x
 
   created -- 97jul18, cca
   --------------------------
*/
void
DPencil_mvm (
   DPencil   *pencil,
   DV        *yDV,
   DV        *xDV
) ;
/*
   ---------------------------
   compute y = (A + sigma*B)x
   where A and B are symmetric
 
   created -- 97jul18, cca
   ---------------------------
*/
void
DPencil_symmetricMVM (
   DPencil   *pencil,
   DV        *yDV,
   DV        *xDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in setup.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------------
   initialize the matrix pencil A + sigma*B

   myid -- id of process, used in MPI implementation
      if myid = 0 then 
         the pencil is loaded with the matrices read from the files
      else
         the pencil is loaded with the empty matrices 
      endif
   inpmtxAfile  -- filename for A
   sigma        -- scaling factor
   inpmtxBfile  -- filename for B
   symmetryflag -- symmetry flag, 
      if zero, drop entries in lower triangle 
   randomflag   -- random flag, 
      if 1 then fill with random numbers
   drand        -- random number generator
   msglvl       -- message level
   msgFile      -- message file

   return value -- pointer to a DPencil object

   created -- 97jul18, cca
   ----------------------------------------------------------------
*/
DPencil *
DPencil_setup (
   int        myid,
   char       *inpmtxAfile,
   double     sigma,
   char       *inpmtxBfile,
   int        symmetryflag,
   int        randomflag,
   Drand      *drand,
   int        msglvl,
   FILE       *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------
   sort and compress the pencil's entries
 
   created -- 97jul18, cca
   --------------------------------------
*/
void
DPencil_sortAndCompress (
   DPencil   *pencil 
) ;
/*
   ------------------------------
   convert the storage to vectors
 
   created -- 97jul18, cca
   ------------------------------
*/
void
DPencil_convertToVectors (
   DPencil   *pencil 
) ;
/*
   ----------------------------------------------
   map entries to the lower triangle,
   used after a permutation of a symmetric matrix
 
   created -- 97jul18, cca
   ----------------------------------------------
*/
void
DPencil_mapToLowerTriangle (
   DPencil   *pencil 
) ;
/*
   ----------------------------------------------
   map entries to the upper triangle,
   used after a permutation of a symmetric matrix
 
   created -- 97jul18, cca
   ----------------------------------------------
*/
void
DPencil_mapToUpperTriangle (
   DPencil   *pencil
) ;
/*
   -------------------------------------------------------------
   purpose -- to return the full, symmetric adjacency IVL object
              for the graph of (A + B) + sigma * (A + B)^T
 
   created -- 97nov05, cca
   -------------------------------------------------------------
*/
IVL *
DPencil_fullAdjacency (
   DPencil  *pencil
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------------------
   purpose -- to read in a DInpMtx object from a file
 
   input --
 
      fn -- filename, must be *.inpmtxb or *.inpmtxf
 
   return value -- 1 if success, 0 if failure
 
   created -- 97jul18, cca
   --------------------------------------------------
*/
int
DPencil_readFromFiles (
   DPencil   *pencil,
   char      *inpmtxAfileName,
   char      *inpmtxBfileName
) ;
/*
   ----------------------------------------------------
   purpose -- to write a DPencil object for a human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jul18, cca
   ----------------------------------------------------
*/
int
DPencil_writeForHumanEye (
   DPencil   *pencil,
   FILE      *fp
) ;
/*
   -------------------------------------------------------------
   purpose -- to write out the statistics for the DPencil object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jul18, cca
   -------------------------------------------------------------
*/
int
DPencil_writeStats ( 
   DPencil   *pencil, 
   FILE      *fp 
) ;
/*--------------------------------------------------------------------*/
