/*  setup.c  */

#include "../DLinSystem.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- to set up a linear system (A + sigma*B) solmtx = rhsmtx

   myid         -- id of process, used in MPI applications
   neqns        -- number of equations
   nrhs         -- number of right hand sides
   inpmtxAfn    -- file name for matrix A
   sigma        -- shift value
   inpmtxBfn    -- file name for matrix B
   symmetryflag -- flag to denote symmetry of the pencil
      symmetryflag == 0 --> pencil is symmetric
      symmetryflag != 0 --> pencil is not symmetric
   randomflag -- if not zero, fill pencil entries with random numbers
   drand      -- random number generator
   msglvl     -- message level
   msgFile    -- message file

   created -- 97jul28, cca
   ------------------------------------------------------------------
*/
void
DLinSystem_setup (
   DLinSystem   *linsys,
   int          myid,
   int          neqns,
   int          nrhs,
   char         *inpmtxAfn,
   double       sigma,
   char         *inpmtxBfn,
   int          symmetryflag,
   int          randomflag,
   Drand        *drand,
   int          msglvl,
   FILE         *msgFile
) {
DDenseMtx   *rhsmtx, *solmtx ;
DPencil     *pencil ;
/*
   -------------------------
   read in the matrix pencil
   -------------------------
*/
pencil = DPencil_setup(myid, inpmtxAfn, sigma, inpmtxBfn, symmetryflag,
                       randomflag, drand, msglvl, msgFile) ;
solmtx = DDenseMtx_new() ;
rhsmtx = DDenseMtx_new() ;
if ( 1 ) {
/*
   -----------------------------------------------
   construct the solution, compute right hand side
   -----------------------------------------------
*/
   if ( myid == 0 ) {
      double   t1, t2 ;
      double   *b, *x ;
      DV       *bDV, *xDV ;
      int      irhs, ncol, nrow ;
      int      *colind, *rowind ;
/*
      -----------------------------------------
      initialize the solmtx and rhsmtx matrices
      -----------------------------------------
*/
      MARKTIME(t1) ;
      DDenseMtx_init(solmtx, 0, -1, neqns, nrhs, 1, neqns) ;
      DDenseMtx_rowIndices(solmtx, &nrow, &rowind) ;
      IVramp(nrow, rowind, 0, 1) ;
      DDenseMtx_columnIndices(solmtx, &ncol, &colind) ;
      IVramp(ncol, colind, 0, 1) ;
      DDenseMtx_init(rhsmtx, 1, -1, neqns, nrhs, 1, neqns) ;
      DDenseMtx_rowIndices(rhsmtx, &nrow, &rowind) ;
      IVramp(nrow, rowind, 0, 1) ;
      DDenseMtx_columnIndices(rhsmtx, &ncol, &colind) ;
      IVramp(ncol, colind, 0, 1) ;
      MARKTIME(t2) ;
      fprintf(msgFile, 
              "\n CPU %8.3f : initialize solmtx and rhsmtx ", t2 - t1) ;
/*
      -------------------------------
      fill solmtx with random numbers
      -------------------------------
*/
      MARKTIME(t1) ;
      DVfill(neqns*nrhs, DDenseMtx_entries(solmtx), 1.0) ;
      Drand_fillDvector(drand, neqns*nrhs, DDenseMtx_entries(solmtx)) ;
      if ( msglvl > 3 ) {
         fprintf(msgFile, 
                 "\n\n solmtx matrix filled with random numbers") ;
         DDenseMtx_writeForHumanEye(solmtx, msgFile) ;
         fflush(msgFile) ;
      }
      MARKTIME(t2) ;
      fprintf(msgFile,
              "\n CPU %8.3f : fill solmtx with random numbers ", 
              t2 - t1) ;
/*
      -----------
      zero rhsmtx
      -----------
*/
      DVfill(neqns*nrhs, DDenseMtx_entries(rhsmtx), 0.0) ;
/*
      ----------------------------------
      compute the matrix vector multiply
      ----------------------------------
*/
      MARKTIME(t1) ;
      bDV = DV_new() ;
      xDV = DV_new() ;
      x   = DDenseMtx_entries(solmtx) ;
      b   = DDenseMtx_entries(rhsmtx) ;
      for ( irhs = 0 ; irhs < nrhs ; irhs++ ) {
         DV_init(bDV, neqns, b) ;
         DV_init(xDV, neqns, x) ;
         if ( symmetryflag == 0 ) {
            DPencil_symmetricMVM(pencil, bDV, xDV) ;
         } else {
            DPencil_mvm(pencil, bDV, xDV) ;
         }
         x += neqns ;
         b += neqns ;
      }
      MARKTIME(t2) ;
      fprintf(msgFile, "\n CPU %8.3f : compute MVM", t2 - t1) ;
      if ( msglvl > 3 ) {
         fprintf(msgFile, "\n\n rhsmtx matrix ") ;
         DDenseMtx_writeForHumanEye(rhsmtx, msgFile) ;
         fflush(msgFile) ;
      }
      DV_free(bDV) ;
      DV_free(xDV) ;
   } else {
      DDenseMtx_init(solmtx, -1, -1, 0, nrhs, 1, 0) ;
      DDenseMtx_init(rhsmtx, -1, -1, 0, nrhs, 1, 0) ;
   }
}
/*
   ---------------------
   initialize the object
   ---------------------
*/
DLinSystem_init(linsys, neqns, nrhs, symmetryflag, 
                pencil, solmtx, rhsmtx) ;

return ; }

/*--------------------------------------------------------------------*/
