/*  DLinSystem.h  */

#include "../DPencil.h"
#include "../DDenseMtx.h"
#include "../Drand.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   this object stores a linear system
      (A + sigma * B) X = Y

   created -- 97jul28, cca
   ------------------------------------------------------------------
*/
typedef struct _DLinSystem   DLinSystem ;
struct _DLinSystem {
   int         neqns   ;
   int         nrhs    ;
   int         symflag ;
   DPencil     *pencil ;
   DDenseMtx   *solmtx ;
   DDenseMtx   *rhsmtx ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97jul28, cca
   -----------------------
*/
DLinSystem *
DLinSystem_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97jul28, cca
   -----------------------
*/
void
DLinSystem_setDefaultFields (
   DLinSystem   *linsys
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97jul28, cca
   --------------------------------------------------
*/
void
DLinSystem_clearData (
   DLinSystem   *linsys
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97jul28, cca
   ------------------------------------------
*/
DLinSystem *
DLinSystem_free (
   DLinSystem   *linsys
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   initialize the object
 
   created -- 97jul28, cca
   -----------------------
*/
void
DLinSystem_init (
  DLinSystem   *linsys,
  int          neqns,
  int          nrhs,
  int          symflag,
  DPencil      *pencil,
  DDenseMtx    *solmtx,
  DDenseMtx    *rhsmtx
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in setup.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- to set up a linear system (A + sigma*B) solmtx = rhsmtx
 
   myid         -- id of process, used in MPI applications
   neqns        -- number of equations
   nrhs         -- number of right hand sides
   inpmtxAfn    -- file name for matrix A
   sigma        -- shift value
   inpmtxBfn    -- file name for matrix B
   symmetryflag -- flag to denote symmetry of the pencil
      symmetryflag == 0 --> pencil is symmetric
      symmetryflag != 0 --> pencil is not symmetric
   randomflag -- if not zero, fill pencil entries with random numbers
   drand      -- random number generator
   msglvl     -- message level
   msgFile    -- message file
 
   created -- 97jul28, cca
   ------------------------------------------------------------------
*/
void
DLinSystem_setup (
   DLinSystem   *linsys,
   int          myid,
   int          neqns,
   int          nrhs,
   char         *inpmtxAfn,
   double       sigma,
   char         *inpmtxBfn,
   int          symmetryflag,
   int          randomflag,
   Drand        *drand,
   int          msglvl,
   FILE         *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in permute.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------
   permute the linear system
   
   created -- 97jul28, cca
   -------------------------
*/
void
DLinSystem_permute (
   DLinSystem   *linsys,
   IV           *oldToNewIV,
   int          msglvl,
   FILE         *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------
   purpose -- to write a DLinSystem object for a human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jul29, cca
   -------------------------------------------------------
*/
int
DLinSystem_writeForHumanEye ( 
   DLinSystem   *linsys, 
   FILE         *fp 
) ;
/*
   ----------------------------------------------------------------
 purpose -- to write out the statistics for the DLinSystem object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jul29, cca
   ----------------------------------------------------------------
*/
int
DLinSystem_writeStats (
   DLinSystem   *linsys,
   FILE         *fp
) ;
