/*  mvm.c  */

#include "../DInpMtx.h"

/*--------------------------------------------------------------------*/
/*
   --------------------------------
   compute a matrix-vector multiply

   created -- 96jul06, cca
   --------------------------------
*/
void
DInpMtx_mvm (
   DInpMtx   *inpmtx,
   DV        *yDV,
   double    alpha,
   DV        *xDV
) {
double   *dvec, *x, *y ;
int      *ivec1, *ivec2 ;
int      chev, col, ii, nent, off, row ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || yDV == NULL || xDV == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_mvm(%p,%p,%f,%p)"
           "\n bad input\n", inpmtx, yDV, alpha, xDV) ;
   exit(-1) ;
}
if ( inpmtx->coordType < 1 || 3 < inpmtx->coordType 
   || inpmtx->storageMode < 1 || 3 < inpmtx->storageMode 
   || inpmtx->inputMode != 2 ) {
   fprintf(stderr, "\n fatal error in DInpMtx_mvm(%p,%p,%f,%p)"
           "\n coordType = %d, storageMode = %d, inputMode = %d\n", 
           inpmtx, yDV, alpha, xDV, inpmtx->coordType,
           inpmtx->storageMode, inpmtx->inputMode) ;
   exit(-1) ;
}
x = DV_entries(xDV) ;
y = DV_entries(yDV) ;
if ( x == NULL || y == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_mvm(%p,%p,%f,%p)"
           "\n x entries = %p, y entries = %p\n", 
           inpmtx, yDV, alpha, xDV, x, y) ;
   exit(-1) ;
}
/*
   --------------------------------
   data is stored as triples
   (deal with vector storage later)
   --------------------------------
*/
ivec1 = DInpMtx_ivec1(inpmtx) ;
ivec2 = DInpMtx_ivec2(inpmtx) ;
dvec  = DInpMtx_dvec(inpmtx) ;
if ( ivec1 == NULL || ivec2 == NULL || dvec == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_mvm(%p,%p,%f,%p)"
           "\n bad input\n", inpmtx, yDV, alpha, xDV) ;
   exit(-1) ;
}
nent  = inpmtx->nent ;
switch ( inpmtx->coordType ) {
case 1 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         row = ivec1[ii] ;
         col = ivec2[ii] ;
         y[row] += dvec[ii] * x[col] ;
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         row = ivec1[ii] ;
         col = ivec2[ii] ;
         y[row] += alpha * dvec[ii] * x[col] ;
      }
   }
   break ;
case 2 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         col = ivec1[ii] ;
         row = ivec2[ii] ;
         y[row] += dvec[ii] * x[col] ;
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         col = ivec1[ii] ;
         row = ivec2[ii] ;
         y[row] += alpha * dvec[ii] * x[col] ;
      }
   }
   break ;
case 3 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         chev = ivec1[ii] ;
         off  = ivec2[ii] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         y[row] += dvec[ii] * x[col] ;
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         chev = ivec1[ii] ;
         off  = ivec2[ii] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         y[row] += alpha * dvec[ii] * x[col] ;
      }
   }
   break ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------
   compute a symmetric matrix-vector multiply

   created -- 97jan11, cca
   ------------------------------------------
*/
void
DInpMtx_symmetricMVM (
   DInpMtx   *inpmtx,
   DV        *yDV,
   double    alpha,
   DV        *xDV
) {
double   *dvec, *x, *y ;
int      *ivec1, *ivec2 ;
int      chev, col, ii, nent, off, row ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || yDV == NULL || xDV == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_symmetricMVM(%p,%p,%f,%p)"
           "\n bad input\n", inpmtx, yDV, alpha, xDV) ;
   exit(-1) ;
}
if ( inpmtx->coordType < 1 || 3 < inpmtx->coordType 
   || inpmtx->storageMode < 1 || 3 < inpmtx->storageMode 
   || inpmtx->inputMode != 2 ) {
   fprintf(stderr, "\n fatal error in DInpMtx_symmetricMVM(%p,%p,%f,%p)"
           "\n coordType = %d, storageMode = %d, inputMode = %d\n", 
           inpmtx, yDV, alpha, xDV, inpmtx->coordType,
           inpmtx->storageMode, inpmtx->inputMode) ;
   exit(-1) ;
}
x = DV_entries(xDV) ;
y = DV_entries(yDV) ;
if ( x == NULL || y == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_symmetricMVM(%p,%p,%f,%p)"
           "\n x entries = %p, y entries = %p\n", 
           inpmtx, yDV, alpha, xDV, x, y) ;
   exit(-1) ;
}
/*
   --------------------------------
   data is stored as triples
   (deal with vector storage later)
   --------------------------------
*/
ivec1 = DInpMtx_ivec1(inpmtx) ;
ivec2 = DInpMtx_ivec2(inpmtx) ;
dvec  = DInpMtx_dvec(inpmtx) ;
if ( ivec1 == NULL || ivec2 == NULL || dvec == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_symmetricMVM(%p,%p,%f,%p)"
           "\n bad input\n", inpmtx, yDV, alpha, xDV) ;
   exit(-1) ;
}
nent  = inpmtx->nent ;
switch ( inpmtx->coordType ) {
case 1 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         row = ivec1[ii] ;
         col = ivec2[ii] ;
         y[row] += dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += dvec[ii] * x[row] ;
         }
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         row = ivec1[ii] ;
         col = ivec2[ii] ;
         y[row] += alpha * dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += alpha * dvec[ii] * x[row] ;
         }
      }
   }
   break ;
case 2 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         col = ivec1[ii] ;
         row = ivec2[ii] ;
         y[row] += dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += dvec[ii] * x[row] ;
         }
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         col = ivec1[ii] ;
         row = ivec2[ii] ;
         y[row] += alpha * dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += alpha * dvec[ii] * x[row] ;
         }
      }
   }
   break ;
case 3 :
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         chev = ivec1[ii] ;
         off  = ivec2[ii] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         y[row] += dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += dvec[ii] * x[row] ;
         }
      }
   } else {
      for ( ii = 0 ; ii < nent ; ii++ ) {
         chev = ivec1[ii] ;
         off  = ivec2[ii] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         y[row] += alpha * dvec[ii] * x[col] ;
         if ( row != col ) {
            y[col] += alpha * dvec[ii] * x[row] ;
         }
      }
   }
   break ;
}
return ; }

/*--------------------------------------------------------------------*/
