/*  input.c  */

#include "../DInpMtx.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   input a single entry in the matrix

   created -- 96jul05, cca
   ----------------------------------
*/
void
DInpMtx_inputEntry (
   DInpMtx   *inpmtx,
   int       row,
   int       col,
   double    entry
) {
double   *dvec ;
int      jj, newmaxnent ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if ( inpmtx == NULL || row < 0 || col < 0 ) {
   fprintf(stderr, "\n fatal error in DInpMtx_inputEntry(%p,%d,%d,%e)"
           "\n bad input\n", inpmtx, row, col, entry) ;
   exit(-1) ;
}
if ( inpmtx->coordType < 1 || 3 < inpmtx->coordType ) {
   fprintf(stderr, "\n fatal error in DInpMtx_inputEntry(%p,%d,%d,%e)"
           "\n bad coordType = %d\n", inpmtx, row, col, entry,
           inpmtx->coordType) ;
   exit(-1) ;
}
if ( inpmtx->nent == inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent == inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->nent + 1 ) {
      newmaxnent = inpmtx->nent + 1 ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_inputEntry(%p,%d,%d,%e)"
           "\n ivec1 = %p, ivec2 = %p\n", 
           inpmtx, row, col, entry, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_inputEntry(%p,%d,%d,%e)"
              "\n dvec = %p\n", inpmtx, row, col, entry, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   ----------------
   input the triple
   ----------------
*/
jj = inpmtx->nent ;
if ( inpmtx->coordType == 1 ) {
   ivec1[jj] = row ;
   ivec2[jj] = col ;
} else if ( inpmtx->coordType == 2 ) {
   ivec1[jj] = col ;
   ivec2[jj] = row ;
} else if ( inpmtx->coordType == 3 ) {
   if ( row <= col ) {
      ivec1[jj] = row ;
      ivec2[jj] = col - row ;
   } else {
      ivec1[jj] = col ;
      ivec2[jj] = col - row ;
   }
}
if ( inpmtx->inputMode == 2 ) {
   dvec[jj] = entry  ;
}
inpmtx->nent++ ;
/*
fprintf(stdout, 
 "\n ivec1[%d] = %d, ivec2[%d] = %d, dvec[%d] = %e, inpmtx->nent = %d", 
 jj, ivec1[jj], jj, ivec2[jj], jj, dvec[jj], inpmtx->nent) ;
*/
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------
   input a row in the matrix

   created -- 96jul05, cca
   -------------------------
*/
void
DInpMtx_inputRow (
   DInpMtx   *inpmtx,
   int       row,
   int       rowsize,
   int       rowind[],
   double    rowent[]
) {
double   *dvec ;
int      col, ii, jj, newmaxnent ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if (  inpmtx == NULL || row < 0 || rowsize < 0 || rowind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputRow(%p,%d,%d,%p,%p)"
           "\n bad input\n", inpmtx, row, rowsize, rowind, rowent) ;
   exit(-1) ;
}
if ( inpmtx->inputMode != 2 && rowent != NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputRow(%p,%d,%d,%p,%p)"
           "\n inputMode = indices only, rowent not NULL \n", 
           inpmtx, row, rowsize, rowind, rowent) ;
   exit(-1) ;
}
if ( inpmtx->nent + rowsize > inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent + rowsize > inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->maxnent + rowsize ) {
      newmaxnent = inpmtx->maxnent + rowsize ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputRow(%p,%d,%d,%p,%p)"
           "\n ivec1 = %p, ivec2 = %p",
           inpmtx, row, rowsize, rowind, rowent, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_inputRow(%p,%d,%d,%p,%p)"
              "\n dvec = %p",
              inpmtx, row, rowsize, rowind, rowent, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   -------------
   input the row
   -------------
*/
if ( inpmtx->coordType == 1 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < rowsize ; ii++, jj++ ) {
      ivec1[jj] = row ;
      ivec2[jj] = rowind[ii] ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = rowent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 2 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < rowsize ; ii++, jj++ ) {
      ivec1[jj] = rowind[ii] ;
      ivec2[jj] = row ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = rowent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 3 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < rowsize ; ii++, jj++ ) {
      col = rowind[ii] ;
      if ( row <= col ) {
         ivec1[jj] = row ;
      } else {
         ivec1[jj] = col ;
      }
      ivec2[jj] = col - row ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = rowent[ii]  ;
      }
   }
}
inpmtx->nent += rowsize ;
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   input a column in the matrix

   created -- 96jul05, cca
   ----------------------------
*/
void
DInpMtx_inputColumn (
   DInpMtx   *inpmtx,
   int       col,
   int       colsize,
   int       colind[],
   double    colent[]
) {
double   *dvec ;
int      ii, jj, newmaxnent, row ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if (  inpmtx == NULL || col < 0 || colsize < 0 || colind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputColumn(%p,%d,%d,%p,%p)"
           "\n bad input\n", inpmtx, col, colsize, colind, colent) ;
   exit(-1) ;
}
if ( inpmtx->inputMode != 2 && colent != NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputColumn(%p,%d,%d,%p,%p)"
           "\n inputMode = indices only, colent not NULL \n", 
           inpmtx, col, colsize, colind, colent) ;
   exit(-1) ;
}
if ( inpmtx->nent + colsize > inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent + colsize > inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->maxnent + colsize ) {
      newmaxnent = inpmtx->maxnent + colsize ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputColumn(%p,%d,%d,%p,%p)"
           "\n ivec1 = %p, ivec2 = %p\n", 
           inpmtx, col, colsize, colind, colent, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_inputColumn(%p,%d,%d,%p,%p)"
              "\n dvec = %p\n", 
              inpmtx, col, colsize, colind, colent, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   ----------------
   input the column
   ----------------
*/
if ( inpmtx->coordType == 1 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < colsize ; ii++, jj++ ) {
      ivec1[jj] = colind[ii] ;
      ivec2[jj] = col ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = colent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 2 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < colsize ; ii++, jj++ ) {
      ivec1[jj] = col ;
      ivec2[jj] = colind[ii] ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = colent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 3 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < colsize ; ii++, jj++ ) {
      row = colind[ii] ;
      if ( row <= col ) {
         ivec1[jj] = row ;
      } else {
         ivec1[jj] = col ;
      }
      ivec2[jj] = col - row ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = colent[ii]  ;
      }
   }
}
inpmtx->nent += colsize ;
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------
   input a chevron in the matrix

   created -- 97feb08, cca
   -----------------------------
*/
void
DInpMtx_inputChevron (
   DInpMtx   *inpmtx,
   int       chv,
   int       chvsize,
   int       chvind[],
   double    chvent[]
) {
double   *dvec ;
int      col, ii, jj, newmaxnent, offset, row ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if (  inpmtx == NULL || chv < 0 || chvsize < 0 || chvind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputChevron(%p,%d,%d,%p,%p)"
           "\n bad input\n", inpmtx, chv, chvsize, chvind, chvent) ;
   exit(-1) ;
}
if ( inpmtx->inputMode != 2 && chvent != NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputChevron(%p,%d,%d,%p,%p)"
           "\n inputMode = indices only, chvent not NULL \n", 
           inpmtx, chv, chvsize, chvind, chvent) ;
   exit(-1) ;
}
if ( inpmtx->nent + chvsize > inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent + chvsize > inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->maxnent + chvsize ) {
      newmaxnent = inpmtx->maxnent + chvsize ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputChevron(%p,%d,%d,%p,%p)"
           "\n ivec1 = %p, ivec2 = %p\n", 
           inpmtx, chv, chvsize, chvind, chvent, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_inputChevron(%p,%d,%d,%p,%p)"
              "\n dvec = %p\n", 
              inpmtx, chv, chvsize, chvind, chvent, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   -----------------
   input the chevron
   -----------------
*/
if ( inpmtx->coordType == 1 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < chvsize ; ii++, jj++ ) {
      if ( (offset = chvind[ii]) >= 0 ) {
         row = chv ;
         col = chv + offset ;
      } else {
         col = chv ;
         row = chv - offset ;
      }
      ivec1[jj] = row ;
      ivec2[jj] = col ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = chvent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 2 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < chvsize ; ii++, jj++ ) {
      if ( (offset = chvind[ii]) >= 0 ) {
         row = chv ;
         col = chv + offset ;
      } else {
         col = chv ;
         row = chv - offset ;
      }
      ivec1[jj] = col ;
      ivec2[jj] = row ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = chvent[ii]  ;
      }
   }
} else if ( inpmtx->coordType == 3 ) {
   for ( ii = 0, jj = inpmtx->nent ; ii < chvsize ; ii++, jj++ ) {
      ivec1[jj] = chv ;
      ivec2[jj] = chvind[ii] ;
      if ( inpmtx->inputMode == 2 ) {
         dvec[jj]  = chvent[ii]  ;
      }
   }
}
inpmtx->nent += chvsize ;
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   input a matrix

   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_inputMatrix (
   DInpMtx   *inpmtx,
   int       nrow,
   int       ncol,
   int       rowstride,
   int       colstride,
   int       rowind[],
   int       colind[],
   double    mtxent[]
) {
double   *dvec ;
int      col, ii, jj, kk, newmaxnent, row ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if (  inpmtx == NULL || nrow < 0 || ncol < 0 
   || rowstride < 1 || colstride < 1
   || rowind == NULL || colind == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DInpMtx_inputMatrix(%p,%d,%d,%d,%d,%p,%p,%p)"
        "\n bad input\n", inpmtx, nrow, ncol, rowstride, colstride, 
        rowind, colind, mtxent) ;
   exit(-1) ;
}
if ( inpmtx->inputMode != 2 && mtxent != NULL ) {
   fprintf(stderr, 
        "\n fatal error in DInpMtx_inputMatrix(%p,%d,%d,%d,%d,%p,%p,%p)"
        "\n inputMode = indices only, mtxent not NULL \n", 
        inpmtx, nrow, ncol, rowstride, colstride, 
        rowind, colind, mtxent) ;
   exit(-1) ;
}
if ( nrow == 0 || ncol == 0 ) {
   return ;
}
if ( inpmtx->nent + nrow*ncol > inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent + nrow*ncol > inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->maxnent + nrow*ncol ) {
      newmaxnent = inpmtx->maxnent + nrow*ncol ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DInpMtx_inputMatrix(%p,%d,%d,%d,%d,%p,%p,%p)"
        "\n ivec1 = %p, ivec2 = %p\n", inpmtx, nrow, ncol, rowstride, 
        colstride, rowind, colind, mtxent, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
        "\n fatal error in DInpMtx_inputMatrix(%p,%d,%d,%d,%d,%p,%p,%p)"
        "\n dvec = %p\n", inpmtx, nrow, ncol, rowstride, 
        colstride, rowind, colind, mtxent, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   ----------------
   input the matrix
   ----------------
*/
if ( inpmtx->coordType == 1 ) {
   for ( jj = 0, kk = inpmtx->nent ; jj < ncol ; jj++ ) {
      col = colind[jj] ;
      for ( ii = 0 ; ii < nrow ; ii++, kk++ ) {
         row = rowind[ii] ;
         ivec1[kk] = row ;
         ivec2[kk] = col ;
         if ( inpmtx->inputMode == 2 ) {
            dvec[kk]  = mtxent[ii*rowstride + jj*colstride]  ;
         }
      }
   }
} else if ( inpmtx->coordType == 2 ) {
   for ( jj = 0, kk = inpmtx->nent ; jj < ncol ; jj++ ) {
      col = colind[jj] ;
      for ( ii = 0 ; ii < nrow ; ii++, kk++ ) {
         row = rowind[ii] ;
         ivec1[kk] = col ;
         ivec2[kk] = row ;
         if ( inpmtx->inputMode == 2 ) {
            dvec[kk]  = mtxent[ii*rowstride + jj*colstride]  ;
         }
      }
   }
} else if ( inpmtx->coordType == 3 ) {
   for ( jj = 0, kk = inpmtx->nent ; jj < ncol ; jj++ ) {
      col = colind[jj] ;
      for ( ii = 0 ; ii < nrow ; ii++, kk++ ) {
         row = rowind[ii] ;
         if ( row <= col ) {
            ivec1[kk] = row ;
         } else {
            ivec1[kk] = col ;
         }
         ivec2[kk] = col - row ;
         if ( inpmtx->inputMode == 2 ) {
            dvec[kk]  = mtxent[ii*rowstride + jj*colstride]  ;
         }
      }
   }
}
inpmtx->nent += nrow*ncol ;
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   input a number of (row,column, entry) triples into the matrix

   created -- 97apr10, cca
   -------------------------------------------------------------
*/
void
DInpMtx_inputTriples (
   DInpMtx   *inpmtx,
   int       ntriples,
   int       rowids[],
   int       colids[],
   double    entries[]
) {
double   *dvec ;
int      ii, jj, newmaxnent ;
int      *ivec1, *ivec2 ;
/*
   --------------
   check the data
   --------------
*/
if ( inpmtx == NULL || ntriples < 0 || rowids == NULL || colids == NULL 
   || (inpmtx->inputMode == 2 && entries == NULL) ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputTriples(%p,%d,%p,%p,%p)"
           "\n bad input\n", 
           inpmtx, ntriples, rowids, colids, entries) ;
   exit(-1) ;
}
if ( inpmtx->coordType < 1 || 3 < inpmtx->coordType ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputEntry(%p,%d,%p,%p,%p)"
           "\n bad coordType = %d\n", inpmtx, ntriples, rowids,
           colids, entries, inpmtx->coordType) ;
   exit(-1) ;
}
if ( inpmtx->nent + ntriples > inpmtx->maxnent ) {
/*
   -----------------------------------
   vectors are full, sort and compress
   -----------------------------------
*/
   DInpMtx_sortAndCompress(inpmtx) ;
   inpmtx->storageMode = 1 ;
}
if ( inpmtx->nent + ntriples > inpmtx->maxnent ) {
/*
   ------------------------------
   vectors are still full, resize
   ------------------------------
*/
   newmaxnent = inpmtx->maxnent * inpmtx->resizeMultiple ;
   if ( newmaxnent < inpmtx->nent + ntriples ) {
      newmaxnent = inpmtx->nent + ntriples ;
   }
   DInpMtx_setMaxnent(inpmtx, newmaxnent) ;
}
ivec1 = IV_entries(&inpmtx->ivec1IV) ;
ivec2 = IV_entries(&inpmtx->ivec2IV) ;
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_inputEntry(%p,%d,%p,%p,%p)"
           "\n ivec1 = %p, ivec2 = %p\n", 
           inpmtx, ntriples, rowids, colids, entries, ivec1, ivec2) ;
   exit(-1) ;
}
if ( inpmtx->inputMode == 2 ) {
   dvec = DV_entries(&inpmtx->dvecDV) ;
   if ( dvec == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_inputEntry(%p,%d,%p,%p,%p)"
              "\n dvec = %p\n", inpmtx, ntriples, rowids, colids,
              entries, dvec) ;
      exit(-1) ;
   }
} else {
   dvec = NULL ;
}
/*
   ---------------------
   loop over the triples
   ---------------------
*/
for ( ii = 0 ; ii < ntriples ; ii++ ) {
   jj = inpmtx->nent ;
   ivec1[jj] = rowids[ii] ;
   ivec2[jj] = colids[ii] ;
   if ( dvec != NULL ) {
      dvec[jj]  = entries[ii] ;
   }
   inpmtx->nent++ ;
}
/*
fprintf(stdout, 
 "\n ivec1[%d] = %d, ivec2[%d] = %d, dvec[%d] = %e, inpmtx->nent = %d", 
 jj, ivec1[jj], jj, ivec2[jj], jj, dvec[jj], inpmtx->nent) ;
*/
IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
if ( inpmtx->inputMode == 2 ) {
   DV_setSize(&inpmtx->dvecDV,  inpmtx->nent) ;
}
return ; }

/*--------------------------------------------------------------------*/
