/*  init.c  */

#include "../DInpMtx.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   initialize the object

   coordType -- coordinate type, input supported for types 1, 2 and 3
      1 -- row triples (i, j, a_{i,j})
      2 -- column triples (i, j, a_{j,i})
      3 -- chevron triples 
           (i, k, a_{i,i+k}) if k >= 0
           (i, k, a_{i-k,i}) if k < 0
           i is the chevron, k is the offset
      4 -- custom coordinate, e.g., one could store (I, k, a_{i,j})
           where I is the front where a_{i,j} will be assembled
           and k is the offset into the vector that holds the 
           entries in the front
   inputMode -- mode for input
      1 --> indices only
      2 --> indices and entries
   maxnent  -- upper bound on the number of entries,
      also equal to the workspace, so if the assembly includes
      overlapping data, give enough elbow room for efficiency.
   maxnvector -- upper bound on the number of vectors to be supported. 
      this may not be known ahead of time (e.g., the number of vectors 
      may be the number of fronts which is not known before the
      ordering is done and front tree constructed).

   created -- 96jul05, cca
   ------------------------------------------------------------------
*/
void
DInpMtx_init (
  DInpMtx   *inpmtx,
  int       coordType,
  int       inputMode,
  int       maxnent,
  int       maxnvector
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  inpmtx == NULL 
   || (coordType < 1 && coordType > 3)
   || (inputMode < 1 && inputMode > 2)
   || maxnent < 0 || maxnvector < 0 ) {
   fprintf(stderr, "\n fatal error in DInpMtx_init(%p,%d,%d,%d,%d)"
           "\n bad input\n", 
           inpmtx, coordType, inputMode, maxnent, maxnvector) ;
   exit(-1) ;
}
/*
   ------------------------
   clear the data structure
   ------------------------
*/
DInpMtx_clearData(inpmtx) ;
/*
   -------------------------------------------------------------------
   if maxnent > 0 then allocate the ivec1[], ivec2[] and dvec[] vectors
   -------------------------------------------------------------------
*/
inpmtx->coordType = coordType ;
inpmtx->inputMode = inputMode ;
if ( maxnent > 0 ) {
   DInpMtx_setMaxnent(inpmtx, maxnent) ;
}
/*
   ---------------------------------------
   if nvector > 0 initialize the vecids[], 
   sizes[] and offsets[] vectors.
   ---------------------------------------
*/
if ( maxnvector > 0 ) {
   DInpMtx_setMaxnvector(inpmtx, maxnvector) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------
   change the coordinate type

   created -- 96jul05, cca
   --------------------------
*/
void
DInpMtx_changeCoordType (
   DInpMtx   *inpmtx,
   int       newType
) {
int   chev, col, i, nent, off, oldType, row, temp ;
int   *ivec1, *ivec2 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || newType <= 0 || newType > 4 ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_changeCoordType(%p,%d)"
           "\n bad input\n", inpmtx, newType) ;
   exit(-1) ;
}
oldType = inpmtx->coordType ;
if ( oldType == newType ) {
   return ;
}
if ( oldType >= 4 ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_changeCoordType(%p,%d)"
           "\n bad coordType = %d\n", inpmtx, newType, oldType) ;
   exit(-1) ;
}
if ( newType == 4 ) {
/*
   ----------------------------------------------------
   custom type, just set the coordType field and return
   ----------------------------------------------------
*/
   inpmtx->coordType = newType ;
   return ;
}
nent  = inpmtx->nent ;
ivec1 = DInpMtx_ivec1(inpmtx) ;
ivec2 = DInpMtx_ivec2(inpmtx) ;
/*
if ( ivec1 == NULL || ivec2 == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_changeCoordType(%p,%d)"
           "\n ivec1 = %p, ivec2 = %p\n", 
           inpmtx, newType, ivec1, ivec2) ;
   exit(-1) ;
}
*/
if ( oldType == 0 ) {
   inpmtx->coordType = newType ;
} else if ( oldType == 1 ) {
   if ( newType == 2 ) {
/*
      ----------------------------------
      row coordType --> column coordType
      ----------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         temp = ivec1[i] ;
         ivec1[i] = ivec2[i] ;
         ivec2[i] = temp ;
      }
      inpmtx->coordType = 2 ;
      inpmtx->storageMode = 1 ;
   } else if ( newType == 3 ) {
/*
      -----------------------------------
      row coordType --> chevron coordType
      -----------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         row = ivec1[i] ;
         col = ivec2[i] ;
         if ( row <= col ) {
            ivec1[i] = row ;
            ivec2[i] = col - row ;
         } else {
            ivec1[i] = col ;
            ivec2[i] = col - row ;
         }
      }
      inpmtx->coordType = 3 ;
      inpmtx->storageMode = 1 ;
   }
} else if ( oldType == 2 ) {
   if ( newType == 1 ) {
/*
      ----------------------------------
      column coordType --> row coordType
      ----------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         temp = ivec1[i] ;
         ivec1[i] = ivec2[i] ;
         ivec2[i] = temp ;
      }
      inpmtx->coordType = 1 ;
      inpmtx->storageMode = 1 ;
   } else if ( newType == 3 ) {
/*
      --------------------------------------
      column coordType --> chevron coordType
      --------------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         col = ivec1[i] ;
         row = ivec2[i] ;
         if ( row <= col ) {
            ivec1[i] = row ;
            ivec2[i] = col - row ;
         } else {
            ivec1[i] = col ;
            ivec2[i] = col - row ;
         }
      }
      inpmtx->coordType = 3 ;
      inpmtx->storageMode = 1 ;
   }
} else if ( oldType == 3 ) {
   if ( newType == 1 ) {
/*
      -----------------------------------
      chevron coordType --> row coordType
      -----------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         chev = ivec1[i] ;
         off  = ivec2[i] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         ivec1[i] = row ;
         ivec2[i] = col ;
      }
      inpmtx->coordType = 1 ;
      inpmtx->storageMode = 1 ;
   } else if ( newType == 2 ) {
/*
      --------------------------------------
      chevron coordType --> column coordType
      --------------------------------------
*/
      for ( i = 0 ; i < nent ; i++ ) {
         chev = ivec1[i] ;
         off  = ivec2[i] ;
         if ( off >= 0 ) {
            row = chev ;
            col = chev + off ;
         } else {
            col = chev ;
            row = chev - off ;
         }
         ivec1[i] = col ;
         ivec2[i] = row ;
      }
      inpmtx->coordType = 2 ;
      inpmtx->storageMode = 1 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   change the storage mode

   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_changeStorageMode (
   DInpMtx   *inpmtx,
   int       newMode
) {
int   oldMode ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || newMode < 0 || newMode >= 4 ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_changeStorageMode(%p,%d)"
           "\n bad input\n", inpmtx, newMode) ;
   exit(-1) ;
}
oldMode = inpmtx->storageMode ;
if ( oldMode == newMode ) {
   return ;
}
if ( oldMode == 0 ) {
   inpmtx->storageMode = newMode ;
} else if ( oldMode == 1 ) {
   if ( newMode == 2 ) {
/*
      -------------------------------------
      raw triples --> sorted and compressed 
      -------------------------------------
*/
      DInpMtx_sortAndCompress(inpmtx) ;
   } else if ( newMode == 3 ) {
/*
      -----------------------
      raw triples --> vectors
      -----------------------
*/
      DInpMtx_sortAndCompress(inpmtx) ;
      DInpMtx_convertToVectors(inpmtx) ;
   }
} else if ( oldMode == 2 ) {
   if ( newMode == 1 ) {
/*
      --------------------------------------
      sorted and compressed --> raw triples, 
      simply set the storageMode field
      --------------------------------------
*/
      inpmtx->storageMode = 1 ;
   } else if ( newMode == 3 ) {
/*
      ---------------------------------
      sorted and compressed --> vectors
      ---------------------------------
*/
      DInpMtx_convertToVectors(inpmtx) ;
   }
} else if ( oldMode == 3 ) {
   if ( newMode == 1 || newMode == 2 ) {
/*
      --------------------------------
      vectors --> triples, 
      simply set the storageMode field
      --------------------------------
*/
      inpmtx->storageMode = newMode ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
