/*  IO.c  */

#include "../DInpMtx.h"

static const char *suffixb = ".dinpmtxb" ;
static const char *suffixf = ".dinpmtxf" ;

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------
   purpose -- to read in a DInpMtx object from a file

   input --

      fn -- filename, must be *.inpmtxb or *.inpmtxf

   return value -- 1 if success, 0 if failure

   created -- 96jul05, cca
   --------------------------------------------------
*/
int
DInpMtx_readFromFile ( 
   DInpMtx   *inpmtx, 
   char    *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fn == NULL ) {
   fprintf(stderr, "\n error in DInpMtx_readFromFile(%p,%s)"
           "\n bad input\n", inpmtx, fn) ;
   return(0) ;
}
/*
   -------------
   read the file
   -------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "rb")) == NULL ) {
         fprintf(stderr, "\n error in DInpMtx_readFromFile(%p,%s)"
                 "\n unable to open file %s", inpmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DInpMtx_readFromBinaryFile(inpmtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "r")) == NULL ) {
         fprintf(stderr, "\n error in DInpMtx_readFromFile(%p,%s)"
                 "\n unable to open file %s", inpmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DInpMtx_readFromFormattedFile(inpmtx, fp) ;
         fclose(fp) ;
      }
   } else {
      fprintf(stderr, "\n error in DInpMtx_readFromFile(%p,%s)"
              "\n bad DInpMtx file name %s,"
              "\n must end in %s (binary) or %s (formatted)\n",
              inpmtx, fn, fn, suffixb, suffixf) ;
      rc = 0 ;
   }
} else {
   fprintf(stderr, "\n error in DInpMtx_readFromFile(%p,%s)"
       "\n bad DInpMtx file name %s,"
       "\n must end in %s (binary) or %s (formatted)\n",
       inpmtx, fn, fn, suffixb, suffixf) ;
   rc = 0 ;
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------
   purpose -- to read a DInpMtx object from a formatted file

   return value -- 1 if success, 0 if failure

   created -- 96jul05, cca
   --------------------------------------------------------
*/
int
DInpMtx_readFromFormattedFile ( 
   DInpMtx   *inpmtx, 
   FILE      *fp 
) {
int   rc ;
int   itemp[5] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DInpMtx_readFromFormattedFile(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DInpMtx_clearData(inpmtx) ;
/*
   --------------------------------------------------------
   read in the five scalar parameters
   coordinate type, storage mode, input mode, nent, nvector
   --------------------------------------------------------
*/
if ( (rc = IVfscanf(fp, 5, itemp)) != 5 ) {
   fprintf(stderr, "\n error in DInpMtx_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", inpmtx, fp, rc, 5) ;
   return(0) ;
}
inpmtx->coordType   = itemp[0] ;
inpmtx->storageMode = itemp[1] ;
inpmtx->inputMode   = itemp[2] ;
inpmtx->nent        = itemp[3] ;
inpmtx->nvector     = itemp[4] ;
if ( inpmtx->nent > 0 ) {
   IV_readFromFormattedFile(&inpmtx->ivec1IV, fp) ;
   IV_readFromFormattedFile(&inpmtx->ivec2IV, fp) ;
   if ( inpmtx->inputMode == 2 ) {
      DV_readFromFormattedFile(&inpmtx->dvecDV, fp) ;
   }
}
if ( inpmtx->nvector > 0 ) {
   IV_readFromFormattedFile(&inpmtx->vecidsIV,  fp) ;
   IV_readFromFormattedFile(&inpmtx->sizesIV,   fp) ;
   IV_readFromFormattedFile(&inpmtx->offsetsIV, fp) ;
}
inpmtx->maxnent = inpmtx->nent ;

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   purpose -- to read a DInpMtx object from a binary file

   return value -- 1 if success, 0  if failure

   created -- 96jul05, cca
   ------------------------------------------------------
*/
int
DInpMtx_readFromBinaryFile ( 
   DInpMtx   *inpmtx, 
   FILE    *fp 
) {
int   rc ;
int   itemp[5] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_readFromBinaryFile(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DInpMtx_clearData(inpmtx) ;
/*
   ---------------------------------------------
   read in the five scalar parameters
   coordType storageMode inputMode nent nvector
   ---------------------------------------------
*/
if ( (rc = fread((void *) itemp, sizeof(int), 5, fp)) != 5 ) {
   fprintf(stderr, "\n error in DInpMtx_readFromBinaryFile(%p,%p)"
           "\n %d items of %d read\n", inpmtx, fp, rc, 5) ;
   return(0) ;
}
inpmtx->coordType   = itemp[0] ;
inpmtx->storageMode = itemp[1] ;
inpmtx->inputMode   = itemp[2] ;
inpmtx->nent        = itemp[3] ;
inpmtx->nvector     = itemp[4] ;
if ( inpmtx->nent > 0 ) {
   IV_readFromBinaryFile(&inpmtx->ivec1IV, fp) ;
   IV_readFromBinaryFile(&inpmtx->ivec2IV, fp) ;
   if ( inpmtx->inputMode == 2 ) {
      DV_readFromBinaryFile(&inpmtx->dvecDV, fp) ;
   }
}
if ( inpmtx->nvector > 0 ) {
   IV_readFromBinaryFile(&inpmtx->vecidsIV,  fp) ;
   IV_readFromBinaryFile(&inpmtx->sizesIV,   fp) ;
   IV_readFromBinaryFile(&inpmtx->offsetsIV, fp) ;
}
inpmtx->maxnent = inpmtx->nent ;

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   purpose -- to read a DInpMtx object from a TRANAIR file

   return value -- 1 if success, 0  if failure

   created -- 97feb14, cca
   -------------------------------------------------------
*/
int
DInpMtx_readFromTRANAIRfile ( 
   DInpMtx   *inpmtx, 
   FILE    *fp 
) {
double   entry ;
int      ii, irow, jcol, jrow, ncol, nrow, size, rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_readFromTRANAIRfile(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DInpMtx_clearData(inpmtx) ;
/*
   --------------------------------------
   read in the number of rows and columns
   --------------------------------------
*/
rc = fscanf(fp, " %d %d", &nrow, &ncol) ;
if ( rc != 2 ) {
   fprintf(stderr, "\n error in DInpMtx_readFromTRANAIRfile(%p,%p)"
           "\n rc = %d reading nrow and ncol \n", inpmtx, fp, rc) ;
   return(0) ;
}
if ( nrow != ncol ) {
   fprintf(stderr, "\n error in DInpMtx_readFromTRANAIRfile(%p,%p)"
           "\n nrow = %d, ncol = %d\n", inpmtx, fp, nrow, ncol) ;
   return(0) ;
}
/*
   -----------------------------
   initialize the DInpMtx object
   -----------------------------
*/
DInpMtx_init(inpmtx, 1, 2, nrow, 0) ;
DInpMtx_setResizeMultiple(inpmtx, 2.0) ;
/*
   ----------------
   read in the data
   ----------------
*/
for ( irow = 1 ; irow <= nrow ; irow++ ) {
   rc = fscanf(fp, " %d %d", &jrow, &size) ;
   if ( rc != 2 ) {
      fprintf(stderr, "\n error in DInpMtx_readFromTRANAIRfile(%p,%p)"
              "\n rc = %d reading row %d\n", inpmtx, fp, rc, irow) ;
      return(0) ;
   }
   if ( jrow != irow ) {
      fprintf(stderr, "\n error in DInpMtx_readFromTRANAIRfile(%p,%p)"
              "\n jrow = %d reading row %d\n", inpmtx, fp, jrow, irow) ;
      return(0) ;
   }
/*
fprintf(stdout, "\n begin: irow %d, jrow %d, nent %d, maxnent %d", 
        irow, jrow, inpmtx->nent, inpmtx->maxnent) ;
*/
   for ( ii = 0 ; ii < size ; ii++ ) {
      rc = fscanf(fp, " %d %le", &jcol, &entry) ;
      if ( rc != 2 ) {
         fprintf(stderr, 
                 "\n error in DInpMtx_readFromTRANAIRfile(%p,%p)"
                 "\n rc = %d reading row %d, entry %d\n", 
                 inpmtx, fp, rc, irow, ii) ;
         return(0) ;
      }
      DInpMtx_inputEntry(inpmtx, jrow - 1, jcol - 1, entry) ;
   }
/*
fprintf(stdout, "\n end: irow %d, jrow %d, nent %d, maxnent %d", 
        irow, jrow, inpmtx->nent, inpmtx->maxnent) ;
*/
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   purpose -- to write a DInpMtx object to a file

   input --

      fn -- filename
        *.inpmtxb -- binary
        *.inpmtxf -- formatted
        anything else -- for human eye

   return value -- 1 if success, 0 otherwise

   created -- 96jul05, cca
   ----------------------------------------------
*/
int
DInpMtx_writeToFile ( 
   DInpMtx   *inpmtx, 
   char    *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fn == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_writeToFile(%p,%s)"
    "\n bad input\n", inpmtx, fn) ; 
   return(0) ;
}
/*
   ------------------
   write out the file
   ------------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "wb")) == NULL ) {
         fprintf(stderr, "\n error in DInpMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", inpmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DInpMtx_writeToBinaryFile(inpmtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "w")) == NULL ) {
         fprintf(stderr, "\n error in DInpMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", inpmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DInpMtx_writeToFormattedFile(inpmtx, fp) ;
         fclose(fp) ;
      }
   } else {
      if ( (fp = fopen(fn, "a")) == NULL ) {
         fprintf(stderr, "\n error in DInpMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", inpmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DInpMtx_writeForHumanEye(inpmtx, fp) ;
         fclose(fp) ;
      }
   }
} else {
   if ( (fp = fopen(fn, "a")) == NULL ) {
      fprintf(stderr, "\n error in DInpMtx_writeToFile(%p,%s)"
              "\n unable to open file %s", inpmtx, fn, fn) ;
      rc = 0 ;
   } else {
      rc = DInpMtx_writeForHumanEye(inpmtx, fp) ;
      fclose(fp) ;
   }
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   purpose -- to write a DInpMtx object to a formatted file

   return value -- 1 if success, 0 otherwise

   created -- 96jul05, cca
   ------------------------------------------------------
*/
int
DInpMtx_writeToFormattedFile ( 
   DInpMtx   *inpmtx, 
   FILE    *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   return(0) ;
}
/*
   ------------------------------------
   write out the five scalar parameters
   ------------------------------------
*/
rc = fprintf(fp, "\n %d %d %d %d %d", 
             inpmtx->coordType, inpmtx->storageMode, 
             inpmtx->inputMode, inpmtx->nent, inpmtx->nvector) ;
if ( rc < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
           "\n rc = %d, return from first fprintf\n", inpmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------
   write out the vectors
   ---------------------
*/
if ( inpmtx->nent > 0 ) {
   IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
   rc = IV_writeToFormattedFile(&inpmtx->ivec1IV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
              "\n rc = %d, return from writing ivec1\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
   rc = IV_writeToFormattedFile(&inpmtx->ivec2IV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
              "\n rc = %d, return from writing ivec2\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   if ( inpmtx->inputMode == 2 ) {
      DV_setSize(&inpmtx->dvecDV, inpmtx->nent) ;
      rc = DV_writeToFormattedFile(&inpmtx->dvecDV,  fp) ;
      if ( rc < 0 ) {
         fprintf(stderr, 
                 "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
                 "\n rc = %d, return from writing dvec\n", 
                 inpmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( inpmtx->nvector > 0 ) {
   rc = IV_writeToFormattedFile(&inpmtx->vecidsIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
              "\n rc = %d, return from writing vecids\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   rc = IV_writeToFormattedFile(&inpmtx->sizesIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
              "\n rc = %d, return from writing sizes\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   rc = IV_writeToFormattedFile(&inpmtx->offsetsIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToFormattedFile(%p,%p)"
              "\n rc = %d, return from writing offsets\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------
   purpose -- to write a DInpMtx object to a binary file

   return value -- 1 if success, 0 otherwise

   created -- 96jul05, cca
   ---------------------------------------------------
*/
int
DInpMtx_writeToBinaryFile ( 
   DInpMtx    *inpmtx, 
   FILE   *fp 
) {
int   rc ;
int   itemp[6] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   return(0) ;
}
/*
   ------------------------------------
   write out the five scalar parameters
   ------------------------------------
*/
itemp[0] = inpmtx->coordType   ;
itemp[1] = inpmtx->storageMode ;
itemp[2] = inpmtx->inputMode   ;
itemp[3] = inpmtx->nent        ;
itemp[4] = inpmtx->nvector     ;
rc = fwrite((void *) itemp, sizeof(int), 5, fp) ;
if ( rc != 5 ) {
   fprintf(stderr, "\n error in DInpMtx_writeToBinaryFile(%p,%p)"
           "\n %d of %d scalar items written\n", inpmtx, fp, rc, 5) ;
   return(0) ;
}
/*
   ---------------------
   write out the vectors
   ---------------------
*/
if ( inpmtx->nent > 0 ) {
   IV_setSize(&inpmtx->ivec1IV, inpmtx->nent) ;
   rc = IV_writeToBinaryFile(&inpmtx->ivec1IV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
              "\n rc = %d, return from writing ivec1\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   IV_setSize(&inpmtx->ivec2IV, inpmtx->nent) ;
   rc = IV_writeToBinaryFile(&inpmtx->ivec2IV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
              "\n rc = %d, return from writing ivec2\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   if ( inpmtx->inputMode == 2 ) {
      DV_setSize(&inpmtx->dvecDV, inpmtx->nent) ;
      rc = DV_writeToBinaryFile(&inpmtx->dvecDV, fp) ;
      if ( rc < 0 ) {
         fprintf(stderr, 
                 "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
                 "\n rc = %d, return from writing dvec\n", 
                 inpmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( inpmtx->nvector > 0 ) {
   rc = IV_writeToBinaryFile(&inpmtx->vecidsIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
              "\n rc = %d, return from writing vecids\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   rc = IV_writeToBinaryFile(&inpmtx->sizesIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
              "\n rc = %d, return from writing sizes\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
   rc = IV_writeToBinaryFile(&inpmtx->offsetsIV, fp) ;
   if ( rc < 0 ) {
      fprintf(stderr, 
              "\n fatal error in DInpMtx_writeToBinaryFile(%p,%p)"
              "\n rc = %d, return from writing offsets\n", 
              inpmtx, fp, rc) ;
      return(0) ;
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------
   purpose -- to write a DInpMtx object for a human eye

   return value -- 1 if success, 0 otherwise

   created -- 96jul05, cca
   ----------------------------------------------------
*/
int
DInpMtx_writeForHumanEye ( 
   DInpMtx    *inpmtx, 
   FILE   *fp 
) {
double   *dvec, *entries ;
int      ierr, ii, iv, rc, size ;
int      *indices, *ivec1, *ivec2, *offsets, *sizes, *vecids ;

if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DInpMtx_writeForHumanEye(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   exit(-1) ;
}
/*
   ------------------------
   write out the statistics
   ------------------------
*/
if ( (rc = DInpMtx_writeStats(inpmtx, fp)) == 0 ) {
   fprintf(stderr, "\n fatal error in DInpMtx_writeForHumanEye(%p,%p)"
           "\n rc = %d, return from DInpMtx_writeStats(%p,%p)\n",
           inpmtx, fp, rc, inpmtx, fp) ;
   return(0) ;
}
if (  inpmtx->nent > 0 ) {
   if ( inpmtx->storageMode == 1 || inpmtx->storageMode == 2 ) {
      ivec1 = DInpMtx_ivec1(inpmtx) ;
      ivec2 = DInpMtx_ivec2(inpmtx) ;
      fprintf(fp, "\n data via triples") ;
      if ( inpmtx->inputMode == 1 ) {
         for ( ii = 0 ; ii < inpmtx->nent ; ii++ ) {
            if ( ii % 4 == 0 ) {   
               fprintf(fp, "\n") ; 
            }
            fprintf(fp, " <%6d,%6d>", ivec1[ii], ivec2[ii]) ;
         }
      } else if ( inpmtx->inputMode == 2 ) {
         dvec = DInpMtx_dvec(inpmtx) ;
         for ( ii = 0 ; ii < inpmtx->nent ; ii++ ) {
            if ( ii % 2 == 0 ) {   
               fprintf(fp, "\n") ; 
            }
            fprintf(fp, " <%6d,%6d,%20.12e>", 
                    ivec1[ii], ivec2[ii], dvec[ii]) ;
         }
      }
   } else if ( inpmtx->storageMode == 3 && inpmtx->nvector > 0 ) {
      fprintf(fp, "\n data via vectors") ;
      ivec2   = DInpMtx_ivec2(inpmtx) ;
      dvec    = DInpMtx_dvec(inpmtx) ;
      vecids  = DInpMtx_vecids(inpmtx) ;
      sizes   = DInpMtx_sizes(inpmtx) ;
      offsets = DInpMtx_offsets(inpmtx) ;
      if ( inpmtx->inputMode == 1 ) {
         for ( iv = 0 ; iv < inpmtx->nvector ; iv++ ) {
            DInpMtx_vector(inpmtx, iv, &size, &indices, &entries) ;
            fprintf(fp, "\n %6d : ", vecids[iv]) ;
            IVfp80(fp, sizes[iv], ivec2 + offsets[iv], 10, &ierr) ;
         }
      } else if ( inpmtx->inputMode == 2 ) {
         for ( iv = 0 ; iv < inpmtx->nvector ; iv++ ) {
            DInpMtx_vector(inpmtx, iv, &size, &indices, &entries) ;
            fprintf(fp, "\n %6d : ", vecids[iv]) ;
            IVfp80(fp, sizes[iv], ivec2 + offsets[iv], 10, &ierr) ;
            DVfprintf(fp, sizes[iv], dvec + offsets[iv]) ;
         }
      }
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------
   purpose -- to write out the statistics for the DInpMtx object

   return value -- 1 if success, 0 otherwise

   created -- 96jul05, cca
   -----------------------------------------------------------
*/
int
DInpMtx_writeStats ( 
   DInpMtx    *inpmtx, 
   FILE   *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( inpmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DInpMtx_writeStats(%p,%p)"
           "\n bad input\n", inpmtx, fp) ;
   exit(-1) ;
}
fprintf(fp, "\n DInpMtx : double precision input matrix object :") ;
switch ( inpmtx->coordType ) {
case 0 : 
   rc = fprintf(fp, "\n coordType = 0 --> none specified") ;
   break ;
case 1 : 
   rc = fprintf(fp, "\n coordType = 1 --> row triples") ;
   break ;
case 2 : 
   rc = fprintf(fp, "\n coordType = 2 --> column triples") ;
   break ;
case 3 : 
   rc = fprintf(fp, "\n coordType = 3 --> chevron triples") ;
   break ;
case 4 : 
   rc = fprintf(fp, "\n coordType = 4 --> custom triples") ;
   break ;
default :
   fprintf(stderr, "\n fatal error in DInpMtx_writeStats(%p,%p)"
           "\n invalid inpmtx->coordType = %d\n", 
           inpmtx, fp, inpmtx->coordType) ;
   return(0) ;
}
if ( rc < 0 ) { goto IO_error ; }
switch ( inpmtx->storageMode ) {
case 0 : 
   rc = fprintf(fp, "\n storageMode = 0 --> none specified") ;
   break ;
case 1 : 
   rc = fprintf(fp, "\n storageMode = 1 --> raw triples") ;
   break ;
case 2 : 
   rc = fprintf(fp, 
                "\n storageMode = 2 --> sorted and distinct triples") ;
   break ;
case 3 : 
   rc = fprintf(fp, 
                "\n storageMode = 3 --> vectors by first coordinate") ;
   break ;
default :
   fprintf(stderr, "\n fatal error in DInpMtx_writeStats(%p,%p)"
           "\n invalid inpmtx->storageMode = %d\n", 
           inpmtx, fp, inpmtx->storageMode) ;
   return(0) ;
}
if ( rc < 0 ) { goto IO_error ; }
switch ( inpmtx->inputMode ) {
case 0 : 
   rc = fprintf(fp, "\n inputMode = 0 --> none specified") ;
   break ;
case 1 : 
   rc = fprintf(fp, "\n inputMode = 1 --> indices only") ;
   break ;
case 2 : 
   rc = fprintf(fp, 
                "\n inputMode = 2 --> indices and entries") ;
   break ;
default :
   fprintf(stderr, "\n fatal error in DInpMtx_writeStats(%p,%p)"
           "\n invalid inpmtx->inputMode = %d\n", 
           inpmtx, fp, inpmtx->inputMode) ;
   return(0) ;
}
rc = fprintf(fp, "\n maxnent = %d --> maximum number of entries",
             inpmtx->maxnent) ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, "\n nent = %d --> present number of entries",
             inpmtx->nent) ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, "\n resizeMultiple = %.4g --> resize multiple",
             inpmtx->resizeMultiple) ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, "\n maxnvector = %d --> maximum number of vectors",
             inpmtx->maxnvector) ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, "\n nvector = %d --> present number of vectors",
             inpmtx->nvector) ;
if ( rc < 0 ) { goto IO_error ; }
fflush(fp) ;
return(1) ;

IO_error :
   fprintf(stderr, "\n fatal error in DInpMtx_writeStats(%p,%p)"
           "\n rc = %d, return from fprintf\n", inpmtx, fp, rc) ;
   return(0) ;
}

/*--------------------------------------------------------------------*/
