/*  testHBIO.c  */

#include "../DInpMtx.h"
#include "../../misc.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------
   read in a Harwell-Boeing matrix, 
   convert to a DInpMtx object and write to a file.

   created -- 97mar13, cca
   ---------------------------------------------------
*/
{
char      *inFileName, *outFileName, *type ;
double    *values ;
int       ierr, ii, iiend, iistart, inputMode, jcol, msglvl, ncol, 
          nnonzeros, nrhs, nrow, rc ;
int       *colptr, *colind, *rowind ;
DInpMtx   *inpmtx ;
FILE      *msgFile ;

if ( argc != 5 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile inFile outFile"
      "\n    msglvl   -- message level"
      "\n    msgFile  -- message file"
      "\n    inFile   -- input file, must be Harwell-Boeing format"
      "\n    outFile  -- output file, must be *.dinpmtxf or *.dinpmtxb"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inFileName  = argv[3] ;
outFileName = argv[4] ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl   -- %d" 
        "\n msgFile  -- %s" 
        "\n inFile   -- %s" 
        "\n outFile  -- %s" 
        "\n",
        argv[0], msglvl, argv[2], inFileName, outFileName) ;
fflush(msgFile) ;
/*
   ---------------------------------------------
   read in the Harwell-Boeing matrix information
   ---------------------------------------------
*/
if ( strcmp(inFileName, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
readHB_info(inFileName, &nrow, &ncol, &nnonzeros, &type, &nrhs) ;
fprintf(msgFile, 
        "\n matrix is %d x %d with %d entries, type = %s, nrhs = %d",
        nrow, ncol, nnonzeros, type, nrhs) ;
fflush(msgFile) ;
switch ( type[0] ) {
case 'P' :
   inputMode = 1 ;
   break ;
case 'R' :
   inputMode = 2 ;
   break ;
default :
   fprintf(stderr, "\n fatal error in %s, type = %s"
           "\n first character must be 'P' or 'R'",
           argv[0], type) ;
   exit(-1) ;
   break ;
}
/*
   -----------------------------
   initialize the DInpMtx object
   -----------------------------
*/
inpmtx = DInpMtx_new() ;
DInpMtx_init(inpmtx, 2, inputMode, nnonzeros, ncol+1) ;
colptr = IVinit(ncol+1, -1) ;
colind = DInpMtx_ivec1(inpmtx)   ;
rowind = DInpMtx_ivec2(inpmtx)   ;
values = DInpMtx_dvec(inpmtx)    ;
/*
   -------------------------------
   read in the indices and entries
   -------------------------------
*/
readHB_mat_double(inFileName, colptr, rowind, values) ;
/*
   --------------------------------------------
   decrement the column offsets and row indices
   --------------------------------------------
*/
for ( jcol = 0 ; jcol <= ncol ; jcol++ ) {
   colptr[jcol]-- ;
}
for ( ii = 0 ; ii < nnonzeros ; ii++ ) {
   rowind[ii]-- ;
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n colptr") ;
   IVfp80(msgFile, ncol+1, colptr, 80, &ierr) ;
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n rowind") ;
   IVfp80(msgFile, nnonzeros, rowind, 80, &ierr) ;
}
if ( values != NULL && msglvl > 3 ) {
   fprintf(msgFile, "\n\n values") ;
   DVfprintf(msgFile, nnonzeros, values) ;
}
/*
   -------------------------------------------
   fill the ivec1[] vector with column indices
   -------------------------------------------
*/
for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
   iistart = colptr[jcol] ;
   iiend   = colptr[jcol+1] - 1 ;
   for ( ii = iistart ; ii <= iiend ; ii++ ) {
      colind[ii] = jcol ;
   }
}
DInpMtx_setNent(inpmtx, nnonzeros) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n DInpMtx object after loading") ;
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   ----------------------------
   write out the DInpMtx object
   ----------------------------
*/
if ( strcmp(outFileName, "none") != 0 ) {
   rc = DInpMtx_writeToFile(inpmtx, outFileName) ;
   fprintf(msgFile, 
           "\n return value %d from DInpMtx_writeToFile(%p,%s)",
           rc, inpmtx, outFileName) ;
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
DInpMtx_free(inpmtx) ;
IVfree(colptr) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
