/*  DInpMtx.h  */

#include "../IV.h"
#include "../IVL.h"
#include "../DV.h"
#include "../cfiles.h"

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   coordType -- coordinate type
      0 -- no type specified
      1 -- row triples (i, j, a_{i,j})
      2 -- column triples (i, j, a_{j,i})
      3 -- chevron triples 
           (i, k, a_{i,i+k}) if k >= 0
           (i, k, a_{i-k,i}) if k < 0
           i is the chevron, k is the offset
      4 -- custom coordinate, e.g., one could store (I, k, a_{i,j})
           where I is the front where a_{i,j} will be assembled
           and k is the offset into the vector that holds the 
           entries in the front
   storageMode -- storage mode
      0 -- no mode specified
      1 -- filled with raw triples
      2 -- filled with sorted and distinct triples
      3 -- vectors by the first coordinate, ivec1[*] ignored
   inputMode -- input mode
      0 -- no input allowed
      1 -- indices only
      2 -- indices and entries
   maxnent -- present maximum number of entries
   nent    -- present number of entries
   resizeMultiple -- when resizing is done, 
      new maxnent = old maxnent * resizeMultiple
   ivec1IV    -- IV object that holds the first coordinates
   ivec2IV    -- IV object that holds the second coordinates
   dvecDV     -- DV object that holds the entries
   maxnvector -- present number of vectors
   nvector    -- present number of vectors
   vecidsIV   -- IV object that holds vector ids
   sizesIV    -- IV object that holds vector sizes
   offsetsIV  -- IV object that holds vector offsets

   created -- 96jul05, cca
   ------------------------------------------------------------------
*/
typedef struct _DInpMtx   DInpMtx ;
struct _DInpMtx {
   int      coordType      ;
   int      storageMode    ;
   int      inputMode      ;
   int      maxnent        ;
   int      nent           ;
   double   resizeMultiple ;
   IV       ivec1IV        ;
   IV       ivec2IV        ;
   DV       dvecDV         ;
   int      maxnvector     ;
   int      nvector        ;
   IV       vecidsIV       ;
   IV       sizesIV        ;
   IV       offsetsIV      ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor

   created -- 96jul05, cca
   -----------------------
*/
DInpMtx *
DInpMtx_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields

   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_setDefaultFields (
   DInpMtx   *inpmtx
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage

   created -- 96jul05, cca
   --------------------------------------------------
*/
void
DInpMtx_clearData ( 
   DInpMtx   *inpmtx 
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data

   created -- 96jul05, cca
   ------------------------------------------
*/
DInpMtx *
DInpMtx_free ( 
   DInpMtx   *inpmtx 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in instance.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------
   returns input mode

   created -- 97feb06 jcp
   ----------------------
*/
int   
DInpMtx_inputMode (
   DInpMtx  *inpmtx
) ;
/*
   ----------------------
   returns storage mode

   created -- 97feb06 jcp
    ----------------------
*/
int   
DInpMtx_storageMode (
   DInpMtx  *inpmtx
) ;
/*
   -----------------------
   returns coordinate type

   created -- 97feb06 jcp
   -----------------------
*/
int   
DInpMtx_coordType (
   DInpMtx  *inpmtx
) ;
/*
   ------------------------------
   returns inpmtx->resizeMultiple

   created -- 97feb06 jcp
   ------------------------------
*/
double   
DInpMtx_resizeMultiple (
   DInpMtx  *inpmtx
) ;
/*
   -----------------------
   returns inpmtx->mxnent

   created -- 97feb06 jcp
   -----------------------
*/
int   
DInpMtx_maxnent (
   DInpMtx  *inpmtx
) ;
/*
   -----------------------
   returns inpmtx->nent

   created -- 97feb06 jcp
   -----------------------
*/
int   
DInpMtx_nent (
   DInpMtx  *inpmtx
) ;
/*
   -------------------------
   returns inpmtx->mxnvector

   created -- 97feb06 jcp
   -------------------------
*/
int   
DInpMtx_maxnvector (
   DInpMtx  *inpmtx
) ;
/*
   -----------------------
   returns inpmtx->nvector

   created -- 97feb06 jcp
   -----------------------
*/
int   
DInpMtx_nvector (
   DInpMtx  *inpmtx
) ;
/*
   ----------------------------------
   returns pointer to vecids[] vector

   created -- 97jul11 cca
   ----------------------------------
*/
int *   
DInpMtx_vecids (
   DInpMtx  *inpmtx
) ;
/*
   ---------------------------------
   returns pointer to sizes[] vector

   created -- 97feb06 jcp
   ---------------------------------
*/
int *   
DInpMtx_sizes (
   DInpMtx  *inpmtx
) ;
/*
   -----------------------------------
   returns pointer to offsets[] vector

   created -- 97feb06 jcp
   -----------------------------------
*/
int *   
DInpMtx_offsets (
   DInpMtx  *inpmtx
) ;
/*
   ---------------------------------
   returns pointer to ivec1[] vector

   created -- 97feb06 jcp
   ---------------------------------
*/
int *   
DInpMtx_ivec1 (
   DInpMtx  *inpmtx
) ;
/*
   ---------------------------------
   returns pointer to ivec2[] vector

   created -- 97feb06 jcp
   ---------------------------------
*/
int *   
DInpMtx_ivec2 (
   DInpMtx  *inpmtx
) ;
/*
   --------------------------------
   returns pointer to dvec[] vector

   created -- 97feb06 jcp
   --------------------------------
*/
double *   
DInpMtx_dvec (
   DInpMtx  *inpmtx
) ;
/*
   ---------------------------------------
   retrieve requested vector
   set *pnent to # of entries
       *pindices to address of first index
       *pentries to address of first entry

   created -- 97jan09, jcp
   ---------------------------------------
*/
void
DInpMtx_vector (
   DInpMtx   *inpmtx,
   int       id,
   int       *pnent,
   int       **pindices,
   double    **pentries
) ;
/*
   --------------------------------------------------------------
   sets the maximum numnber of entries.  this methods resizes the
   ivec1[], ivece2[] and dvec[] vectors if newmaxnent != maxnent

   created -- 97feb06 jcp
   --------------------------------------------------------------
*/
void   
DInpMtx_setMaxnent (
   DInpMtx  *inpmtx, 
   int      newmaxnent
) ;
/*
   ---------------------------------
   set the present number of entries

   created -- 97Jan21, jcp
   --------------------------------
*/
void
DInpMtx_setNent (
   DInpMtx   *inpmtx,
   int       newnent
) ;
/*
   ----------------------------------------------------------
   sets the maximum numnber of vectors. this methods resizes 
   the sizes[] and offsets[] vectors if newmaxnent != maxnent

   created -- 97feb06 jcp
    ----------------------------------------------------------
*/
void   
DInpMtx_setMaxnvector (
   DInpMtx  *inpmtx, 
   int      newmaxnvector
) ;
/*
   ---------------------------------
   set the present number of vectors

   created -- 97feb08, cca
   ---------------------------------
*/
void   
DInpMtx_setNvector (
   DInpMtx   *inpmtx, 
   int       newnvector
) ;
/*
   ---------------------------
   sets inpmtx->resizeMultiple

   created -- 97feb06 jcp
    ---------------------------
*/
void   
DInpMtx_setResizeMultiple (
   DInpMtx   *inpmtx, 
   double    resizeMultiple
) ;
/*
   --------------------------------------------
   sets coordType of DInpMtx structure to
   allow user to define custom coordinate type.
   Note, new type must be > 3.

   created -- 97feb06 jcp
    --------------------------------------------
*/
void   
DInpMtx_setCoordType (
   DInpMtx  *inpmtx, 
   int      type
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   initialize the object
 
   coordType -- coordinate type, input supported for types 1, 2 and 3
      1 -- row triples (i, j, a_{i,j})
      2 -- column triples (i, j, a_{j,i})
      3 -- chevron triples 
           (i, k, a_{i,i+k}) if k >= 0
           (i, k, a_{i-k,i}) if k < 0
           i is the chevron, k is the offset
      4 -- custom coordinate, e.g., one could store (I, k, a_{i,j})
          where I is the front where a_{i,j} will be assembled
           and k is the offset into the vector that holds the 
           entries in the front
   inputMode -- mode for input
      1 --> indices only
      2 --> indices and entries
   maxnent  -- upper bound on the number of entries,
      also equal to the workspace, so if the assembly includes
      overlapping data, give enough elbow room for efficiency.
   maxnvector -- upper bound on the number of vectors to be supported. 
      this may not be known ahead of time (e.g., the number of vectors 
      may be the number of fronts which is not known before the
      ordering is done and front tree constructed).
 
   created -- 96jul05, cca
   ------------------------------------------------------------------
*/
void
DInpMtx_init (
  DInpMtx   *inpmtx,
  int       coordType,
  int       inputMode,
  int       maxnent,
  int       maxnvector
) ;
/*
   --------------------------
   change the coordinate type
 
   created -- 96jul05, cca
   --------------------------
*/
void
DInpMtx_changeCoordType (
   DInpMtx   *inpmtx,
   int       newType
) ;
/*
   -----------------------
   change the storage mode
 
   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_changeStorageMode (
   DInpMtx   *inpmtx,
   int       newMode
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in input.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------
   input a single entry in the matrix

   created -- 96jul05, cca
   ----------------------------------
*/
void
DInpMtx_inputEntry (
   DInpMtx   *inpmtx,
   int       row,
   int       col,
   double    entry
) ;
/*
   -------------------------
   input a row in the matrix

   created -- 96jul05, cca
   -------------------------
*/
void
DInpMtx_inputRow (
   DInpMtx   *inpmtx,
   int       row,
   int       rowsize,
   int       rowind[],
   double    rowent[]
) ;
/*
   ----------------------------
   input a column in the matrix

   created -- 96jul05, cca
   ----------------------------
*/
void
DInpMtx_inputColumn (
   DInpMtx   *inpmtx,
   int       col,
   int       colsize,
   int       colind[],
   double    colent[]
) ;
/*
   -----------------------------
   input a chevron in the matrix

   created -- 97feb08, cca
   -----------------------------
*/
void
DInpMtx_inputChevron (
   DInpMtx   *inpmtx,
   int       chv,
   int       chvsize,
   int       chvind[],
   double    chvent[]
) ;
/*
   -----------------------
   input a matrix

   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_inputMatrix (
   DInpMtx   *inpmtx,
   int       nrow,
   int       ncol,
   int       rowstride,
   int       colstride,
   int       rowind[],
   int       colind[],
   double    mtxent[]
) ;
/*
   -------------------------------------------------------------
   input a number of (row,column, entry) triples into the matrix
 
   created -- 97apr10, cca
   -------------------------------------------------------------
*/
void
DInpMtx_inputTriples (
   DInpMtx   *inpmtx,
   int       ntriples,
   int       rowids[],
   int       colids[],
   double    entries[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in mvm.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------
   compute a matrix-vector multiply
 
   created -- 96jul06, cca
   --------------------------------
*/
void
DInpMtx_mvm (
   DInpMtx   *inpmtx,
   DV        *yDV,
   double    alpha,
   DV        *xDV
) ;
/*
   ------------------------------------------
   compute a symmetric matrix-vector multiply
 
   created -- 97jan11, cca
   ------------------------------------------
*/
void
DInpMtx_symmetricMVM (
   DInpMtx   *inpmtx,
   DV        *yDV,
   double    alpha,
   DV        *xDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in profile.c ---------------------------------------
------------------------------------------------------------------------
*/
/*

------------------------------------------------------------------
   to fill xDV and yDV with a log10 profile of the magnitudes of
   the entries in the DInpMtx object. tausmall and tau big provide
   cutoffs within which to examine the entries. pnsmall and pnbig 
   are address to hold the number of entries smaller than tausmall,

   and larger than taubig, respectively.
 
   created -- 97feb14, cca

------------------------------------------------------------------
*/
void
DInpMtx_log10profile (
   DInpMtx    *inpmtx,
   int        npts,
   DV         *xDV,
   DV         *yDV,
   double     tausmall,
   double     taubig,
   int        *pnzero,
   int        *pnsmall,
   int        *pnbig
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------
   given the data is in raw triples,
   sort and compress the data
 
   created -- 96jul05, cca
   ---------------------------------
*/
void
DInpMtx_sortAndCompress (
   DInpMtx   *inpmtx
) ;
/*
   ----------------------------------------------------
   convert from sorted and compressed triples to vector
 
   created -- 96jul05, cca
   ----------------------------------------------------
*/
void
DInpMtx_convertToVectors (
   DInpMtx   *inpmtx
) ;
/*
   -------------------------
   drop off-diagonal entries
 
   created -- 97jan11, cca
   -------------------------
*/
void
DInpMtx_dropOffdiagonalEntries (
   DInpMtx   *inpmtx
) ;
/*
   ----------------------------------
   drop entries in the lower triangle
 
   created -- 97jan11, cca
   ----------------------------------
*/
void
DInpMtx_dropLowerTriangle (
   DInpMtx   *inpmtx
) ;
/*
   ----------------------------------
   drop entries in the upper triangle
 
   created -- 97jan11, cca
   ----------------------------------
*/
void
DInpMtx_dropUpperTriangle (
   DInpMtx   *inpmtx
) ;
/*
   -----------------------------------
   map entries into the lower triangle
 
   created -- 97jan11, cca
   -----------------------------------
*/
void
DInpMtx_mapToLowerTriangle (
   DInpMtx   *inpmtx
) ;
/*
   -----------------------------------
   map entries into the upper triangle
 
   created -- 97jan11, cca
   -----------------------------------
*/
void
DInpMtx_mapToUpperTriangle (
   DInpMtx   *inpmtx
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in permute.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   permute the entries
 
   created -- 96jul05, cca
   -----------------------
*/
void
DInpMtx_permute (
   DInpMtx   *inpmtx,
   int       rowOldToNew[],
   int       colOldToNew[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in extract.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
   purpose -- extract vectors from the matrix.
   note: if the coordinate type is rows, we extract full rows.
         if the coordinate type is columns, we extract full columns.
         if the coordinate type is chevrons, we extract full chevrons.
 
   created -- 97feb28, cca
   -------------------------------------------------------------------
*/
DInpMtx *
DInpMtx_extractVectors (
   DInpMtx   *inpmtx,
   IV        *idsIV
) ;
/*
   ------------------------------------------------
   purpose -- extract a submatrix.
 
   entry a(i,j) is extracted
   if rowmark[i] == rowtag and colmark[i] == coltag
 
   created -- 97mar01, cca
   ------------------------------------------------
*/
DInpMtx *
DInpMtx_extractSubmatrix (
   DInpMtx   *inpmtx,
   IV        *rowmarkIV,
   int       rowtag,
   IV        *colmarkIV,
   int       coltag
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in fullAdj.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   purpose -- to return the full, symmetric adjacency IVL object
              for the graph of A + A^T
 
   created -- 97oct22, cca
   -------------------------------------------------------------
*/
IVL *
DInpMtx_fullAdjacency (
   DInpMtx   *inpmtx
) ;
/*
   -------------------------------------------------------------
   purpose -- to return the full, symmetric adjacency IVL object
              for the graph of (A + B) + (A + B)^T
 
   created -- 97nov05, cca
   -------------------------------------------------------------
*/
IVL *
DInpMtx_fullAdjacency2 (
   DInpMtx   *inpmtxA,
   DInpMtx   *inpmtxB
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in adjForATA.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------
   return an IVL object that contains 
   the adjacency structure of A^TA.
 
   created -- 97nov17, cca
   ----------------------------------
*/
IVL *
DInpMtx_adjForATA (
   DInpMtx   *inpmtxA
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------------------
   purpose -- to read in a DInpMtx object from a file
 
   input --
 
      fn -- filename, must be *.inpmtxb or *.inpmtxf
 
   return value -- 1 if success, 0 if failure
 
   created -- 96jul05, cca
   --------------------------------------------------
*/
int
DInpMtx_readFromFile (
   DInpMtx   *inpmtx,
   char    *fn
) ;
/*
   --------------------------------------------------------
   purpose -- to read a DInpMtx object from a formatted file
 
   return value -- 1 if success, 0 if failure
 
   created -- 96jul05, cca
   --------------------------------------------------------
*/
int
DInpMtx_readFromFormattedFile (
   DInpMtx   *inpmtx,
   FILE    *fp
) ;
/*
   ----------------------------------------------------
   purpose -- to read a DInpMtx object from a binary file
 
   return value -- 1 if success, 0  if failure
 
   created -- 96jul05, cca
   ----------------------------------------------------
*/
int
DInpMtx_readFromBinaryFile (
   DInpMtx   *inpmtx,
   FILE    *fp
) ;
/*
   -------------------------------------------------------
   purpose -- to read a DInpMtx object from a TRANAIR file
 
   return value -- 1 if success, 0  if failure
 
   created -- 97feb14, cca
   -------------------------------------------------------
*/
int
DInpMtx_readFromTRANAIRfile ( 
   DInpMtx   *inpmtx, 
   FILE    *fp 
) ;
/*
   ----------------------------------------------
   purpose -- to write a DInpMtx object to a file
 
   input --
 
      fn -- filename
        *.inpmtxb -- binary
        *.inpmtxf -- formatted
        anything else -- for human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 96jul05, cca
   ----------------------------------------------
*/
int
DInpMtx_writeToFile (
   DInpMtx   *inpmtx,
   char    *fn
) ;
/*
   ------------------------------------------------------
   purpose -- to write a DInpMtx object to a formatted file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 96jul05, cca
   ------------------------------------------------------
*/
int
DInpMtx_writeToFormattedFile (
   DInpMtx   *inpmtx,
   FILE    *fp
) ;
/*
   ---------------------------------------------------
   purpose -- to write a DInpMtx object to a binary file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 96jul05, cca
   ---------------------------------------------------
*/
int
DInpMtx_writeToBinaryFile (
   DInpMtx    *inpmtx,
   FILE   *fp
) ;
/*
   ----------------------------------------------------
   purpose -- to write a DInpMtx object for a human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 96jul05, cca
   ----------------------------------------------------
*/
int
DInpMtx_writeForHumanEye (
   DInpMtx    *inpmtx,
   FILE   *fp
) ;
/*
   -----------------------------------------------------------
   purpose -- to write out the statistics for the DInpMtx object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 96jul05, cca
   -----------------------------------------------------------
*/
int
DInpMtx_writeStats (
   DInpMtx    *inpmtx,
   FILE   *fp
) ;
/*--------------------------------------------------------------------*/
