/*  util.c  */

#include "../DFrontMtx.h"

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   create and return the willUpdate vector
   willUpdate[J] == 'Y' --> this thread or process myid owns J
                            or a descendent with support with J
   willUpdate[J] == 'N' --> this thread or process myid does not own J
                            or any descendent with support with J
                  
   created -- 97jul03, cca
   -------------------------------------------------------------------
*/
char *
DFrontMtx_willUpdate (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         myid
) {
char   *willUpdate ;
int    ii, J, K, Kold, nfront, sizeJ ;
int    *frontOwners, *indicesJ, *vtxToFront ;
IVL    *symbfacIVL ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || frontOwnersIV == NULL || myid < 0 ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_willUpdate(%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, myid) ;
   exit(-1) ;
}
symbfacIVL = frontmtx->symbfacIVL ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
vtxToFront = ETree_vtxToFront(frontmtx->frontETree) ;
/*
   -------------------------------------
   create and fill the willUpdate vector
   -------------------------------------
*/
willUpdate = CVinit(nfront, 'N') ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( frontOwners[J] == myid ) {
      IVL_listAndSize(symbfacIVL, J, &sizeJ, &indicesJ) ;
      willUpdate[J] = 'Y' ;
      for ( ii = sizeJ - 1, Kold = -1 ; ii >= 0 ; ii-- ) {
         if ( (K = vtxToFront[indicesJ[ii]]) == J ) {
            break ;
         } else if ( K != Kold ) {
            Kold = K ;
            willUpdate[K] = 'Y' ;
         }
      }
   }
}
return(willUpdate) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   create and return the status vector
   status[J] == 'F' --> J is not on any active path for this thread 
   status[J] == 'W' --> J is on any active path for this thread 
                  
   created -- 97jul03, cca
   -------------------------------------------------------------------
*/
char *
DFrontMtx_status (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         myid
) {
char   *status ;
int    J, K, nfront ;
int    *frontOwners, *par ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || frontOwnersIV == NULL || myid < 0 ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_status(%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, myid) ;
   exit(-1) ;
}
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
par = ETree_par(frontmtx->frontETree) ;
/*
   -------------------------------------
   create and fill the willUpdate vector
   -------------------------------------
*/
status = CVinit(nfront, 'F') ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( frontOwners[J] == myid ) {
      for ( K = J ; K != -1 && status[K] == 'F' ; K = par[K] ) {
         status[K] = 'W' ;
      }
   }
}
return(status) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   create and return the nactiveChild vector.
   nactiveChild[J] contains the number of children 
   of J that belong to an active path
                  
   created -- 97jul03, cca
   -----------------------------------------------
*/
int *
DFrontMtx_nactiveChild (
   DFrontMtx   *frontmtx,
   char        *status,
   int         myid
) {
int    J, K, nfront ;
int    *nactiveChild, *par ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || status == NULL || myid < 0 ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_nativeChild(%p,%p,%d)"
           "\n bad input\n", frontmtx, status, myid) ;
   exit(-1) ;
}
nfront = frontmtx->nfront ;
par    = ETree_par(frontmtx->frontETree) ;
/*
   ---------------------------------------
   create and fill the nactiveChild vector
   ---------------------------------------
*/
nactiveChild = IVinit(nfront, 0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   if ( status[J] == 'W' && (K = par[J]) != -1 ) {
      nactiveChild[K]++ ;
   }
}
return(nactiveChild) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   create, initialize and return a Ideq object
   loaded with the leaves of the active paths.

   status vector should be already set via a call to DFrontMtx_status()
                  
   created -- 97jul03, cca
   --------------------------------------------------------------------
*/
Ideq *
DFrontMtx_setUpDequeue (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   char        *status,
   int         myid
) {
Ideq   *dequeue ;
int    I, J, npath ;
int    *fch, *frontOwners, *par, *sib ;
Tree   *tree ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontOwnersIV == NULL 
   || status == NULL || myid < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_setUpDequeue(%p,%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, status, myid) ;
   exit(-1) ;
}
frontOwners = IV_entries(frontOwnersIV) ;
tree        = ETree_tree(frontmtx->frontETree) ;
par         = tree->par ;
fch         = tree->fch ;
sib         = tree->sib ;
/*
   -------------------------------------
   count the number of paths in the tree
   -------------------------------------
*/
npath = 0 ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
   if ( frontOwners[J] == myid ) {
      for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
         if ( status[I] == 'W' ) {
            break ;
         }
      }
      if ( I == -1 ) {
         npath++ ;
      }
   }
}
/*
fprintf(stdout, "\n factor NPATH = %d", npath) ;
*/
dequeue = Ideq_new() ;
Ideq_resize(dequeue, npath) ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
   if ( frontOwners[J] == myid ) {
      for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
         if ( status[I] == 'W' ) {
            break ;
         }
      }
      if ( I == -1 ) {
         Ideq_insertAtTail(dequeue, J) ;
      }
   }
}
return(dequeue) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with postponed chevrons
                  
   created -- 97jul03, cca
   -----------------------------------------------
*/
DChvList *
DFrontMtx_postList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) {
char       *flags ;
DChvList   *postList ;
int        count, I, J, jthread, nchild, nfront, nthread ;
int        *counts, *fch, *frontOwners, *mark, *sib ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontOwnersIV == NULL 
   || lockflag < 0 || lockflag > 2 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_postList(%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, lockflag) ;
   exit(-1) ;
}
fch = ETree_fch(frontmtx->frontETree) ;
sib = ETree_sib(frontmtx->frontETree) ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
counts = IVinit(nfront+1, 0) ;
if ( lockflag > 0 ) {
   flags = CVinit(nfront+1, 'N') ;
} else {
   flags = NULL ;
}
nthread = 1 + IV_max(frontOwnersIV) ;
mark    = IVinit(nthread, -1) ;
/*
   --------------------
   loop over the fronts
   --------------------
*/
for ( J = 0 ; J < nfront ; J++ ) {
   count = nchild = 0 ;
   for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
      nchild++ ;
      jthread = frontOwners[I] ;
      if ( mark[jthread] != J ) {
         mark[jthread] = J ;
         count++ ;
      }
   }
   counts[J] = nchild ;
   if ( flags != NULL ) {
      if ( count > 1 ) {
         flags[J] = 'Y' ;
      } else {
         flags[J] = 'N' ;
      }
   }
}
count = nchild = 0 ;
for ( J = ETree_root(frontmtx->frontETree) ; J != -1 ; J = sib[J] ) {
   nchild++ ;
   jthread = frontOwners[J] ;
   if ( mark[jthread] != J ) {
      mark[jthread] = J ;
      count++ ;
   }
}
counts[nfront] = nchild ;
if ( flags != NULL ) {
   if ( count > 1 ) {
      flags[nfront] = 'Y' ;
   } else {
      flags[nfront] = 'N' ;
   }
}
/*
   -----------------------------------------
   create and initialize the DChvList object
   -----------------------------------------
*/
postList = DChvList_new() ;
DChvList_init(postList, nfront+1, counts, lockflag, flags) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(mark) ;
IVfree(counts) ;
if ( flags != NULL ) {
   CVfree(flags) ;
}

return(postList) ; }

/*--------------------------------------------------------------------*/
#define MYDEBUG 0
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with aggregate chevrons
                  
   created -- 97jul03, cca
   -----------------------------------------------
*/
DChvList *
DFrontMtx_aggregateList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) {
char       *flags ;
DChvList   *aggList ;
int        count, ii, I, J, jthread, K, myid, nfront, nthread, size ;
int        *counts, *frontOwners, *head, *indices, *link, 
           *mark, *offsets, *vtxToFront ;
IVL        *symbfacIVL ;

/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontOwnersIV == NULL 
   || lockflag < 0 || lockflag > 2 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_aggregateList(%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, lockflag) ;
   exit(-1) ;
}
symbfacIVL = frontmtx->symbfacIVL ;
vtxToFront = ETree_vtxToFront(frontmtx->frontETree) ;
IV_sizeAndEntries(frontOwnersIV, &nfront, &frontOwners) ;
nthread = 1 + IV_max(frontOwnersIV) ;
mark    = IVinit(nthread, -1) ;
head    = IVinit(nfront,  -1) ;
link    = IVinit(nfront,  -1) ;
offsets = IVinit(nfront,   0) ;
counts  = IVinit(nfront,   0) ;
if ( lockflag > 0 ) {
   flags = CVinit(nfront, 'N') ;
} else {
   flags = NULL ;
}
/*
   --------------------
   loop over the fronts
   --------------------
*/
for ( J = 0 ; J < nfront ; J++ ) {
   myid = frontOwners[J] ;
#if MYDEBUG > 0
   fprintf(stdout, "\n\n front %d, owner %d", J, myid) ;
   fflush(stdout) ;
#endif
   mark[myid] = J ;
   count = 0 ;
/*
   ---------------------------------------------------
   loop over all descendent fronts that might update J
   ---------------------------------------------------
*/
   while ( (I = head[J]) != -1 ) {
      head[J] = link[I] ;
      jthread = frontOwners[I] ;
#if MYDEBUG > 0
      fprintf(stdout, "\n descendent front %d, owner %d", I,
jthread) ;
      fflush(stdout) ;
#endif
      if ( mark[jthread] != J ) {
/*
         --------------------------------
         expect an aggregate from jthread
         --------------------------------
*/
#if MYDEBUG > 0
         fprintf(stdout, ", incoming aggregate") ;
         fflush(stdout) ;
#endif
         mark[jthread] = J ;
         count++ ;
      }
/*
      --------------------------------------------------
      link front I to next ancestor that it does not own
      --------------------------------------------------
*/
      IVL_listAndSize(symbfacIVL, I, &size, &indices) ;
      for ( ii = offsets[I] ; ii < size ; ii++ ) {
         if (  (K = vtxToFront[indices[ii]]) > J
            && frontOwners[K] != jthread ) {
#if MYDEBUG > 0
            fprintf(stdout, ", link to %d", K) ;
            fflush(stdout) ;
#endif
            offsets[I] = ii ;
            link[I] = head[K] ;
            head[K] = I ;
            break ;
         }
      }
   }
/*
   -------------------------------------
   set the number of incoming aggregates
   -------------------------------------
*/
   counts[J] = count ;
#if MYDEBUG > 0
   fprintf(stdout, "\n counts[%d] = %d", J, counts[J]) ;
   fflush(stdout) ;
#endif
/*
   ---------------------------------------------------
   set the flags to see if the list needs to be locked
   ---------------------------------------------------
*/
   if ( flags != NULL ) {
      if ( count > 1 ) {
         flags[J] = 'Y' ;
      } else {
         flags[J] = 'N' ;
      }
#if MYDEBUG > 0
      fprintf(stdout, ", flags[%d] = %c", J, flags[J]) ;
      fflush(stdout) ;
#endif
   }
/*
   --------------------------------------------------
   link front J to next ancestor that it does not own
   --------------------------------------------------
*/
   IVL_listAndSize(symbfacIVL, J, &size, &indices) ;
   for ( ii = 0 ; ii < size ; ii++ ) {
      if (  (K = vtxToFront[indices[ii]]) > J && frontOwners[K] != myid
) {
#if MYDEBUG > 0
         fprintf(stdout, ", linking to %d", K) ;
         fflush(stdout) ;
#endif
         offsets[J] = ii ;
         link[J] = head[K] ;
         head[K] = J ;
         break ;
      }
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n counts") ;
IVfprintf(stdout, nfront, counts) ;
fflush(stdout) ;
#endif
/*
   -----------------------------------------
   create and initialize the DChvList object
   -----------------------------------------
*/
aggList = DChvList_new() ;
DChvList_init(aggList, nfront, counts, lockflag, flags) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(counts) ;
IVfree(head) ;
IVfree(link) ;
IVfree(offsets) ;
IVfree(mark) ;
if ( flags != NULL ) {
   CVfree(flags) ;
}
return(aggList) ; }

#define MYDEBUG 0
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------
   create, initialize and return a DDenseMtxList object
   to deal with the dense matrices in a solve
                  
   created -- 97jul28, cca
   ----------------------------------------------------
*/
DDenseMtxList *
DFrontMtx_solveList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) {
DDenseMtxList   *matrixList ;
int             nfront ;
IV              *nchildIV ;
Tree            *tree ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL 
   || (frontOwnersIV == NULL && lockflag == 0) ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_solveList(%p,%p,%d)"
           "\n bad input\n", frontmtx, frontOwnersIV, lockflag) ;
   exit(-1) ;
}
tree       = frontmtx->frontETree->tree ;
nfront     = frontmtx->frontETree->nfront ;
matrixList = DDenseMtxList_new() ;
nchildIV   = Tree_nchildIV(tree) ;
if ( frontOwnersIV == NULL || lockflag == 0 ) {
/*
   ----------------------------------------------
   simple case, single owner or no lock on object
   ----------------------------------------------
*/
   DDenseMtxList_init(matrixList, 
                      nfront, IV_entries(nchildIV), 0, NULL) ;
} else {
/*
   -----------------------------------
   multiple owners with lock on object
   -----------------------------------
*/
   char   *flags ;
   int    count, I, J, jthread, nthread ;
   int    *fch, *mark, *owners, *sib ;

   fch = tree->fch ;
   sib = tree->sib ;
   IV_sizeAndEntries(frontOwnersIV, &nfront, &owners) ;
   nthread = 1 + IV_max(frontOwnersIV) ;
   flags   = CVinit(nfront, 'N') ;
   mark    = IVinit(nthread, -1) ;
   for ( J = 0 ; J < nfront ; J++ ) {
      count = 1 ;
      mark[owners[J]] = J ;
      for ( I = fch[J] ; I != -1 ; I = sib[I] ) {
         jthread = owners[I] ;
         if ( mark[jthread] != J ) {
            mark[jthread] = J ;
            count++ ;
         }
      }
      if ( count > 1 ) {
         flags[J] = 'Y' ;
      }
   }
   DDenseMtxList_init(matrixList, nfront, IV_entries(nchildIV), 
                      lockflag, flags) ;
   CVfree(flags) ;
   IVfree(mark) ;
}
IV_free(nchildIV) ;

return(matrixList) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   return a DChv object that will hold the biggest 
   front for a factorization without pivoting

   created -- 97aug21, cca
   -----------------------------------------------
*/
DChv *
DFrontMtx_biggestFront (
   DFrontMtx   *frontmtx
) {
DChv   *chv ;
int    *bndwghts, *nodwghts ;
int    J, maxnbytes, nBJ, nbytes, nfront, nJ, symflag ;
 
nfront   = frontmtx->nfront ;
nodwghts = ETree_nodwghts(frontmtx->frontETree) ;
bndwghts = ETree_bndwghts(frontmtx->frontETree) ;
if ( frontmtx->symmetryflag == 2 ) {
   symflag = 1 ;
} else {
   symflag = 0 ;
}
for ( J = 0, maxnbytes = 0 ; J < nfront ; J++ ) {
   nJ     = nodwghts[J] ;
   nBJ    = bndwghts[J] ;
   nbytes = DChv_nbytesNeeded(nJ, nBJ, nBJ, symflag) ;
   if ( nbytes > maxnbytes ) {
      maxnbytes = nbytes ;
   }
}
chv = DChv_new() ;
DChv_setNbytesInWorkspace(chv, maxnbytes) ;

return(chv) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   compute the inertia of a symmetric matrix

   fill *pnnegative with the number of negative eigenvalues of A
   fill *pnzero     with the number of zero eigenvalues of A
   fill *pnpositive with the number of positive eigenvalues of A

   created -- 97aug22, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_inertia (
   DFrontMtx   *frontmtx,
   int         *pnnegative,
   int         *pnzero,
   int         *pnpositive
) {
double   a, arm, b, c, mid, val ;
double   *diagent ;
int      ii, ipivot, J, nent, nfront, nJ, 
         nnegative, npivot, npositive, nzero ;
int      *pivotsizes ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontmtx->symmetryflag != 0
   || pnnegative == NULL || pnzero == NULL || pnpositive == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_inertia(%p,%p,%p,%p)"
           "\n bad input\n", 
           frontmtx, pnnegative, pnzero, pnpositive) ;
   fflush(stdout) ;
}
nfront = frontmtx->nfront ;
nnegative = nzero = npositive = 0 ;
if ( frontmtx->pivotingflag == 0 ) {
/*
   -----------
   no pivoting
   -----------
*/
   for ( J = 0 ; J < nfront ; J++ ) {
      DFrontMtx_diagEntries(frontmtx, J, &nJ, &diagent) ;
      for ( ii = 0 ; ii < nJ ; ii++ ) {
         if ( diagent[ii] < 0.0 ) {
            nnegative++ ;
         } else if ( diagent[ii] > 0.0 ) {
            npositive++ ;
         } else {
            nzero++ ;
         }
      }
   }
} else {
/*
   --------
   pivoting
   --------
*/
   for ( J = 0 ; J < nfront ; J++ ) {
      DFrontMtx_pivotsizes(frontmtx, J, &npivot, &pivotsizes) ;
      DFrontMtx_diagEntries(frontmtx, J, &nent, &diagent) ;
      for ( ipivot = ii = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            val = diagent[ii++] ;
            if ( val < 0.0 ) {
               nnegative++ ;
            } else if ( val > 0.0 ) {
               npositive++ ;
            } else {
               nzero++ ;
            }
         } else {
            a = diagent[ii++] ;
            b = diagent[ii++] ;
            c = diagent[ii++] ;
            mid = 0.5*(a + c) ;
            arm = sqrt(0.25*(a - c)*(a - c) + b*b) ;
            val = mid + arm ;
            if ( val < 0.0 ) {
               nnegative++ ;
            } else if ( val > 0.0 ) {
               npositive++ ;
            } else {
               nzero++ ;
            }
            val = mid - arm ;
            if ( val < 0.0 ) {
               nnegative++ ;
            } else if ( val > 0.0 ) {
               npositive++ ;
            } else {
               nzero++ ;
            }
        
         }
      }
   }
}
*pnnegative = nnegative ;
*pnzero     = nzero     ;
*pnpositive = npositive ;

return ; }

/*--------------------------------------------------------------------*/
