/*  storeFront.c  */

#include "../DFrontMtx.h"

#define MYDEBUG    0
#define CHECK_DROP 0

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   purpose -- to store the factor indices and entries from frontJ and
              link the front into the update list for its parent

   pivotsizesIV -- object that holds the number and sizes of the pivots
   droptol      -- drop tolerance for sparse factorizations

   created -- 97may23, cca
   --------------------------------------------------------------------
*/
void
DFrontMtx_storeFront (
   DFrontMtx   *frontmtx,
   DChv        *frontJ,
   IV          *pivotsizesIV,
   double      droptol
) {
double   *diagent, *lowerent, *upperent ;
int      ipivot, J, ncol, ncolind, nD, nentD, nentL, nentU, nindL,
         nindU, nL, npivot, nrow, nrowind, nU, pivotingflag, size, 
         sparsityflag, symmetryflag ;
int      *colind, *indices, *lowerind, *pivotind, *pivotsizes, 
         *rowind, *sizesL, *sizesU, *upperind ;
#if CHECK_DROP > 0
int      ii ;
#endif
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   first step: compute the number of indices and entries to be stored
   ------------------------------------------------------------------
*/
J = frontJ->id ;
DChv_dimensions(frontJ, &nD, &nL, &nU) ;
symmetryflag = frontmtx->symmetryflag ;
sparsityflag = frontmtx->sparsityflag ;
pivotingflag = frontmtx->pivotingflag ;
/*
   --------------------------
   set all dimensions to zero
   --------------------------
*/
npivot = nentD = nentL = nentU = nindL = nindU = nrowind = ncolind = 0 ;
pivotsizes = NULL ;
if ( nD > 0 ) {
/*
   ------------------------------------------
   some rows and columns have been eliminated
   ------------------------------------------
*/
   if ( symmetryflag == 0 && pivotingflag == 1 ) {
      npivot     = IV_size(pivotsizesIV) ;
      pivotsizes = IV_entries(pivotsizesIV) ;
      for ( ipivot = 0, nentD = 0 ; ipivot < npivot ; ipivot++ ) {
         nentD += (pivotsizes[ipivot]*(pivotsizes[ipivot]+1))/2 ;
      }
   } else {
      nentD = nD ;
   }
   if ( sparsityflag == 0 ) {
/*
      ------------
      dense fronts
      ------------
*/
      nentU = (nD*(nD+1))/2 + nD*nU - nentD ;
      if ( symmetryflag == 1 || symmetryflag == 2 ) {
         nentL = nentU ;
      }
   } else {
/*
      -------------
      sparse fronts
      -------------
*/
      nentU = DChv_countBigEntries(frontJ, npivot, pivotsizes, 
                                   3, droptol) ;
      nindU = nentU ;
      if ( symmetryflag == 1 || symmetryflag == 2 ) {
         nentL = DChv_countBigEntries(frontJ, npivot, pivotsizes, 
                                      1, droptol) ;
         nindL = nentL ;
      }
   }
}
if ( pivotingflag == 1 ) {
   ncolind = nD + nU ;
   if ( symmetryflag == 1 || symmetryflag == 2 ) {
      nrowind = nD + nL ;
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n storeFront, J = %d, nD = %d", J, nD) ;
fprintf(stdout, 
        "\n npivot = %d, nentD = %d, nentL = %d, nentU = %d"
        "\n nindL = %d, nindU = %d, nrowind = %d, ncolind = %d",
        npivot, nentD, nentL, nentU, nindL, nindU, nrowind, ncolind) ;
fflush(stdout) ;
#endif
/*--------------------------------------------------------------------*/
if ( pivotingflag == 1 || sparsityflag == 1 ) {
   if ( frontmtx->lock != NULL ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n front %d, locking", J) ;
      fflush(stdout) ;
#endif
      Lock_lock(frontmtx->lock) ;
      frontmtx->nlocks++ ;
#if MYDEBUG > 0
      fprintf(stdout, ", %d locks", frontmtx->nlocks) ;
      fflush(stdout) ;
#endif
   }
/*
   ------------------------
   need to allocate storage
   ------------------------
*/
   if ( nentD > 0 ) {
      DVL_setList(frontmtx->diagDVL,  J, nentD, NULL) ;
   }
   if ( nentU > 0 ) {
      DVL_setList(frontmtx->upperDVL, J, nentU, NULL) ;
   }
   if ( nentL > 0 ) {
      DVL_setList(frontmtx->lowerDVL, J, nentL, NULL) ;
   }
   if ( npivot > 0 ) {
      IVL_setList(frontmtx->pivotsIVL,  J, npivot, NULL) ;
   }
   if ( nindU > 0 ) {
      IVL_setList(frontmtx->upperIVL,  J, nindU + nD + nU, NULL) ;
   }
   if ( nindL > 0 ) {
      IVL_setList(frontmtx->lowerIVL,  J, nindL + nD + nL, NULL) ;
   }
   if ( nrowind > 0 ) {
      IVL_setList(frontmtx->rowadjIVL, J, nrowind, NULL) ;
   }
   if ( ncolind > 0 ) {
      IVL_setList(frontmtx->coladjIVL, J, ncolind, NULL) ;
   }
   if ( frontmtx->lock != NULL ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n front %d, unlocking", J) ;
      fflush(stdout) ;
#endif
      Lock_unlock(frontmtx->lock) ;
   }
}
/*--------------------------------------------------------------------*/
/*
   -------------------------------------
   get the base addresses of the storage 
   to hold entries and indices
   -------------------------------------
*/
if ( npivot > 0 ) {
   DFrontMtx_pivotsizes(frontmtx, J, &size, &pivotind) ;
   if ( size != npivot || pivotind == NULL ) {
      fprintf(stderr, "\n error in DFrontMtx_storeFront"
              "\n front %d : store %d pivots"
              "\n %d pivots, address %p",
              J, npivot, size, pivotind) ;
      exit(-1) ;
   }
}
if ( nrowind > 0 ) {
   DFrontMtx_rowIndices(frontmtx, J, &nrow, &rowind) ;
   if ( nrowind != nrow || rowind == NULL ) {
      fprintf(stderr, "\n error in DFrontMtx_storeFront"
              "\n front %d : store %d row indices"
              "\n %d row indices, address %p",
              J, nrowind, nrow, rowind) ;
      exit(-1) ;
   }
}
if ( ncolind > 0 ) {
   DFrontMtx_columnIndices(frontmtx, J, &ncol, &colind) ;
   if ( ncolind != ncol || colind == NULL ) {
      fprintf(stderr, "\n error in DFrontMtx_storeFront"
              "\n front %d : store %d column indices"
              "\n %d column indices, address %p",
              J, ncolind, ncol, colind) ;
      exit(-1) ;
   }
}
if ( nentD > 0 ) {
   DFrontMtx_diagEntries(frontmtx, J, &size, &diagent) ;
   if ( size != nentD || diagent == NULL ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_storeFront()"
             "\n front %d : store diagonal entries"
              "\n nentD = %d, size = %d, diagent = %p",
              J, nentD, size, diagent) ;
      exit(-1) ;
   }
}
if ( nentU > 0 ) {
   DFrontMtx_upperEntries(frontmtx, J, &size, &upperent) ;
   if ( size != nentU || upperent == NULL ) {
      fprintf(stderr,
              "\n fatal error in DFrontMtx_storeFront()"
              "\n front %d : store lower entries"
              "\n nentU = %d, size = %d, upperent = %p",
              J, nentU, size, upperent) ;
      exit(-1) ;
   }
}
if ( nentL > 0 ) {
   DFrontMtx_lowerEntries(frontmtx, J, &size, &lowerent) ;
   if ( size != nentL || lowerent == NULL ) {
      fprintf(stderr,
              "\n fatal error in DFrontMtx_storeFront()"
              "\n front %d : store lower entries"
              "\n nentL = %d, size = %d, lowerent = %p",
              J, nentL, size, lowerent) ;
      exit(-1) ;
   }
}
if ( nindU > 0 ) {
   DFrontMtx_upperSparsityInfo(frontmtx, J,
                               &ncol, &sizesU, &upperind) ;
   if ( ncol != nD + nU || sizesU == NULL || upperind == NULL ) {
      fprintf(stderr,
         "\n fatal error in DFrontMtx_storeFront()"
         "\n front %d : get upper sparsity info"
         "\n ncol = %d, nD + nU = %d, sizesU = %p, upperind = %p",
         J, ncol, nD + nU, sizesU, upperind) ;
      exit(-1) ;
   }
}
if ( nindL > 0 ) {
   DFrontMtx_lowerSparsityInfo(frontmtx, J,
                               &nrow, &sizesL, &lowerind) ;
   if ( nrow != nD + nL || sizesL == NULL || lowerind == NULL ) {
      fprintf(stderr,
         "\n fatal error in DFrontMtx_storeFront()"
         "\n front %d : get lower sparsity info"
         "\n nrow = %d, nD + nL = %d, sizesL = %p, lowerind = %p",
         J, nrow, nD + nL, sizesL, lowerind) ;
      exit(-1) ;
   }
}
/*--------------------------------------------------------------------*/
/*
   -----------------------------
   now store indices and entries
   -----------------------------
*/
if ( pivotingflag == 1 ) {
/*
   ---------------------------------------
   pivoting is enabled, set the front size
   ---------------------------------------
*/
   DFrontMtx_setFrontSize(frontmtx, J, frontJ->nD) ;
}
if ( npivot > 0 ) {
/*
   ---------------------
   store the pivot sizes
   ---------------------
*/
   IVcopy(npivot, pivotind, pivotsizes) ;
}
if ( nrowind > 0 ) {
/*
   -----------------
   store row indices
   -----------------
*/
   DChv_rowIndices(frontJ, &nrow, &indices) ;
   IVcopy(nrow, rowind, indices) ;
}
if ( ncolind > 0 ) {
/*
   --------------------
   store column indices
   --------------------
*/
   DChv_columnIndices(frontJ, &ncol, &indices) ;
   IVcopy(ncol, colind, indices) ;
}
/*
   --------------------------
   store the diagonal entries
   --------------------------
*/
if ( nentD > 0 ) {
   DChv_copyEntriesToVector(frontJ, npivot, pivotsizes, nentD,
                            diagent, 2, 0) ;
}
if ( sparsityflag == 0 ) {
/*
   ------------
   dense fronts
   ------------
*/
   if ( nentU > 0 ) {
      DChv_copyEntriesToVector(frontJ, npivot, pivotsizes, nentU,
                               upperent, 3, 1) ;
#if CHECK_DROP > 0
      for ( ii = 0 ; ii < nentU ; ii++ ) {
         if ( fabs(upperent[ii]) < droptol ) {
            upperent[ii] = 0.0 ;
         }
      }
#endif
   }
   if ( nentL > 0 ) {
      DChv_copyEntriesToVector(frontJ, npivot, pivotsizes, nentL,
                               lowerent, 1, 0) ;
#if CHECK_DROP > 0
      for ( ii = 0 ; ii < nentL ; ii++ ) {
         if ( fabs(lowerent[ii]) < droptol ) {
            lowerent[ii] = 0.0 ;
         }
      }
#endif
   }
} else {
/*
   -------------
   sparse fronts
   -------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n front %d, nentD = %d, nentL = %d, nentU = %d",
        frontJ->id, nentD, nentL, nentU) ;
fflush(stdout) ;
#endif
   if ( nentU > 0 ) {
      size = DChv_copyBigEntriesToVector(frontJ, npivot, pivotsizes, 
                           sizesU, upperind, upperent, 3, 1, droptol) ;
      if ( size != nentU ) {
         fprintf(stderr, 
                 "\n fatal error trying to store big upper entries"
                 "\n front %d, nentU = %d from count, size = %d stored",
                 frontJ->id, nentU, size) ;
         exit(-1) ;
      }
   }
   if ( nentL > 0 ) {
      size = DChv_copyBigEntriesToVector(frontJ, npivot, pivotsizes, 
                           sizesL, lowerind, lowerent, 1, 0, droptol) ;
      if ( size != nentL ) {
         fprintf(stderr, 
                 "\n fatal error trying to store big lower entries"
                 "\n front %d, nentL = %d from count, size = %d stored",
                 frontJ->id, nentL, size) ;
         exit(-1) ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
