/*  solveMT.c  */

#include "../DFrontMtx.h"
#include "../../Ideq.h"
#include "../../timings.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   --------------
   static methods
   --------------
*/
void * DFrontMtx_workerSolve ( void *arg ) ;
static void loadNearestOwnedDescendents ( Tree *tree, int seed,
   int owners[], int myid, Ideq *dequeue, int msglvl, FILE *msgFile ) ;
/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   a multithreaded solve of a linear system A X = B 

   frontmtx -- object that stores the factors of A
   solDA2   -- object that stores the solution X
   rhsDA2   -- object that stores the solution B
   manager  -- object that manages working data
   ownersIV -- object that stores the map from fronts to threads
   cpus     -- vector of cpu time breakdowns
      cpus[0] -- initialize rhs matrices
      cpus[1] -- load rhs matrices with rhs 
      cpus[2] -- assemble from children and parent
      cpus[3] -- solve and update
      cpus[4] -- store entries
      cpus[5] -- link and free matrices
      cpus[6] -- miscellaneous
      cpus[7] -- total time
   msglvl   -- message level
   msgFile  -- message file

   created -- 97jun27, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_MT_solve (
   DFrontMtx          *frontmtx,
   DA2                *solDA2,
   DA2                *rhsDA2,
   DDenseMtxManager   *manager,
   IV                 *ownersIV,
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) {
char            buffer[20] ;
DDenseMtx       **p_mtxBJ, **p_mtxJ ;
DDenseMtxList   *matrixList ;
DSolveData      *data, *dataObjects ;
double          t1, t2 ;
FILE            *fp ;
int             ii, J, K, myid, nfront, nthread, ownerJ, rc ;
int             *ndescLeft, *par, *owners ;
Tree            *tree ;
#if THREAD_TYPE == TT_POSIX
pthread_t       *tids ;
#endif
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || solDA2 == NULL || rhsDA2 == NULL
   || manager == NULL || ownersIV == NULL || cpus == NULL 
   || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MT_solve()"
           "\n frontmtx = %p, solDA2 = %p, rhsDA2 = %p"
           "\n manager = %p, ownersIV = %p"
           "\n cpus = %p, msglvl = %d, msgFile = %p"
           "\n\n bad input\n", frontmtx, solDA2, rhsDA2, manager,
           ownersIV, cpus, msglvl, msgFile) ; 
   exit(-1) ;
}
MARKTIME(t1) ;
IV_sizeAndEntries(ownersIV, &nfront, &owners) ;
nthread = 1 + IV_max(ownersIV) ;
tree    = frontmtx->frontETree->tree ;
par     = tree->par ;
/*
   ----------------------
   set up the list object
   ----------------------
*/
matrixList = DFrontMtx_solveList(frontmtx, ownersIV, 1) ;
/*
   ----------------------------------------
   allocate the working storage of pointers
   ----------------------------------------
*/
ALLOCATE(p_mtxJ,  struct _DDenseMtx *, nfront) ;
ALLOCATE(p_mtxBJ, struct _DDenseMtx *, nfront) ;
for ( J = 0 ; J < nfront ; J++ ) {
   p_mtxJ[J] = p_mtxBJ[J] = NULL ;
}
/*
   ------------------------------------
   compute the local ndescLeft[] vector
   that synchronizes the forward solve
   ------------------------------------
*/
ndescLeft = IVinit(nfront, 0) ;
for ( J = 0 ; J < nfront ; J++ ) {
   ownerJ = owners[J] ;
   for ( K = par[J] ; K != -1 ; K = par[K] ) {
      if ( owners[K] == ownerJ ) {
         ndescLeft[K]++ ;
         break ;
      }
   }
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n ndescLeft") ;
   IVfprintf(msgFile, nfront, ndescLeft) ;
   fflush(msgFile) ;
}
/*
   ----------------------------------
   allocate the local data structures
   ----------------------------------
*/
ALLOCATE(dataObjects, struct _DSolveData, nthread) ;
for ( myid = 0, data = dataObjects ; myid < nthread ; myid++, data++ ) {
   DSolveData_init(data, frontmtx, solDA2, rhsDA2, ownersIV, 
                   manager, matrixList, p_mtxJ, p_mtxBJ, ndescLeft) ; 
   if ( msglvl > 0 ) {
      sprintf(buffer, "solve.res.%d", myid) ;
      if ( (fp = fopen(buffer, "w")) == NULL ) {
         fprintf(stderr, "\n fatal error, unable to open file %s",
                 buffer) ;
         exit(-1) ;
      }
      DSolveData_setInfo(data, myid, msglvl, fp) ;
   } else {
      DSolveData_setInfo(data, myid, msglvl, NULL) ;
   }
}
MARKTIME(t2) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n CPU %8.3f : initialization time", t2 - t1) ;
   fflush(msgFile) ;
}
/*
   -------------------
   set the concurrency
   -------------------
*/
#if THREAD_TYPE == TT_SOLARIS
MARKTIME(t1) ;
thr_setconcurrency(nthread) ;
MARKTIME(t2) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n CPU %8.3f : set concurrency time", t2 - t1) ;
}
#endif
/*
#####   NOTE: for SGI machines, this command must be present
#####         for the thread scheduling to be efficient.
#####         this is NOT a POSIX call, but SGI needs it anyway
#if THREAD_TYPE == TT_POSIX
pthread_setconcurrency(nthread) ;
#endif
*/
/*
   ------------------
   create the threads
   ------------------
*/
MARKTIME(t1) ;
#if THREAD_TYPE == TT_SOLARIS
for ( myid = 0, data = dataObjects ; 
      myid < nthread - 1 ;
      myid++, data++ ) {
   rc = thr_create(NULL, 0, DFrontMtx_workerSolve, data, 0, NULL) ;
   if ( rc != 0 ) {
      fprintf(stderr, 
              "\n fatal error, myid = %d, rc = %d from thr_create",
             myid, rc) ;
      exit(-1) ;
   }
}
#endif
#if THREAD_TYPE == TT_POSIX
{
pthread_attr_t   attr ;
pthread_attr_init(&attr) ;
pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM) ;
ALLOCATE(tids, pthread_t, nthread) ;
for ( myid = 0, data = dataObjects ; myid < nthread ; myid++, data++ ) {
   rc = pthread_create(&tids[myid], &attr, 
                       DFrontMtx_workerSolve, data) ;
   if ( rc != 0 ) {
      fprintf(stderr,
              "\n fatal error, myid = %d, rc = %d from pthread_create",
              myid, rc) ;
      exit(-1) ;
   }
}
}
#endif
/*
for ( myid = 0, data = dataObjects ; 
      myid < nthread - 1 ; 
      myid++, data++ ) {
   rc = thr_create(NULL, 0, DFrontMtx_workerSolve, data, 0, NULL) ;
   if ( rc != 0 ) {
      fprintf(stderr, 
              "\n fatal error, myid = %dm rc = %d from thr_create",
              myid, rc) ;
      exit(-1) ;
   }
}
*/
MARKTIME(t2) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n CPU %8.3f : thread creation time", t2 - t1) ;
   fflush(msgFile) ;
}
/*
   ----------------
   join the threads
   ----------------
*/
MARKTIME(t1) ;
/*
DFrontMtx_workerSolve(data) ;
for ( myid = 0 ; myid < nthread - 1 ; myid++ ) {
   thr_join(0, 0, 0) ;
   if ( msglvl > 0 ) {
      fprintf(msgFile, "\n    one thread joined") ;
      fflush(msgFile) ;
   }
}
*/
#if THREAD_TYPE == TT_SOLARIS
DFrontMtx_workerSolve(data) ;
for ( myid = 0 ; myid < nthread - 1 ; myid++ ) {
   thr_join(0, 0, 0) ;
}
#endif
#if THREAD_TYPE == TT_POSIX
{
void       *status ;
for ( myid = 0 ; myid < nthread ; myid++ ) {
   pthread_join(tids[myid], &status) ;
}
}
#endif
MARKTIME(t2) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n CPU %8.3f : thread join time", t2 - t1) ;
   fflush(msgFile) ;
}
if ( msglvl > 1 ) {
   DDenseMtxList_writeForHumanEye(matrixList, msgFile) ;
   fflush(msgFile) ;
}
/*
   ---------------------
   free the working data
   ---------------------
*/
MARKTIME(t1) ;
DVzero(8, cpus) ;
for ( myid = 0, data = dataObjects ; myid < nthread ; myid++, data++ ) {
   for ( ii = 0 ; ii < 8 ; ii++ ) {
      cpus[ii] += data->cpus[ii] ;
   }
   DSolveData_clearData(data) ;
}
FREE(dataObjects) ;
FREE(p_mtxJ) ;
FREE(p_mtxBJ) ;
IVfree(ndescLeft) ;
DDenseMtxList_free(matrixList) ;
MARKTIME(t2) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n CPU %8.3f : free working data", t2 - t1) ;
   fflush(msgFile) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   worker method to solve the linear system

   created -- 97jun27, cca
   ----------------------------------------
*/
void *
DFrontMtx_workerSolve (
   void   *arg
) {
char               *status ;
DA2                *rhsDA2, *solDA2 ;
DDenseMtx          *firstI, *mtx, *mtxBJ, *mtxJ, *mtxK ;
DDenseMtx          **p_mtxJ, **p_mtxBJ ;
DDenseMtxList      *matrixList ;
DDenseMtxManager   *manager ;
DFrontMtx          *frontmtx ;
double             t0, t1, t2, t3 ;
double             *cpus ;
DSolveData         *data ;
FILE               *msgFile ;
Ideq               *dequeue ;
int                I, J, K, msglvl, myid, nbytesNeeded, nDJ, ncolJ, 
                   nfront, nrhs, nrowJ ;
int                *fch, *ndescLeft, *owners, *par, *rowindJ, *sib ;
IV                 *ownersIV ;
Tree               *tree ;
/*
   -------------------------------
   extract pointers and dimensions
   -------------------------------
*/
MARKTIME(t0) ;
data       = (DSolveData *) arg ;
cpus       = data->cpus ;
frontmtx   = data->frontmtx ;
nfront     = frontmtx->nfront ;
tree       = frontmtx->frontETree->tree ;
fch        = tree->fch ;
par        = tree->par ;
sib        = tree->sib ;
solDA2     = data->solDA2   ;
rhsDA2     = data->rhsDA2   ;
nrhs       = solDA2->n2     ;
ownersIV   = data->ownersIV ;
owners     = IV_entries(ownersIV) ;
manager    = data->manager  ;
matrixList = data->matrixList ;
p_mtxJ     = data->p_mtxJ     ;
p_mtxBJ    = data->p_mtxBJ    ;
ndescLeft  = data->ndescLeft  ;
myid       = data->myid ;
msglvl     = data->msglvl ;
msgFile    = data->msgFile ;
/*
   ----------------------------------------
   set up the dequeue for the forward solve
   ----------------------------------------
*/
status = DFrontMtx_status(frontmtx, ownersIV, myid) ;
dequeue = DFrontMtx_setUpDequeue(frontmtx, ownersIV, status, myid) ;
/*
   -------------
   forward solve
   -------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n\n thread %d : starting forward solve", myid) ;
fflush(stdout) ;
#endif
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n\n ### forward solve, checking out J = %d, status %c", 
              J, status[J]) ;
      fflush(msgFile) ;
   }
   if ( status[J] == 'W' ) {
/*
      ------------------------------------------
      wake up front J, initialize mtxJ and mtxBJ
      ------------------------------------------
*/
      MARKTIME(t1) ;
      DFrontMtx_forwInit(frontmtx, J, nrhs, manager, &mtxJ, &mtxBJ) ;
      MARKTIME(t2) ;
      cpus[0] += t2 - t1 ;
      if ( msglvl > 2 ) {
         if ( mtxJ != NULL ) {
            fprintf(msgFile, "\n after initialization, mtxJ") ;
            DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
            fflush(msgFile) ;
         }
         if ( mtxBJ != NULL ) {
            fprintf(msgFile, "\n after initialization, mtxBJ") ;
            DDenseMtx_writeForHumanEye(mtxBJ, msgFile) ;
            fflush(msgFile) ;
         }
      }
      p_mtxJ[J]  = mtxJ  ;
      p_mtxBJ[J] = mtxBJ ;
      if ( mtxJ != NULL ) {
/*
         --------------------------------
         load the right hand side entries
         --------------------------------
*/
         MARKTIME(t1) ;
         DFrontMtx_forwLoadRHS(mtxJ, rhsDA2) ;
         MARKTIME(t2) ;
         cpus[1] += t2 - t1 ;
         if ( msglvl > 2 ) {
            fprintf(msgFile, "\n mtxJ after loading rhs") ;
            DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
            fflush(msgFile) ;
         }
      }
      status[J]  = 'R' ;
   }
   if ( status[J] == 'R' ) {
/*
      --------------
      front is ready
      --------------
*/
      mtxJ  = p_mtxJ[J]  ;
      mtxBJ = p_mtxBJ[J] ;
      if ( DDenseMtxList_isCountZero(matrixList, J) == 1 ) {
/*
         ---------------------------------
         no more updates from any children 
         remain to be placed on the list.
         ---------------------------------
*/
         if ( DDenseMtxList_isListNonempty(matrixList, J) == 1 ) {
/*
            ----------------------------------
            assemble all updates from children
            ----------------------------------
*/
            MARKTIME(t1) ;
            firstI = DDenseMtxList_getList(matrixList, J) ;
            DFrontMtx_forwLoadFromChildren(mtxJ, mtxBJ, firstI) ;
            if ( msglvl > 2 ) {
               if ( mtxJ != NULL ) {
                  fprintf(msgFile, 
                          "\n\n after assembling final updates, mtxJ") ;
                  DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
                  fflush(msgFile) ;
               }
               if ( mtxBJ != NULL ) {
                  fprintf(msgFile, 
                         "\n\n after assembling final updates, mtxBJ") ;
                  DDenseMtx_writeForHumanEye(mtxBJ, msgFile) ;
                  fflush(msgFile) ;
               }
            }
            MARKTIME(t2) ;
            cpus[2] += t2 - t1 ;
            MARKTIME(t1) ;
            DDenseMtxManager_releaseListOfObjects(manager, firstI) ;
            MARKTIME(t2) ;
            cpus[5] += t2 - t1 ;
         }
         if ( mtxJ != NULL ) {
/*
            ------------------------------
            do the forward solve operation
            ------------------------------
*/
            MARKTIME(t1) ;
            DFrontMtx_forwSolveAndUpdate(frontmtx, mtxJ, mtxBJ) ;
            MARKTIME(t2) ;
            cpus[3] += t2 - t1 ;
            if ( msglvl > 2 ) {
               if ( mtxJ != NULL ) {
                  fprintf(msgFile, "\n\n mtxJ after solve and update") ;
                  DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
                  fflush(msgFile) ;
               }
               if ( mtxBJ != NULL ) {
                  fprintf(msgFile, "\n\n mtxBJ after solve and update");
                  DDenseMtx_writeForHumanEye(mtxBJ, msgFile) ;
                  fflush(msgFile) ;
               }
            }
/*
            -----------------------------
            front is done, store solution
            -----------------------------
*/
            MARKTIME(t1) ;
            DFrontMtx_forwStore(rhsDA2, mtxJ) ;
            MARKTIME(t2) ;
            cpus[4] += t2 - t1 ;
            MARKTIME(t1) ;
            DDenseMtxManager_releaseObject(manager, mtxJ) ;
            MARKTIME(t2) ;
            cpus[5] += t2 - t1 ;
         }
         if ( mtxBJ != NULL ) {
/*
            ----------------------
            link updates to parent
            ----------------------
*/
            MARKTIME(t1) ;
            DDenseMtxList_addObjectToList(matrixList, mtxBJ, par[J]) ;
            MARKTIME(t2) ;
            cpus[5] += t2 - t1 ;
         }
         p_mtxJ[J] = p_mtxBJ[J] = NULL ;
         status[J] = 'F' ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n\n front %d complete", J) ;
            fflush(msgFile) ;
         }
#if MYDEBUG > 1
         fprintf(stdout, "\n thread %d : forward solve on %d finished",
                 myid, J) ;
         fflush(stdout) ;
#endif
/*
         --------------------------------
         look for the next owned ancestor
         --------------------------------
*/
         K = par[J] ;
         while ( K != -1 && owners[K] != myid ) {
            K = par[K] ;
         }
         if ( K != -1 && --ndescLeft[K] == 0 ) {
/*
            -----------------------------------------------------------
            next owned ancestor exists and all descendents are finished 
            (in a local sense), place K on the head of the dequeue
            -----------------------------------------------------------
*/
            if ( msglvl > 1 ) {
               fprintf(msgFile, 
                  "\n\n placing next owned ancestor %d on dequeue", K) ;
               fflush(msgFile) ;
            }
            Ideq_insertAtHead(dequeue, K) ;
         }
      }
   }
   if ( status[J] != 'F' ) {
/*
      ---------------------------------------------------
      front is not done, place on the tail of the dequeue
      ---------------------------------------------------
*/
      if ( msglvl > 1 ) {
         fprintf(msgFile, 
            "\n\n front %d not finished, placing on tail of queue", J) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtTail(dequeue, J) ;
   }
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after forward solve") ;
   DA2_writeForHumanEye(rhsDA2, msgFile) ;
   fflush(msgFile) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n thread %d : end of forward solve", myid) ;
fflush(stdout) ;
#endif
/*
   ------------------------------------------------------------
   load the dequeue with the fronts to start the backward solve
   ------------------------------------------------------------
*/
loadNearestOwnedDescendents(tree, -1, owners, myid, dequeue,
                            msglvl, msgFile) ;
CVfill(nfront, status, 'W') ;
/*
   --------------
   backward solve
   --------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n\n thread %d : starting backward solve", myid) ;
fflush(stdout) ;
#endif
while ( (J = Ideq_removeFromHead(dequeue)) != -1 ) {
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
             "\n\n ### backward solve, checking out J = %d, status %c", 
             J, status[J]) ;
      fflush(msgFile) ;
   }
   if ( status[J] == 'W' ) {
/*
      ------------------------------
      initialize the solution object
      ------------------------------
*/
      MARKTIME(t1) ;
      mtxJ = DFrontMtx_backInit(frontmtx, J, nrhs, manager) ;
      MARKTIME(t2) ;
      cpus[0] += t2 - t1 ;
      if ( (nDJ = DFrontMtx_frontSize(frontmtx, J)) > 0 ) {
/*
      --------------------------------
      load the right hand side entries
      --------------------------------
*/
         MARKTIME(t1) ;
         DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
         DFrontMtx_backLoadSolution(mtxJ, nDJ, rowindJ, rhsDA2) ;
         MARKTIME(t2) ;
         cpus[1] += t2 - t1 ;
      }
      p_mtxJ[J] = mtxJ ;
      status[J] = 'R' ;
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n after loading rhs, mtxJ") ;
         DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
         fflush(msgFile) ;
      }
   }
   if ( status[J] == 'R' ) {
      if (  (K = par[J]) == -1 
         || DDenseMtxList_isListNonempty(matrixList, J) == 1 ) {
/*
         ----------------------------------------------
         J is a root or entries from parent are present
         ----------------------------------------------
*/
         mtxJ = p_mtxJ[J] ;
         nDJ  = DFrontMtx_frontSize(frontmtx, J) ;
         if ( K != -1 ) {
            MARKTIME(t1) ;
            mtxK = DDenseMtxList_getList(matrixList, J) ;
            DFrontMtx_backLoadFromParent(mtxJ, nDJ, mtxK) ;
            MARKTIME(t2) ;
            cpus[2] += t2 - t1 ;
            MARKTIME(t1) ;
            DDenseMtxManager_releaseObject(manager, mtxK) ;
            MARKTIME(t2) ;
            cpus[5] += t2 - t1 ;
            if ( msglvl > 2 ) {
               fprintf(msgFile, 
                       "\n\n mtxJ after loading parent's entries") ;
               DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
               fflush(msgFile) ;
            }
         }
         if ( nDJ > 0 ) {
/*
            -------------------------------------
            perform the update and backward solve
            -------------------------------------
*/
            MARKTIME(t1) ;
            DFrontMtx_backSolveAndUpdate(frontmtx, mtxJ) ;
            MARKTIME(t2) ;
            cpus[3] += t2 - t1 ;
            if ( msglvl > 2 ) {
               fprintf(msgFile, "\n\n after solve and update, mtxJ") ;
               DDenseMtx_writeForHumanEye(mtxJ, msgFile) ;
               fflush(msgFile) ;
            }
/*
            --------------------------
            store the solution entries
            --------------------------
*/
            MARKTIME(t1) ;
            DFrontMtx_backStore(mtxJ, nDJ, solDA2) ;
            MARKTIME(t2) ;
            cpus[4] += t2 - t1 ;
         }
         MARKTIME(t1) ;
         if ( (I = fch[J]) != -1 ) {
/*
            -----------------------------------
            place a copy of the solution matrix 
            into each list for the children
            -----------------------------------
*/
            ncolJ = mtxJ->nrow ;
            for ( I = sib[I] ; I != -1 ; I = sib[I] ) {
               nbytesNeeded = DDenseMtx_nbytesNeeded(ncolJ, nrhs) ;
               mtx = DDenseMtxManager_newObjectOfSizeNbytes(manager,
                                                        nbytesNeeded) ;
               DDenseMtx_init(mtx, J, -1, ncolJ, nrhs, 1, ncolJ) ;
               DVcopy(ncolJ*nrhs, mtx->entries, mtxJ->entries) ;
               DDenseMtxList_addObjectToList(matrixList, mtx, I) ;
            }
            DDenseMtxList_addObjectToList(matrixList, mtxJ, fch[J]) ;
         } else {
/*
            -------------------------------
            no children, release the matrix
            -------------------------------
*/
            DDenseMtxManager_releaseObject(manager, mtxJ) ;
         }
         MARKTIME(t2) ;
         cpus[5] += t2 - t1 ;
         if ( msglvl > 1 ) {
            fprintf(msgFile, "\n\n entries stored and linked") ;
            fflush(msgFile) ;
         }
/*
         ----------------------------------------------------
         place the nearest owned descendents into the dequeue
         ----------------------------------------------------
*/
         loadNearestOwnedDescendents(tree, J, owners, myid, dequeue,
                                     msglvl, msgFile) ;
         status[J] = 'F' ;
/*
         p_mtxJ[J] = NULL ;
*/
#if MYDEBUG > 0
         fprintf(stdout, "\n thread %d : backward solve on %d finished",
                 myid, J) ;
         fflush(stdout) ;
#endif
      }
   } 
   if ( status[J] != 'F' ) {
/*
      ---------------------------------------------------------
      front is not yet complete, put on the tail of the dequeue
      ---------------------------------------------------------
*/
      Ideq_insertAtTail(dequeue, J) ;
   }
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
Ideq_free(dequeue) ;
CVfree(status) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n thread %d : leaving solve", myid) ;
fflush(stdout) ;
#endif
MARKTIME(t3) ;
cpus[7] = t3 - t0 ;
cpus[6] = cpus[7] - 
         (cpus[0] + cpus[1] + cpus[2] + cpus[3] + cpus[4] + cpus[5]) ;
#if MYDEBUG > 0
fprintf(stdout, 
     "\n\n thread %d : %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f",
       myid, cpus[0], cpus[1], cpus[2], cpus[3], 
       cpus[4], cpus[5], cpus[6], cpus[7]) ;
fflush(stdout) ;
#endif

return(NULL) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   load the dequeue with the closest descendents 
   of seed that are owned by thread myid

   created -- 97jun27, cca
   ---------------------------------------------
*/
static void
loadNearestOwnedDescendents (
   Tree   *tree,
   int    seed,
   int    owners[],
   int    myid,
   Ideq   *dequeue,
   int    msglvl,
   FILE   *msgFile
) {
int   I ;
int   *fch = tree->fch ;
int   *par = tree->par ;
int   *sib = tree->sib ;

if ( seed != -1 ) {
   I = fch[seed] ;
} else {
   I = tree->root ;
}
while ( I != -1 ) {
   if ( owners[I] == myid ) {
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n loading descendent %d onto queue", I) ;
         fflush(msgFile) ;
      }
      Ideq_insertAtHead(dequeue, I) ;
      while ( sib[I] == -1 && par[I] != seed ) {
         I = par[I] ;
      }
      I = sib[I] ;
   } else {
      if ( fch[I] != -1 ) {
         I = fch[I] ;
      } else {
         while ( sib[I] == -1 && par[I] != seed ) {
            I = par[I] ;
         }
         I = sib[I] ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
