/*  postponed.c  */

#include "../DFrontMtx.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- to assemble any postponed data into frontJ

   frontJ  -- pointer to DChv objec that contains current front
   chvlist -- pointer to a DChvList object that handles the
              lists of postponed DChv objects
   manager -- pointer to a DChvManager object for the list
              of free DChv objects
   pndelay -- pointer to address to contain the # of delayed rows 
              and columns that were assembled into the front

   return value -- pointer to DChv object that contains the new front

   created -- 97may23, cca
   ------------------------------------------------------------------
*/
DChv *
DFrontMtx_assemblePostponedData (
   DFrontMtx     *frontmtx,
   DChv          *frontJ,
   DChvList      *chvlist,
   DChvManager   *manager,
   int           *pndelay
) {
DChv   *child, *child2, *firstchild, *newfrontJ, *nextchild, *prev ;
int    nbytes, nDnew ;

if ( (firstchild = DChvList_getList(chvlist, frontJ->id)) == NULL ) {
/*
   -------------------------------------
   quick return, no children to assemble
   -------------------------------------
*/
   *pndelay = 0 ;
   return(frontJ) ;
}
/*
   -------------------------------------------------------
   order the children in ascending order of their front id
   this is done to ensure that the serial, multithreaded
   and MPI codes all assemble the same frontal matrix.
   -------------------------------------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n postponed children of %d :", frontJ->id) ;
for ( child = firstchild ; child != NULL ; child = child->next ) {
   fprintf(stdout, " %d", child->id) ;
}
fflush(stdout) ;
#endif
for ( child = firstchild, firstchild = NULL ; 
      child != NULL ; 
      child = nextchild ) {
   nextchild = child->next ;
   for ( child2 = firstchild, prev = NULL ; 
         child2 != NULL ; 
         child2 = child2->next ) {
      if ( child2->id > child->id ) {
         break ;
      }
      prev = child2 ;
   }
   if ( prev == NULL ) {
      firstchild = child ;
   } else {
      prev->next = child ;
   }
   child->next = child2 ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n front %d, postponed children reordered :", 
        frontJ->id) ;
for ( child = firstchild ; child != NULL ; child = child->next ) {
   fprintf(stdout, " %d", child->id) ;
}
fflush(stdout) ;
#endif
/*
   --------------------------
   compute the new dimensions
   --------------------------
*/
nDnew = frontJ->nD ;
for ( child = firstchild ; child != NULL ; child = child->next ) {
   nDnew += child->nD ;
}
/*
   ------------------------
   get a new chevron object
   ------------------------
*/
nbytes = DChv_nbytesNeeded(nDnew, frontJ->nL, frontJ->nU, 
                           frontJ->symflag) ;
newfrontJ = DChvManager_newObjectOfSizeNbytes(manager, nbytes) ;
DChv_init(newfrontJ, frontJ->id, nDnew, frontJ->nL, frontJ->nU, 
          frontJ->symflag) ;
/*
   ----------------------------------------------------------
   pivoting has been enabled, assemble any postponed chevrons
   ----------------------------------------------------------
*/
*pndelay = DChv_assemblePostponedData(newfrontJ, frontJ, firstchild) ;
/*
   --------------------------------------------------
   now put the postponed chevrons onto the free list.
   --------------------------------------------------
*/
DChvManager_releaseListOfObjects(manager, firstchild) ;
/*
   -------------------------------------
   set the delay to zero if a root front
   -------------------------------------
*/
if ( frontJ->nU == 0 ) {
   *pndelay = 0 ;
}
return(newfrontJ) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- extract and store the postponed data

   frontJ  -- pointer to present front object
   npost   -- # of postponed rows and columns in frontJ
   K       -- parent of J
   chvlist -- pointer to a DChvList object that handles the
              lists of postponed DChv objects
              a singly linked list to assemble
   manager -- pointer to a DChvManager object for the list
              of free DChv objects

   created -- 97may23, cca
   ---------------------------------------------------------
*/
void
DFrontMtx_storePostponedData (
   DFrontMtx     *frontmtx,
   DChv          *frontJ,
   int           npost,
   int           K,
   DChvList      *chvlist,
   DChvManager   *manager
) {
DChv   *chv ;
int    nbytes, nD, nent, nind, nL, nU ;

if ( npost <= 0 && chvlist != NULL ) {
   if ( K == -1 ) {
      DChvList_addObjectToList(chvlist, NULL, frontmtx->nfront) ;
   } else {
      DChvList_addObjectToList(chvlist, NULL, K) ;
   }
   return ;
} 
/*
   --------------------------------------
   find the number of indices and entries 
   necessary to store the delayed data
   --------------------------------------
*/
DChv_dimensions(frontJ, &nD, &nL, &nU) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n front %d: npost = %d, nD = %d, nL = %d, nU = %d", 
        frontJ->id, npost, nD, nL, nU) ;
fflush(stdout) ;
#endif
if ( frontJ->symflag == 0 ) {
   nind = npost + nU ;
   nent = (npost*(npost+1))/2 + npost*nU ;
} else {
   nind = 2*(npost + nU) ;
   nent = npost*(npost + 2*nU) ;
}
/*
   ------------------------------------
   get a DChv object from the free list
   ------------------------------------
*/
nbytes = DChv_nbytesNeeded(npost, nL, nU, frontJ->symflag) ;
chv = DChvManager_newObjectOfSizeNbytes(manager, nbytes) ;
DChv_init(chv, frontJ->id, npost, nL, nU, frontJ->symflag) ;
/*
   ----------------------
   store the delayed data
   ----------------------
*/
DChv_copyTrailingPortion(chv, frontJ, nD - npost) ;
frontJ->nD -= npost ;
frontJ->nL += npost ;
frontJ->nU += npost ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n postponed chevron %p", chv) ;
DChv_writeForHumanEye(chv, stdout) ;
fflush(stdout) ;
#endif
/*
   ------------------------------
   link the postponed DChv object
   ------------------------------
*/
if ( K == -1 ) {
   DChvList_addObjectToList(chvlist, chv, frontmtx->nfront) ;
} else {
   DChvList_addObjectToList(chvlist, chv, K) ;
}
return ; }
   
/*--------------------------------------------------------------------*/
