/*  permuteFactor.c  */

#include "../DFrontMtx.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
static void densePermuteU  ( DFrontMtx *frontmtx, int J ) ;
static void sparsePermuteU ( DFrontMtx *frontmtx, int J ) ;
static void densePermuteL  ( DFrontMtx *frontmtx, int J ) ;
static void sparsePermuteL ( DFrontMtx *frontmtx, int J ) ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   purpose -- 
     (1) permute the columns of U_{J,bnd{J}} so they are in
         ascending order w.r.t. to global column numbering
     (2) permute the rows of L_{bnd{J},J} so they are in
         ascending order w.r.t. to global row numbering

   created -- 97nov20, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_permuteFactor (
   DFrontMtx   *frontmtx,
   int         msglvl,
   FILE        *msgFile
) {
int    J, K, ncolK, nrowK ;
int    *colindK, *map, *par, *rowindK ;
Tree   *tree ;

tree = frontmtx->frontETree->tree ;
par  = tree->par ;
map  = IVinit(frontmtx->neqns, -1) ;
if ( frontmtx->pivotingflag == 1 ) {
/*
   ----------------------------------------------------------------
   permute U (and L if nonsymmetric) to have columns (and rows) in
   ascending order w.r.t. the final global column (and row) indices
   ----------------------------------------------------------------
*/
   for ( J = Tree_preOTfirst(tree) ;
         J != -1 ;
         J = Tree_preOTnext(tree, J) ) {
      if ( (K = par[J]) != -1 ) {
         DFrontMtx_columnIndices(frontmtx, K, &ncolK, &colindK) ;
         DFrontMtx_permuteColumnsOfU(frontmtx, J, K, ncolK, colindK,
                                     map, msglvl, msgFile) ;
         if (  frontmtx->pivotingflag == 1
            && frontmtx->symmetryflag == 2 ) {
            DFrontMtx_rowIndices(frontmtx, K, &nrowK, &rowindK) ;
            DFrontMtx_permuteRowsOfL(frontmtx, J, K, nrowK, rowindK,
                                     map, msglvl, msgFile) ;
         }
      }
   }
}
IVfree(map) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   permute the columns in U_{J,bnd{J}} so that the 
   columns are in a compatible ordering with its parent.

   created -- 97nov21, cca
   -----------------------------------------------------
*/
void
DFrontMtx_permuteColumnsOfU (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         ncolK,
   int         colindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) {
int   ii, mustdo, ncolJ, nDJ ;
int   *colindJ ;

if ( K != -1 ) {
/*
   --------------------
   work on U_{J,bnd{J}}
   --------------------
*/
   nDJ = DFrontMtx_frontSize(frontmtx, J) ;
   DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &colindJ) ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n front %d, nDJ = %d, ncolJ = %d",
              J, nDJ, ncolJ) ;
      fprintf(msgFile, "\n column indices") ;
      IVfprintf(msgFile, ncolJ, colindJ) ;
      fprintf(msgFile, "\n\n parent front %d, ncolK = %d",
              K, ncolK) ;
      fprintf(msgFile, "\n column indices") ;
      IVfprintf(msgFile, ncolK, colindK) ;
      fflush(msgFile) ;
   }
/*
   -----------------------------------
   overwrite column indices for J with
   indices that are local w.r.t K
   -----------------------------------
*/
   for ( ii = 0 ; ii < ncolK ; ii++ ) {
      map[colindK[ii]] = ii ;
   }
   for ( ii = nDJ ; ii < ncolJ ; ii++ ) {
      colindJ[ii] = map[colindJ[ii]] ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n local column indices for %d", J) ;
      IVfprintf(msgFile, ncolJ, colindJ) ;
      fflush(msgFile) ;
   }
   for ( ii = nDJ + 1, mustdo = 0 ; ii < ncolJ ; ii++ ) {
      if ( colindJ[ii] < colindJ[ii-1] ) {
         mustdo = 1 ;
         break ;
      }
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n mustdo = %d", mustdo) ;
      fflush(msgFile) ;
   }
   if ( nDJ > 0 ) {
      if ( mustdo == 1 ) {
/*
         -------------------------------------------
         we must permute the columns of U_{J,bnd{J}}
         -------------------------------------------
*/
         if ( frontmtx->sparsityflag == 0 ) {
            densePermuteU(frontmtx, J) ;
         } else {
            sparsePermuteU(frontmtx, J) ;
         }
      }
   } else {
/*
      --------------------------------------------
      sort the boundary indices in ascending order
      --------------------------------------------
*/
      IVqsortUp(ncolJ - nDJ, colindJ + nDJ) ;
   }
/*
   -----------------------------------------------
   overwrite the local indices with global indices
   -----------------------------------------------
*/
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n before, global indices for %d", J) ;
      IVfprintf(msgFile, ncolJ, colindJ) ;
      fflush(msgFile) ;
   }
   for ( ii = nDJ ; ii < ncolJ ; ii++ ) {
      colindJ[ii] = colindK[colindJ[ii]] ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n final global indices for %d", J) ;
      IVfprintf(msgFile, ncolJ, colindJ) ;
      fflush(msgFile) ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   permute the rows in L_{bnd{J},J} so that the rows 
   are in a compatible ordering with its parent.

   created -- 97nov21, cca
   -------------------------------------------------
*/
void
DFrontMtx_permuteRowsOfL (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         nrowK,
   int         rowindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) {
int   ii, mustdo, nDJ, nrowJ ;
int   *rowindJ ;

if ( K != -1 ) {
/*
   --------------------
   work on L_{bnd{J},J}
   --------------------
*/
   nDJ = DFrontMtx_frontSize(frontmtx, J) ;
   DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n front %d, nDJ = %d, nrowJ = %d",
              J, nDJ, nrowJ) ;
      fprintf(msgFile, "\n row indices") ;
      IVfprintf(msgFile, nrowJ, rowindJ) ;
      fprintf(msgFile, "\n\n parent front %d, nrowK = %d", K, nrowK) ;
      fprintf(msgFile, "\n row indices") ;
      IVfprintf(msgFile, nrowK, rowindK) ;
      fflush(msgFile) ;
   }
/*
   --------------------------------
   overwrite row indices for J with
   indices that are local w.r.t K
   --------------------------------
*/
   for ( ii = 0 ; ii < nrowK ; ii++ ) {
      map[rowindK[ii]] = ii ;
   }
   for ( ii = nDJ ; ii < nrowJ ; ii++ ) {
      rowindJ[ii] = map[rowindJ[ii]] ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n local row indices for %d", J) ;
      IVfprintf(msgFile, nrowJ, rowindJ) ;
      fflush(msgFile) ;
   }
   for ( ii = nDJ + 1, mustdo = 0 ; ii < nrowJ ; ii++ ) {
      if ( rowindJ[ii] < rowindJ[ii-1] ) {
         mustdo = 1 ;
         break ;
      }
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n mustdo = %d", mustdo) ;
      fflush(msgFile) ;
   }
   if ( nDJ > 0 ) {
      if ( mustdo == 1 ) {
/*
         ----------------------------------------
         we must permute the rows of L_{bnd{J},J}
         ----------------------------------------
*/
         if ( frontmtx->sparsityflag == 0 ) {
            densePermuteL(frontmtx, J) ;
         } else {
            sparsePermuteL(frontmtx, J) ;
         }
      }
   } else {
/*
      --------------------------------------------
      sort the boundary indices in ascending order
      --------------------------------------------
*/
      IVqsortUp(nrowJ - nDJ, rowindJ + nDJ) ;
   }
/*
   -----------------------------------------------
   overwrite the local indices with global indices
   -----------------------------------------------
*/
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n before, global indices for %d", J) ;
      IVfprintf(msgFile, nrowJ, rowindJ) ;
      fflush(msgFile) ;
   }
   for ( ii = nDJ ; ii < nrowJ ; ii++ ) {
      rowindJ[ii] = rowindK[rowindJ[ii]] ;
   }
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n final global indices for %d", J) ;
      IVfprintf(msgFile, nrowJ, rowindJ) ;
      fflush(msgFile) ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   permute the columns of U_{J,bnd{J}} 
   where U_{J,bnd{J}} is dense column major

   note, column indices of J are local w.r.t. its parent
   -----------------------------------------------------
*/
static void
densePermuteU (
   DFrontMtx   *frontmtx,
   int         J
) {
DA2      mtx ;
double   *upperent ;
int      first, ipivot, ncolJ, nDJ, nentJ, npivot, offset ;
int      *colindJ, *pivotsizes ;

nDJ = DFrontMtx_frontSize(frontmtx, J) ;
DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &colindJ) ;
DFrontMtx_upperEntries(frontmtx, J, &nentJ, &upperent) ;
/*
   ---------------------------------------
   get offset into entries of U_{J,bnd{J}}
   ---------------------------------------
*/
if ( frontmtx->symmetryflag == 0 ) {
   DFrontMtx_pivotsizes(frontmtx, J, 
                        &npivot, &pivotsizes) ;
   first = offset = 0 ;
   for ( ipivot = 0 ; ipivot < npivot ; ipivot++ ) {
      offset += first*pivotsizes[ipivot] ;
      first  += pivotsizes[ipivot] ;
   }
} else {
   offset = (nDJ*(nDJ - 1))/2 ;
}
/*
   ---------------------------------
   initialize a DA2 object for 
   U_{J,bnd{J}} and sort the columns
   ---------------------------------
*/
DA2_init(&mtx, nDJ, ncolJ - nDJ, 1, nDJ, upperent + offset) ;
DA2_sortColumnsUp(&mtx, ncolJ - nDJ, colindJ + nDJ) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   permute the columns of U_{J,bnd{J}} 
   where U_{J,bnd{J}} is sparse column major

   note, column indices of J are local w.r.t. its parent
   -----------------------------------------------------
*/
static void
sparsePermuteU (
   DFrontMtx   *frontmtx,
   int         J
) {
double   *upperent ;
int      countU, ii, jcol, kcol, kk, ncolJ, nDJ, nentJ, offset, size ;
int      *colindJ, *ivtemp, *sizes, *upperind ;

nDJ = DFrontMtx_frontSize(frontmtx, J) ;
DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &colindJ) ;
DFrontMtx_upperSparsityInfo(frontmtx, J, &ncolJ, &sizes, &upperind) ;
DFrontMtx_upperEntries(frontmtx, J, &nentJ, &upperent) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n before permutation of U, front %d", J) ;
fprintf(stdout, "\n colindJ") ;
IVfprintf(stdout, ncolJ, colindJ) ;
fprintf(stdout, "\n sizes") ;
IVfprintf(stdout, ncolJ, sizes) ;
fprintf(stdout, "\n upperind") ;
IVfprintf(stdout, nentJ, upperind) ;
fprintf(stdout, "\n upperent") ;
DVfprintf(stdout, nentJ, upperent) ;
for ( jcol = kk = 0 ; jcol < ncolJ ; jcol++ ) {
   fprintf(stdout, "\n column %d, local id %d", jcol, colindJ[jcol]) ;
   for ( ii = 0 ; ii < sizes[jcol] ; ii++, kk++ ) {
      if ( ii % 3 == 0 ) {
         fprintf(stdout, "\n") ;
      }
      fprintf(stdout, " <%4d, %12.4e>", upperind[kk], upperent[kk]) ;
   }
}
#endif
/*
   --------------------------------
   get the offset into U_{J,bnd{J}}
   --------------------------------
*/
for ( jcol = offset = 0 ; jcol < nDJ ; jcol++ ) {
   offset += sizes[jcol] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
fflush(stdout) ;
#endif
/*
   -------------------------------------------
   count the number of entries in U_{J,bnd{J}}
   -------------------------------------------
*/
countU = 0 ; 
for ( jcol = nDJ ; jcol < ncolJ ; jcol++ ) {
   countU += sizes[jcol] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n countU = %d", countU) ;
fflush(stdout) ;
#endif
if ( countU == 0 ) {
   return ;
}
/*
   --------------------------------------
   for each entry in U_{J,bnd{J}}, 
   set ivtemp[] to be its local column id
   --------------------------------------
*/
ivtemp = IVinit(countU, -1) ;
for ( jcol = nDJ, kk = 0 ; jcol < ncolJ ; jcol++ ) {
   for ( ii = 0 ; ii < sizes[jcol] ; ii++ ) {
      ivtemp[kk++] = colindJ[jcol] ;
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n ivtemp") ;
IVfprintf(stdout, countU, ivtemp) ;
fflush(stdout) ;
#endif
/*
   ---------------------------------------------
   set the sizes of each boundary column to zero
   ---------------------------------------------
*/
for ( jcol = nDJ ; jcol < ncolJ ; jcol++ ) {
   sizes[jcol] = 0 ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n sizes") ;
IVfprintf(stdout, ncolJ, sizes) ;
fflush(stdout) ;
#endif
/*
   -------------------------------------------
   sort all the entries so that the columns
   are in ascending order w.r.t. local indices
   -------------------------------------------
*/
IV2DVqsortUp(countU, ivtemp, upperind + offset, upperent + offset) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n after sort, ivtemp") ;
IVfprintf(stdout, countU, ivtemp) ;
fprintf(stdout, "\n\n after sort, upperind + offset") ;
IVfprintf(stdout, countU, upperind + offset) ;
fprintf(stdout, "\n\n after sort, upperent + offset") ;
DVfprintf(stdout, countU, upperent + offset) ;
fflush(stdout) ;
#endif
/*
   -------------------------------------------
   reset the sizes vector and sort each column 
   so row indices are in ascending order
   -------------------------------------------
*/
jcol = nDJ ;
kcol = ivtemp[0] ;
kk   = 1 ;
size = 1 ;
while ( kk < countU ) {
   if ( ivtemp[kk] == kcol ) {
      size++ ;
   } else {
      for ( jcol = nDJ ; jcol < ncolJ ; jcol++ ) {
         if ( colindJ[jcol] == kcol ) {
            break ;
         }
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n jcol = %d, kcol = %d, kk = %d, size = %d",
              jcol, kcol, kk, size) ;
      fflush(stdout) ;
      fprintf(stdout, "\n\n before sort") ;
      IVfprintf(stdout, size, upperind + offset + kk - size) ;
      DVfprintf(stdout, size, upperent + offset + kk - size) ;
      fflush(stdout) ;
#endif
      IVDVqsortUp(size, upperind + offset + kk - size, 
                  upperent + offset + kk - size) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n\n after sort") ;
      IVfprintf(stdout, size, upperind + offset + kk - size) ;
      DVfprintf(stdout, size, upperent + offset + kk - size) ;
      fflush(stdout) ;
#endif
      sizes[jcol] = size ;
      kcol = ivtemp[kk] ;
      size = 1 ;
   }
   kk++ ;
}
#if MYDEBUG > 0
      fprintf(stdout, "\n jcol = %d, kcol = %d, kk = %d, size = %d",
              jcol, kcol, kk, size) ;
      fflush(stdout) ;
      fprintf(stdout, "\n\n before sort") ;
      IVfprintf(stdout, size, upperind + offset + kk - size) ;
      DVfprintf(stdout, size, upperent + offset + kk - size) ;
      fflush(stdout) ;
#endif
for ( jcol = nDJ ; jcol < ncolJ ; jcol++ ) {
   if ( colindJ[jcol] == kcol ) {
      break ;
   }
}
sizes[jcol] = size ;
IVDVqsortUp(size, upperind + offset + kk - size, 
            upperent + offset + kk - size) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n\n after sort") ;
      IVfprintf(stdout, size, upperind + offset + kk - size) ;
      DVfprintf(stdout, size, upperent + offset + kk - size) ;
      fflush(stdout) ;
#endif
/*
   --------------------------------------------
   sort the boundary indices in ascending order
   --------------------------------------------
*/
IV2qsortUp(ncolJ - nDJ, colindJ + nDJ, sizes + nDJ) ;
#if MYDEBUG > 0
fprintf(stdout, "\n colindJ") ;
IVfprintf(stdout, ncolJ, colindJ) ;
fprintf(stdout, "\n sizes") ;
IVfprintf(stdout, ncolJ, sizes) ;
fprintf(stdout, "\n\n after permutation, front %d", J) ;
for ( jcol = kk = 0 ; jcol < ncolJ ; jcol++ ) {
   fprintf(stdout, "\n column %d, local id %d", jcol, colindJ[jcol]) ;
   for ( ii = 0 ; ii < sizes[jcol] ; ii++, kk++ ) {
      if ( ii % 3 == 0 ) {
         fprintf(stdout, "\n") ;
      }
      fprintf(stdout, " <%4d, %12.4e>", upperind[kk], upperent[kk]) ;
   }
}
#endif
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(ivtemp) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   permute the rows of L_{bnd{J},J} 
   where L_{bnd{J},J} is dense row major

   note, column indices of J are local w.r.t. its parent
   -----------------------------------------------------
*/
static void
densePermuteL (
   DFrontMtx   *frontmtx,
   int         J
) {
DA2      mtx ;
double   *lowerent ;
int      nrowJ, nDJ, nentJ, offset ;
int      *rowindJ ;

nDJ = DFrontMtx_frontSize(frontmtx, J) ;
DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
DFrontMtx_lowerEntries(frontmtx, J, &nentJ, &lowerent) ;
/*
   ---------------------------------------
   get offset into entries of L_{bnd{J},J}
   ---------------------------------------
*/
offset = (nDJ*(nDJ - 1))/2 ;
/*
   ------------------------------
   initialize a DA2 object for 
   L_{bnd{J},J} and sort the rows
   ------------------------------
*/
DA2_init(&mtx, nrowJ - nDJ, nDJ, nDJ, 1, lowerent + offset) ;
DA2_sortRowsUp(&mtx, nrowJ - nDJ, rowindJ + nDJ) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   permute the rows of L_{bnd{J},J} 
   where L_{bnd{J},J} is sparse row major

   note, column indices of J are local w.r.t. its parent
   -----------------------------------------------------
*/
static void
sparsePermuteL (
   DFrontMtx   *frontmtx,
   int         J
) {
double   *lowerent ;
int      countL, ii, jrow, kk, krow, nrowJ, nDJ, nentJ, offset, size ;
int      *rowindJ, *ivtemp, *lowerind, *sizes ;

nDJ = DFrontMtx_frontSize(frontmtx, J) ;
DFrontMtx_rowIndices(frontmtx, J, &nrowJ, &rowindJ) ;
DFrontMtx_lowerSparsityInfo(frontmtx, J, &nrowJ, &sizes, &lowerind) ;
DFrontMtx_lowerEntries(frontmtx, J, &nentJ, &lowerent) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n before permutation of L, front %d", J) ;
fprintf(stdout, "\n rowindJ") ;
IVfprintf(stdout, nrowJ, rowindJ) ;
fprintf(stdout, "\n sizes") ;
IVfprintf(stdout, nrowJ, sizes) ;
fprintf(stdout, "\n upperind") ;
IVfprintf(stdout, nentJ, lowerind) ;
fprintf(stdout, "\n lowerent") ;
DVfprintf(stdout, nentJ, lowerent) ;
for ( jrow = kk = 0 ; jrow < nrowJ ; jrow++ ) {
   fprintf(stdout, "\n row %d, local id %d", jrow, rowindJ[jrow]) ;
   for ( ii = 0 ; ii < sizes[jrow] ; ii++, kk++ ) {
      if ( ii % 3 == 0 ) {
         fprintf(stdout, "\n") ;
      }
      fprintf(stdout, " <%4d, %12.4e>", lowerind[kk], lowerent[kk]) ;
   }
}
#endif
/*
   --------------------------------
   get the offset into L_{bnd{J},J}
   --------------------------------
*/
for ( jrow = offset = 0 ; jrow < nDJ ; jrow++ ) {
   offset += sizes[jrow] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
fflush(stdout) ;
#endif
/*
   -------------------------------------------
   count the number of entries in L_{bnd{J},J}
   -------------------------------------------
*/
countL = 0 ; 
for ( jrow = nDJ ; jrow < nrowJ ; jrow++ ) {
   countL += sizes[jrow] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n countL = %d", countL) ;
fflush(stdout) ;
#endif
if ( countL == 0 ) {
   return ;
}
/*
   ------------------------------------
   for each entry in L_{bnd{J},J}, 
   set ivtemp[] to be its local row id
   ------------------------------------
*/
ivtemp = IVinit(countL, -1) ;
for ( jrow = nDJ, kk = 0 ; jrow < nrowJ ; jrow++ ) {
   for ( ii = 0 ; ii < sizes[jrow] ; ii++ ) {
      ivtemp[kk++] = rowindJ[jrow] ;
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n ivtemp") ;
IVfprintf(stdout, countL, ivtemp) ;
fflush(stdout) ;
#endif
/*
   ------------------------------------------
   set the sizes of each boundary row to zero
   ------------------------------------------
*/
for ( jrow = nDJ ; jrow < nrowJ ; jrow++ ) {
   sizes[jrow] = 0 ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n sizes") ;
IVfprintf(stdout, nrowJ, sizes) ;
fflush(stdout) ;
#endif
/*
   -----------------------------------------
   sort all the entries so that the rows are
   in ascending order w.r.t. local indices
   -----------------------------------------
*/
IV2DVqsortUp(countL, ivtemp, lowerind + offset, lowerent + offset) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n after sort, ivtemp") ;
IVfprintf(stdout, countL, ivtemp) ;
fprintf(stdout, "\n\n after sort, lowerind + offset") ;
IVfprintf(stdout, countL, lowerind + offset) ;
fprintf(stdout, "\n\n after sort, lowerent + offset") ;
DVfprintf(stdout, countL, lowerent + offset) ;
fflush(stdout) ;
#endif
/*
   ----------------------------------------
   reset the sizes vector and sort each row 
   so column indices are in ascending order
   ----------------------------------------
*/
jrow = nDJ ;
krow = ivtemp[0] ;
kk   = 1 ;
size = 1 ;
while ( kk < countL ) {
   if ( ivtemp[kk] == krow ) {
      size++ ;
   } else {
      for ( jrow = nDJ ; jrow < nrowJ ; jrow++ ) {
         if ( rowindJ[jrow] == krow ) {
            break ;
         }
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n jrow = %d, krow = %d, kk = %d, size = %d",
              jrow, krow, kk, size) ;
      fprintf(stdout, "\n\n before sort") ;
      IVfprintf(stdout, size, lowerind + offset + kk - size) ;
      DVfprintf(stdout, size, lowerent + offset + kk - size) ;
      fflush(stdout) ;
#endif
      IVDVqsortUp(size, lowerind + offset + kk - size, 
                  lowerent + offset + kk - size) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n\n after sort") ;
      IVfprintf(stdout, size, lowerind + offset + kk - size) ;
      DVfprintf(stdout, size, lowerent + offset + kk - size) ;
      fflush(stdout) ;
#endif
      sizes[jrow] = size ;
      krow = ivtemp[kk] ;
      size = 1 ;
   }
   kk++ ;
}
for ( jrow = nDJ ; jrow < nrowJ ; jrow++ ) {
   if ( rowindJ[jrow] == krow ) {
      break ;
   }
}
sizes[jrow] = size ;
IVDVqsortUp(size, lowerind + offset + kk - size, 
            lowerent + offset + kk - size) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n after sort") ;
IVfprintf(stdout, size, lowerind + offset + kk - size) ;
DVfprintf(stdout, size, lowerent + offset + kk - size) ;
fflush(stdout) ;
#endif
/*
   --------------------------------------------
   sort the boundary indices in ascending order
   --------------------------------------------
*/
IV2qsortUp(nrowJ - nDJ, rowindJ + nDJ, sizes + nDJ) ;
#if MYDEBUG > 0
fprintf(stdout, "\n rowindJ") ;
IVfprintf(stdout, nrowJ, rowindJ) ;
fprintf(stdout, "\n sizes") ;
IVfprintf(stdout, nrowJ, sizes) ;
fprintf(stdout, "\n\n after shuffle and sort, lowerind + offset") ;
IVfprintf(stdout, countL, lowerind + offset) ;
fprintf(stdout, "\n\n after shuffle and sort, lowerent + offset") ;
DVfprintf(stdout, countL, lowerent + offset) ;
fprintf(stdout, "\n\n after permutation, front %d", J) ;
for ( jrow = kk = 0 ; jrow < nrowJ ; jrow++ ) {
   fprintf(stdout, "\n row %d, local id %d", jrow, rowindJ[jrow]) ;
   for ( ii = 0 ; ii < sizes[jrow] ; ii++, kk++ ) {
      if ( ii % 3 == 0 ) {
         fprintf(stdout, "\n") ;
      }
      fprintf(stdout, " <%4d, %12.4e>", lowerind[kk], lowerent[kk]) ;
   }
}
#endif
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(ivtemp) ;

return ; }

/*--------------------------------------------------------------------*/
