/*  instance.c  */

#include "../DFrontMtx.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   simple method to return the dimensions of front J and the number 
   of bytes necessary for the DChv object to hold the front.

   created -- 97nov13, cca
   ----------------------------------------------------------------
*/
void
DFrontMtx_initialFrontDimensions (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnD,
   int         *pnL,
   int         *pnU,
   int         *pnbytes
) {
int   nbytes, nD, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || J < 0 || J >= frontmtx->nfront
   || pnD == NULL || pnL == NULL || pnU == NULL || pnbytes == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_initialFrontDimensions()"
           "\n frontmtx = %p, J = %d, pnD = %p, "
           "pnL = %p, pnU = %p, pnbytes = %p",
           frontmtx, J, pnD, pnL, pnU, pnbytes) ;
   exit(-1) ;
}
nD = frontmtx->frontETree->nodwghtsIV->vec[J] ;
nL = nU = frontmtx->frontETree->bndwghtsIV->vec[J] ;
if ( frontmtx->symmetryflag == 2 ) {
   nbytes = DChv_nbytesNeeded(nD, nL, nU, 2) ;
} else {
   nbytes = DChv_nbytesNeeded(nD, nL, nU, 0) ;
}
*pnD = nD ;
*pnL = nL ;
*pnU = nU ;
*pnbytes = nbytes ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   return the number of internal rows and columns in front J

   created -- 97may12, cca
   ---------------------------------------------------------
*/
int
DFrontMtx_frontSize (
   DFrontMtx   *frontmtx,
   int         J
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || frontmtx->frontsizesIV == NULL 
   || J < 0 || J >= frontmtx->nfront ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_frontSize(%p,%d)"
           "\n bad input\n", frontmtx, J) ;
   exit(-1) ;
}
return(IV_entry(frontmtx->frontsizesIV, J)) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------
   set the number of internal rows and columns in front J

   created -- 97may12, cca
   ------------------------------------------------------
*/
void
DFrontMtx_setFrontSize (
   DFrontMtx   *frontmtx,
   int         J,
   int         size
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || frontmtx->frontsizesIV == NULL 
   || J < 0 || J >= frontmtx->nfront || size < 0 ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_setFrontSize(%p,%d,%d)"
           "\n bad input\n", frontmtx, J, size) ;
   exit(-1) ;
}
IV_setEntry(frontmtx->frontsizesIV, J, size) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   return the maximum number of internal rows and columns in a front

   created -- 97may12, cca
   -----------------------------------------------------------------
*/
int
DFrontMtx_maxFrontSize (
   DFrontMtx   *frontmtx
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_maxFrontSize(%p)"
           "\n bad input\n", frontmtx) ;
   exit(-1) ;
}
return(IV_max(frontmtx->frontsizesIV)) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------
   return the number of rows in front J

   created -- 97may12, cca
   ------------------------------------
*/
int
DFrontMtx_nrowInFront (
   DFrontMtx   *frontmtx,
   int         J
) {
int   nrow ;
int   *ivec ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_nrowInFront(%p,%d)"
           "\n bad input\n", frontmtx, J) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   IVL_listAndSize(frontmtx->symbfacIVL, J, &nrow, &ivec) ;
} else if ( frontmtx->symmetryflag < 2 ) {
   IVL_listAndSize(frontmtx->coladjIVL, J, &nrow, &ivec) ;
} else {
   IVL_listAndSize(frontmtx->rowadjIVL, J, &nrow, &ivec) ;
}
return(nrow) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------
   return the maximum number of rows in a front

   created -- 97may12, cca
   --------------------------------------------
*/
int
DFrontMtx_maxNrowInFront (
   DFrontMtx   *frontmtx
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_maxNrowInFront(%p)"
           "\n bad input\n", frontmtx) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   return(IVL_maxListSize(frontmtx->symbfacIVL)) ;
} else if ( frontmtx->symmetryflag < 2 ) {
   return(IVL_maxListSize(frontmtx->coladjIVL)) ;
} else {
   return(IVL_maxListSize(frontmtx->rowadjIVL)) ;
}
}
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   return the number of columns in front J

   created -- 97may12, cca
   ---------------------------------------
*/
int
DFrontMtx_ncolInFront (
   DFrontMtx   *frontmtx,
   int         J
) {
int   ncol ;
int   *ivec ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_ncolInFront(%p,%d)"
           "\n bad input\n", frontmtx, J) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   IVL_listAndSize(frontmtx->symbfacIVL, J, &ncol, &ivec) ;
} else {
   IVL_listAndSize(frontmtx->coladjIVL, J, &ncol, &ivec) ;
}
return(ncol) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   return the maximum number of columns in a front

   created -- 97may12, cca
   -----------------------------------------------
*/
int
DFrontMtx_maxNcolInFront (
   DFrontMtx   *frontmtx
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_maxNcolInFront(%p)"
           "\n bad input\n", frontmtx) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   return(IVL_maxListSize(frontmtx->symbfacIVL)) ;
} else {
   return(IVL_maxListSize(frontmtx->coladjIVL)) ;
}
}
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   fill *pncol with the number of columns and 
   *pcolind with a pointer to the column indices

   created -- 97may12, cca
   ---------------------------------------------
*/
void
DFrontMtx_columnIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pncol,
   int         **pcolind
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pncol == NULL || pcolind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_columnIndices(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pncol, pcolind) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   IVL_listAndSize(frontmtx->symbfacIVL, J, pncol, pcolind) ;
} else {
   IVL_listAndSize(frontmtx->coladjIVL, J, pncol, pcolind) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   fill *pnrow with the number of rows and 
   *prowind with a pointer to the rows indices

   created -- 97may12, cca
   -------------------------------------------
*/
void
DFrontMtx_rowIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnrow,
   int         **prowind
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnrow == NULL || prowind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_rowIndices(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pnrow, prowind) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   IVL_listAndSize(frontmtx->symbfacIVL, J, pnrow, prowind) ;
} else if ( frontmtx->symmetryflag < 2 ) {
   IVL_listAndSize(frontmtx->coladjIVL, J, pnrow, prowind) ;
} else {
   IVL_listAndSize(frontmtx->rowadjIVL, J, pnrow, prowind) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   fill *pnpivot with the number of npivots and 
   *ppivotsizes with a pointer to the pivot sizes

   created -- 97may12, cca
   ----------------------------------------------
*/
void
DFrontMtx_pivotsizes (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnpivot,
   int         **ppivotsizes
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnpivot == NULL || ppivotsizes == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_pivotsizes(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pnpivot, ppivotsizes) ;
   exit(-1) ;
}
if ( frontmtx->pivotingflag == 0 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_pivotsizes(%p,%d,%p,%p)"
           "\n no pivoting\n", frontmtx, J, pnpivot, ppivotsizes) ;
   exit(-1) ;
}
if ( frontmtx->symmetryflag != 0 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_pivotsizes(%p,%d,%p,%p)"
           "\n not symmetric\n", frontmtx, J, pnpivot, ppivotsizes) ;
   exit(-1) ;
}
if ( frontmtx->pivotsIVL != NULL ) {
   IVL_listAndSize(frontmtx->pivotsIVL, J, pnpivot, ppivotsizes) ;
} else {
   *pnpivot     =   0  ;
   *ppivotsizes = NULL ;
}


return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------
   fill *pnent with the number of diagonal entries and 
   *pdiagent with a pointer to the diagonal entries.

   created -- 97may12, cca
   ---------------------------------------------------
*/
void
DFrontMtx_diagEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **pdiagent
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnent == NULL || pdiagent == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_diagent(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pnent, pdiagent) ;
   exit(-1) ;
}
if ( frontmtx->diagDVL != NULL ) {
   DVL_listAndSize(frontmtx->diagDVL, J, pnent, pdiagent) ;
} else {
   *pnent    =   0  ;
   *pdiagent = NULL ;
}


return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   fill *pnent with the number of lower entries and
   *plowerent with a pointer to the lower entries.

   created -- 97may12, cca
   -----------------------------------------------
*/
void
DFrontMtx_lowerEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **plowerent
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnent == NULL || plowerent == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_lowerEntries(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pnent, plowerent) ;
   exit(-1) ;
}
if ( frontmtx->lowerDVL != NULL ) {
   DVL_listAndSize(frontmtx->lowerDVL, J, pnent, plowerent) ;
} else {
   *pnent     =   0  ;
   *plowerent = NULL ;
}


return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   fill *pnent with the number of upper entries and
   *pupperent with a pointer to the upper entries.

   created -- 97may12, cca
   ------------------------------------------------
*/
void
DFrontMtx_upperEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **pupperent
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnent == NULL || pupperent == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_upperEntries(%p,%d,%p,%p)"
           "\n bad input\n", frontmtx, J, pnent, pupperent) ;
   exit(-1) ;
}
if ( frontmtx->upperDVL != NULL ) {
   DVL_listAndSize(frontmtx->upperDVL, J, pnent, pupperent) ;
} else {
   *pnent     =   0  ;
   *pupperent = NULL ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   fill *pnrow with the number of rows in J,
   *psizes with the sizes vector for the sparse rows of L
   *pindices with the column indices for the sparse rows of L

   created -- 97may12, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_lowerSparsityInfo (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnrow,
   int         **psizes,
   int         **plowerind
) {
int   nent ;
int   *ivec ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pnrow == NULL || psizes == NULL || plowerind == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_lowerSparsityInfo(%p,%d,%p,%p,%p)"
        "\n bad input\n", frontmtx, J, pnrow, psizes, plowerind) ;
   exit(-1) ;
}
if ( frontmtx->lowerIVL != NULL ) { 
   *pnrow = DFrontMtx_nrowInFront(frontmtx, J) ;
   IVL_listAndSize(frontmtx->lowerIVL, J, &nent, &ivec) ;
   *psizes = ivec ;
   *plowerind = ivec + *pnrow ;
} else {
   *psizes = 0 ;
   *plowerind = NULL ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   fill *pncol with the number of columns in J,
   *psizes with the sizes vector for the sparse columns of U
   *pindices with the row indices for the sparse columns of U

   created -- 97may12, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_upperSparsityInfo (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pncol,
   int         **psizes,
   int         **pupperind
) {
int   nent ;
int   *ivec ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || J < 0 || J >= frontmtx->nfront 
   || pncol == NULL || psizes == NULL || pupperind == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_upperSparsityInfo(%p,%d,%p,%p,%p)"
        "\n bad input\n", frontmtx, J, pncol, psizes, pupperind) ;
   exit(-1) ;
}
if ( frontmtx->upperIVL != NULL ) { 
   *pncol = DFrontMtx_ncolInFront(frontmtx, J) ;
   IVL_listAndSize(frontmtx->upperIVL, J, &nent, &ivec) ;
   *psizes = ivec ;
   *pupperind = ivec + *pncol ;
} else {
   *pncol     =   0  ;
   *psizes    = NULL ;
   *pupperind = NULL ;
}
return ; }

/*--------------------------------------------------------------------*/
