/*  init.c  */

#include "../DFrontMtx.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- basic initializer

   frontETree   -- ETree object that stores the front tree
   symbfacIVL   -- IVL object that stores the symbolic factorization
   symmetryflag -- symmetry flag,
      0 --> matrix has symmetric structure, symmetric entries
      1 --> matrix has symmetric structure, nonsymmetric entries
      2 --> matrix has nonsymmetric structure, nonsymmetric entries
      3 --> matrix from QR factorization
   sparsityflag -- flag to specify dense or sparse fronts
      0 --> dense fronts
      1 --> sparse fronts
   pivotingflag -- flag to specify pivoting enabled
      0 --> pivoting not used
      1 --> pivoting used

   in a multithreaded environment, we need to protect the critical
   section where data is allocated. we use a lockflag to do this.
   in a serial or distributed environment, use lockflag = 0.
 
   lockflag -- flag to specify lock status
      0 --> mutex lock is not allocated or initialized
      1 --> mutex lock is allocated and it can synchronize
            only threads in this process.
      2 --> mutex lock is allocated and it can synchronize
            only threads in this and other processes.

   in a distributed environment we need to specify which process
   owns each front. when we can preallocate data structures
   (when there is no pivoting and dense fronts are stored) we
   need each process to determine what parts of the data it
   can allocate and set up. in a serial or multithreaded 
   environment, use ownersIV = NULL.

      ownersIV -- map from fronts to owning processes
      myid     -- id of this process.

   created  -- 97apr23, cca
   modified -- 97jul11, cca
      ownersIV and myid added
   ------------------------------------------------------------------
*/
void
DFrontMtx_init (
   DFrontMtx   *frontmtx,
   ETree       *frontETree,
   IVL         *symbfacIVL,
   int         symmetryflag,
   int         sparsityflag,
   int         pivotingflag,
   int         lockflag,
   int         myid,
   IV          *ownersIV
) {
int   J, neqns, nDJ, nentU, nfront, nUJ ;
int   *bndwghts, *nodwghts, *owners, *sizes ;
/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || frontETree == NULL || symbfacIVL == NULL
   || symmetryflag < 0 || symmetryflag > 3
   || sparsityflag < 0 || sparsityflag > 1
   || pivotingflag < 0 || pivotingflag > 1 
   || lockflag < 0 || lockflag > 2
   || (ownersIV != NULL && myid < 0) ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_init(%p,%p,%p,%d,%d,%d,%p,%d)"
           "\n bad input\n", frontmtx, frontETree, symbfacIVL,
           symmetryflag, sparsityflag, pivotingflag, ownersIV, myid) ;
   exit(-1) ;
}
nfront   = frontETree->nfront         ;
neqns    = frontETree->nvtx           ;
nodwghts = ETree_nodwghts(frontETree) ;
bndwghts = ETree_bndwghts(frontETree) ;
/*
   ----------------------
   set the default fields
   ----------------------
*/
DFrontMtx_setDefaultFields(frontmtx) ;
/*
   ---------------------
   set the scalar fields
   ---------------------
*/
frontmtx->nfront       = nfront       ;
frontmtx->neqns        = neqns        ;
frontmtx->symmetryflag = symmetryflag ;
frontmtx->sparsityflag = sparsityflag ;
frontmtx->pivotingflag = pivotingflag ;
/*
   ---------------------------------------------------------------
   set the front tree ETree and symbolic factorization IVL objects
   ---------------------------------------------------------------
*/
frontmtx->frontETree = frontETree ;
frontmtx->symbfacIVL = symbfacIVL ;
/*
   -------------------------------
   set the IV, IVL and DVL objects
   -------------------------------
*/
if ( pivotingflag == 0 && sparsityflag == 0 ) {
/*
   -----------------------------------
   no pivoting and store dense fronts,
   preallocate the data structures
   -----------------------------------
*/
   frontmtx->frontsizesIV = IV_new() ;
   IV_init(frontmtx->frontsizesIV, nfront, nodwghts) ;
   sizes = IVinit(nfront, 0) ;
   if ( ownersIV == NULL ) {
      frontmtx->diagDVL = DVL_new() ;
      DVL_init3(frontmtx->diagDVL, DVL_CHUNKED, nfront, nodwghts) ;
      for ( J = 0 ; J < nfront ; J++ ) {
         nDJ = nodwghts[J] ;
         nUJ = bndwghts[J] ;
         nentU = (nDJ*(nDJ-1))/2 + nDJ*nUJ ;
         sizes[J] = nentU ;
      }
      frontmtx->upperDVL = DVL_new() ;
      DVL_init3(frontmtx->upperDVL, DVL_CHUNKED, nfront, sizes) ;
      if ( symmetryflag == 1 || symmetryflag == 2 ) {
         frontmtx->lowerDVL = DVL_new() ;
         DVL_init3(frontmtx->lowerDVL, DVL_CHUNKED, nfront, sizes) ;
      }
   } else {
      owners = IV_entries(ownersIV) ;
      for ( J = 0 ; J < nfront ; J++ ) {
         if ( myid == owners[J] ) {
            sizes[J] = nodwghts[J] ;
         } else {
            sizes[J] = 0 ;
         }
      }
      frontmtx->diagDVL = DVL_new() ;
      DVL_init3(frontmtx->diagDVL, DVL_CHUNKED, nfront, sizes) ;
      for ( J = 0 ; J < nfront ; J++ ) {
         if ( myid == owners[J] ) {
            nDJ      = nodwghts[J] ;
            nUJ      = bndwghts[J] ;
            nentU    = (nDJ*(nDJ-1))/2 + nDJ*nUJ ;
            sizes[J] = nentU ;
         } else {
            sizes[J] = 0 ;
         }
      }
      frontmtx->upperDVL = DVL_new() ;
      DVL_init3(frontmtx->upperDVL, DVL_CHUNKED, nfront, sizes) ;
      if ( symmetryflag == 1 || symmetryflag == 2 ) {
         frontmtx->lowerDVL = DVL_new() ;
         DVL_init3(frontmtx->lowerDVL, DVL_CHUNKED, nfront, sizes) ;
      }
   }
   IVfree(sizes) ;
} else {
/*
   -------------------------------------------
   we must allocate data structures on the fly
   -------------------------------------------
*/
   if ( pivotingflag == 1 ) {
/*
      ----------------------------------------
      pivoting enabled, we need column indices
      ----------------------------------------
*/
      frontmtx->coladjIVL = IVL_new() ;
      IVL_init1(frontmtx->coladjIVL, IVL_CHUNKED, nfront) ;
/*
      ---------------------------------------
      front sizes cannot be set ahead of time
      ---------------------------------------
*/
      frontmtx->frontsizesIV = IV_new() ;
      IV_init(frontmtx->frontsizesIV, nfront, NULL) ;
      IV_fill(frontmtx->frontsizesIV, 0) ;
      if ( symmetryflag == 0 ) {
/*
         ------------------------------------------------------
         symmetric factorization, we need the pivots IVL object
         ------------------------------------------------------
*/
         frontmtx->pivotsIVL = IVL_new() ;
         IVL_init2(frontmtx->pivotsIVL, IVL_CHUNKED, nfront, neqns) ;
      } else if ( symmetryflag == 2 ) {
/*
         --------------------------------------------------
         nonsymmetric permutations, we need the row indices
         --------------------------------------------------
*/
         frontmtx->rowadjIVL = IVL_new() ;
         IVL_init1(frontmtx->rowadjIVL, IVL_CHUNKED, nfront) ;
      }
      if ( symmetryflag == 0 ) {
/*
         -----------------------------------------------
         set up DVL for diagonal entries, size not known
         -----------------------------------------------
*/
         frontmtx->diagDVL = DVL_new() ;
         DVL_init1(frontmtx->diagDVL, DVL_CHUNKED, nfront) ;
      } else {
/*
         -------------------------------------------
         set up DVL for diagonal entries, size known
         -------------------------------------------
*/
         frontmtx->diagDVL = DVL_new() ;
         DVL_init2(frontmtx->diagDVL, DVL_CHUNKED, nfront, neqns) ;
      }
   } else {
/*
      ---------------------------------------
      no pivoting, front sizes can be aliased
      ---------------------------------------
*/
      frontmtx->frontsizesIV = IV_new() ;
      IV_init(frontmtx->frontsizesIV, nfront, nodwghts) ;
/*
      -------------------------------------------
      set up DVL for diagonal entries, size known
      -------------------------------------------
*/
      frontmtx->diagDVL = DVL_new() ;
      DVL_init3(frontmtx->diagDVL, DVL_CHUNKED, nfront, nodwghts) ;
   }
   if ( sparsityflag == 1 ) {
/*
      -----------------------------------------------------------
      fronts are sparse, we need the row indices for entries in U
      -----------------------------------------------------------
*/
      frontmtx->upperIVL = IVL_new() ;
      IVL_init1(frontmtx->upperIVL, IVL_CHUNKED, frontmtx->nfront) ;
      if ( symmetryflag == 2 ) {
/*
      ----------------------------------------------------------------
      nonsymmetric matrix, we need the column indices for entries in L
      ----------------------------------------------------------------
*/
         frontmtx->lowerIVL = IVL_new() ;
         IVL_init1(frontmtx->lowerIVL, IVL_CHUNKED, frontmtx->nfront) ;
      }
   }
/*
   --------------------------------------
   initialize the object for entries in U
   --------------------------------------
*/
   frontmtx->upperDVL = DVL_new() ;
   DVL_init1(frontmtx->upperDVL, DVL_CHUNKED, frontmtx->nfront) ;
   if ( symmetryflag == 1 || symmetryflag == 2 ) {
/*
      -------------------------------------------
      initialize the object to store entries in L
      -------------------------------------------
*/
      frontmtx->lowerDVL = DVL_new() ;
      DVL_init1(frontmtx->lowerDVL, DVL_CHUNKED, frontmtx->nfront) ;
   }
}
if ( lockflag > 0 ) {
/*
   -----------------
   allocate the lock
   -----------------
*/
   frontmtx->lock = Lock_new() ;
   Lock_init(frontmtx->lock, lockflag) ;
/*
#if THREAD_TYPE == TT_SOLARIS
   ALLOCATE(frontmtx->lock, mutex_t, 1) ;
   if ( lockflag == 1 ) {
      mutex_init(frontmtx->lock, USYNC_THREAD, NULL) ;
   } else if ( lockflag == 2 ) {
      mutex_init(frontmtx->lock, USYNC_PROCESS, NULL) ;
   }
#endif
#if THREAD_TYPE == TT_POSIX
   ALLOCATE(frontmtx->lock, pthread_mutex_t, 1) ;
   pthread_mutex_init(frontmtx->lock, NULL) ;
#endif
*/
}
#if MYDEBUG > 0
fprintf(stdout, "\n frontmtx->lock = %p", frontmtx->lock) ;
#endif
   
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------
   purpose -- initialize a front

   created -- 97may23, cca
   -----------------------------
*/
void
DFrontMtx_initializeFront (
   DFrontMtx   *frontmtx,
   DChv        *frontJ,
   int         J
) {
int   ncolJ, nD, nrowJ ;
int   *colindJ, *ivec ;
/*
   -----------------------------------
   get the number of internal vertices
   -----------------------------------
*/
nD = ETree_frontSize(frontmtx->frontETree, J) ;
/*
   --------------------------------------
   get the internal and boundary vertices
   --------------------------------------
*/
IVL_listAndSize(frontmtx->symbfacIVL, J, &ncolJ, &colindJ) ;
#if MYDEBUG > 0
fprintf(stdout, "\n front %d, column indices", J) ;
IVfprintf(stdout, ncolJ, colindJ) ;
fflush(stdout) ;
#endif
/*
   ------------------------------------------------------
   initialize the front's dimensions, indices and entries
   ------------------------------------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n front %d, nD %d, nU %d", J, nD, ncolJ - nD) ;
fflush(stdout) ;
#endif
DChv_init(frontJ, J, nD, ncolJ - nD, ncolJ - nD, 
          frontmtx->symmetryflag) ;
/*
   -----------------------
   fill the column indices
   -----------------------
*/
DChv_columnIndices(frontJ, &ncolJ, &ivec) ;
IVcopy(ncolJ, ivec, colindJ) ;
#if MYDEBUG > 0
fprintf(stdout, "\n front %d, colind = %p", J, ivec) ;
IVfprintf(stdout, ncolJ, ivec) ;
fflush(stdout) ;
#endif
if ( frontmtx->symmetryflag == 2 ) {
/*
   --------------------
   fill the row indices
   --------------------
*/
   DChv_rowIndices(frontJ, &nrowJ, &ivec) ;
   IVcopy(nrowJ, ivec, colindJ) ;
}
/*
   ------------------------
   zero the front's entries
   ------------------------
*/
DChv_zero(frontJ) ;
#if MYDEBUG > 0
fprintf(stdout, "\n leaving DFront_initializeFront()") ;
DChv_writeForHumanEye(frontJ, stdout) ;
fflush(stdout) ;
#endif

return ; }

/*--------------------------------------------------------------------*/
