/*  factor2.c  */

#include "../DFrontMtx.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization of A.
   this is a wrapper method around DFrontMtx_factor().

   input --

      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file

   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factorDInpMtx (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtx,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) {
DChv      *rootchv ;
DPencil   pencil ;

DPencil_setDefaultFields(&pencil) ;
DPencil_init(&pencil, inpmtx, 0.0, NULL) ;
rootchv = DFrontMtx_factor(frontmtx, &pencil, tau, droptol, cpus,
                           stats, msglvl, msgFile) ;

return(rootchv) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization 
   of A + sigma*B. this is a wrapper method around DFrontMtx_factor().

   input --

      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file

   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factorDPencil (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) {
DChv      *rootchv ;

rootchv = DFrontMtx_factor(frontmtx, pencil, tau, droptol, cpus,
                           stats, msglvl, msgFile) ;

return(rootchv) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization
   of A - sigma*B.

   input --

      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file

   created -- 97jul18, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factor (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) {
DChvList      *postponedlist ;
DChvManager   *manager ;
DChv          *frontJ, *keepfront, *rootchv ;
double        t0, t1, t2, t3 ;
DV            tmpDV ;
ETree         *frontETree ;
int           J, K, ndelay, ndelayed, nDJ, 
              nelim, nfront, nLJ, npivots, npost, ntests, nUJ ;
int           *head, *link, *offsets, *par ;
IV            markIV, pivotsizesIV ;
Tree          *tree ;
/*
   ---------------
   check the input
   ---------------
*/
MARKTIME(t0) ;
if ( frontmtx == NULL || pencil == NULL || cpus == NULL || stats == NULL
   || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_factor()"
           "\n frontmtx = %p, pencil = %p"
           "\n tau = %e, droptol = %e, cpus = %p"
           "\n msglvl = %d, msgFile = %p"
           "\n bad input\n",
           frontmtx, pencil, tau, droptol, cpus, msglvl, msgFile) ;
   exit(-1) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n INSIDE DFrontMtx_factor()") ;
   fflush(msgFile) ;
}
/*
   -------------------------------
   extract pointers and dimensions
   -------------------------------
*/
frontETree = frontmtx->frontETree ;
nfront     = ETree_nfront(frontETree) ;
tree       = ETree_tree(frontETree) ;
par        = ETree_par(frontETree) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n got pointers and dimensions") ;
   fflush(msgFile) ;
}
/*
   ---------------------------------------
   allocate and initialize working storage
   ---------------------------------------
*/
head    = IVinit(nfront, -1) ;
link    = IVinit(nfront, -1) ;
offsets = IVinit(nfront, 0) ;
DV_setDefaultFields(&tmpDV) ;
IV_setDefaultFields(&markIV) ;
IV_setDefaultFields(&pivotsizesIV) ;
manager = DChvManager_new() ;
postponedlist = DChvList_new() ;
DChvList_init(postponedlist, nfront+1, NULL, 0, NULL) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n allocated working storage") ;
   fflush(msgFile) ;
}
keepfront = DFrontMtx_biggestFront(frontmtx) ;
/*
   --------------------------------------------
   loop over the tree in a post-order traversal
   --------------------------------------------
*/
npivots = ndelayed = ntests = 0 ;
rootchv = NULL ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
   K = par[J] ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n\n ##### working on front %d, parent %d", 
              J, K) ;
      fflush(msgFile) ;
   }
/*
   --------------------
   initialize the front
   --------------------
*/
   frontJ = keepfront ;
   MARKTIME(t1) ;
   DFrontMtx_initializeFront(frontmtx, frontJ, J) ;
   DChv_dimensions(frontJ, &nDJ, &nLJ, &nUJ) ;
   MARKTIME(t2) ;
   cpus[0] += t2 - t1 ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, "\n\n front %d after initialization", J) ;
      DChv_writeForHumanEye(frontJ, msgFile) ;
   }
/*
   -------------------------
   load the original entries
   -------------------------
*/
   MARKTIME(t1) ;
   DFrontMtx_loadEntries(frontJ, pencil, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[1] += t2 - t1 ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, 
              "\n\n front %d after original entries loaded", J) ;
      DChv_writeForHumanEye(frontJ, msgFile) ;
   }
/*
   ----------------------------------
   get updates from descendent fronts
   ----------------------------------
*/
   MARKTIME(t1) ;
   DFrontMtx_update(frontmtx, frontJ, head, link, offsets, 
                    &tmpDV, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[2] += t2 - t1 ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, "\n\n front %d after updates", J) ;
      DChv_writeForHumanEye(frontJ, msgFile) ;
   }
/*
   ---------------------------
   assemble any postponed data
   ---------------------------
*/
   MARKTIME(t1) ;
   ndelay = 0 ;
   if ( frontmtx->pivotingflag == 1 ) {
      frontJ = DFrontMtx_assemblePostponedData(frontmtx, frontJ, 
                  postponedlist, manager, &ndelay) ;
      if ( msglvl > 3 ) {
         fprintf(msgFile, 
           "\n\n front %d after assembling postponed data, ndelay = %d",
           J, ndelay) ;
         DChv_writeForHumanEye(frontJ, msgFile) ;
      }
      DChv_dimensions(frontJ, &nDJ, &nLJ, &nUJ) ;
      if ( msglvl > 3 ) {
         fprintf(msgFile, 
           "\n\n front %d after assembling postponed data, ndelay = %d",
           J, ndelay) ;
         DChv_writeForHumanEye(frontJ, msgFile) ;
      }
   }
   MARKTIME(t2) ;
   cpus[3] += t2 - t1 ;
/*
   ----------------
   factor the front
   ----------------
*/
   MARKTIME(t1) ;
   nelim = DChv_factor(frontJ, ndelay, frontmtx->pivotingflag,
                       &pivotsizesIV, &markIV, tau, &ntests) ;
   npost = nDJ - nelim ;
   MARKTIME(t2) ;
   cpus[4] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, 
              "\n\n front %d after factoring, nelim = %d, npost = %d", 
              J, nelim, npost) ;
      DChv_writeForHumanEye(frontJ, msgFile) ;
   }
   ndelayed += npost ;
   if ( frontmtx->symmetryflag == 0 && frontmtx->pivotingflag == 1 ) {
      npivots += IV_size(&pivotsizesIV) ;
   } else {
      npivots += nelim ;
   }
#if MYDEBUG > 0
   fprintf(msgFile, "\n nelim = %d, nDJ = %d, npost = %d", 
           nelim, nDJ, npost) ;
#endif
   if ( nelim > 0 && K != -1 ) {
/*
      -----------------------------------------
      row(s) and column(s) have been eliminated
      link the front to its first ancestor
      -----------------------------------------
*/
      link[J] = head[K] ;
      head[K] = J ;
      offsets[J] = frontJ->nD ;
   }
/*
   -----------------------------------------------
   store factor entries and indices (if necessary)
   -----------------------------------------------
*/
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n preparing to store factor data") ;
   }
   MARKTIME(t1) ;
   frontJ->nD -= npost ;
   frontJ->nU += npost ;
   frontJ->nL += npost ;
   DFrontMtx_storeFront(frontmtx, frontJ, &pivotsizesIV, droptol) ;
   frontJ->nD += npost ;
   frontJ->nU -= npost ;
   frontJ->nL -= npost ;
   MARKTIME(t2) ;
   cpus[6] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n factor data stored") ;
   }
   if ( npost > 0 ) {
      MARKTIME(t1) ;
/*
      --------------------------------------
      there is postponed data, extract the 
      postponed chevron and link with parent
      --------------------------------------
*/
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n preparing to store postponed data") ;
      }
      DFrontMtx_storePostponedData(frontmtx, frontJ, npost, K,
                                   postponedlist, manager) ;
      MARKTIME(t2) ;
      cpus[5] += t2 - t1 ;
      if ( msglvl > 2 ) {
         fprintf(msgFile, "\n\n postponed data stored") ;
      }
   }
   if ( frontJ != keepfront ) {
      DChvManager_releaseObject(manager, frontJ) ;
   }
}
/*
   ------------------
   set the statistics
   ------------------
*/
stats[0] = npivots  ;
stats[1] = ntests   ;
stats[2] = ndelayed ;
stats[3] = frontmtx->diagDVL->tsize ;
if ( frontmtx->lowerDVL != NULL ) {
   stats[4] = frontmtx->lowerDVL->tsize ;
} else {
   stats[4] = 0 ;
}
stats[5] = frontmtx->upperDVL->tsize ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
IVfree(head) ;
IVfree(link) ;
IVfree(offsets) ;
DV_clearData(&tmpDV) ;
IV_clearData(&markIV) ;
IV_clearData(&pivotsizesIV) ;
DChvList_free(postponedlist) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n front %p, %d bytes, free'd", 
           keepfront, DChv_nbytesInWorkspace(keepfront)) ;
   fflush(msgFile) ;
}
DChv_free(keepfront) ;
if ( msglvl > 1 ) {
   DChvManager_writeForHumanEye(manager, msgFile) ;
}
DChvManager_free(manager) ;
/*
   --------------------------------
   set final and miscellaneous time
   --------------------------------
*/
MARKTIME(t3) ;
cpus[8] = t3 - t0 ;
cpus[7] = cpus[8] - cpus[0] - cpus[1] - cpus[2]
        - cpus[3] - cpus[4] - cpus[5] - cpus[6] ;

return(rootchv) ; }

/*--------------------------------------------------------------------*/
