/*  QRsolveMT.c  */

#include "../DFrontMtx.h"
#include "../../Ideq.h"
#include "../../timings.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   a multithreaded solve of a linear system A X = B 

   solve (U^T + D) * (D + U) sol = rhs
      D is diagonal

   frontmtx   -- object that stores the factors of A
   inpmtxA    -- object that stores the input matix of A
   newToOldIV -- object that stores the new-to-old permutation of A
   solDA2     -- object that stores the solution X
   rhsDA2     -- object that stores the solution B
   ownersIV   -- object that stores the map from fronts to threads
   cpus       -- vector of cpu time breakdowns
      cpus[0] -- initialize rhs matrices
      cpus[1] -- load rhs matrices with rhs 
      cpus[2] -- assemble from children and parent
      cpus[3] -- solve and update
      cpus[4] -- store entries
      cpus[5] -- link and free matrices
      cpus[6] -- miscellaneous
      cpus[7] -- total time
   msglvl   -- message level
   msgFile  -- message file

   created -- 97jun27, dkw
   -------------------------------------------------------------
*/
void
DFrontMtx_MT_QRsolve (
   DFrontMtx          *frontmtx,
   DInpMtx            *inpmtxA,
   IV                 *newToOldIV,
   DA2                *solDA2,
   DA2                *rhsDA2,
   IV                 *ownersIV,
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) {
DA2               *AtranRHS ;
DDenseMtxManager  *manager ; 
double            *dvec, *rhs, *rhsloc ;
double            cpumisc, cpusolve, cputotal, t0, t1, t2, t3 ;
int               col, ii, jrhs, ncolsol, nent, nrhs, nrowsol, rc, row ;
int               *ivec1, *ivec2, *ivptr ;

/*
   ---------------
   check the input
   ---------------
*/
if (  frontmtx == NULL || inpmtxA == NULL || newToOldIV == NULL
   || solDA2 == NULL || rhsDA2 == NULL || ownersIV == NULL
   || cpus == NULL || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_MT_QRsolve()"
           "\n frontmtx = %p, inpmtxA = %p, newToOldIV = %p"
           "\n solDA2 = %p, rhsDA2 = %p, ownersIV = %p"
           "\n cpus = %p, msglvl = %d, msgFile = %p"
           "\n\n bad input\n", frontmtx, inpmtxA, newToOldIV, solDA2,
           rhsDA2, ownersIV, cpus, msglvl, msgFile) ; 
   exit(-1) ;
}
MARKTIME(t0) ;
/*
   ---------------------------------
   set inpmtxA and initilize manager
   ---------------------------------
*/
rc = DInpMtx_coordType(inpmtxA) ;
if ( rc != 2 ) DInpMtx_changeCoordType(inpmtxA, 2) ;
rc = DInpMtx_storageMode(inpmtxA) ;
if ( rc != 1 ) DInpMtx_changeStorageMode(inpmtxA, 1) ;

manager = DDenseMtxManager_new() ;
DDenseMtxManager_init(manager, 1) ;
/*
   --------------------------
   set up the working storage
   --------------------------
*/
AtranRHS = DA2_new() ;
nrowsol = DA2_nrow(solDA2) ;
ncolsol = DA2_ncol(solDA2) ;
DA2_init(AtranRHS, nrowsol, ncolsol, 1, nrowsol, NULL) ;
DVzero(nrowsol*ncolsol, AtranRHS->entries) ;
/*
   ------------------------------
   replace the rhs with A^T * rhs
   ------------------------------
*/
nrhs  = DA2_ncol(rhsDA2) ;
ivec1 = DInpMtx_ivec1(inpmtxA) ;
ivec2 = DInpMtx_ivec2(inpmtxA) ;
dvec  = DInpMtx_dvec(inpmtxA) ;
nent  = inpmtxA->nent ;
for ( jrhs = 0 ; jrhs < nrhs ; jrhs++ ) {
   rhs    = DA2_column(rhsDA2, jrhs) ;
   rhsloc = DA2_column(AtranRHS, jrhs) ;
   for ( ii = 0 ; ii < nent ; ii++ ) {
       row = ivec1[ii] ;
       col = ivec2[ii] ;
       rhsloc[row] += dvec[ii] * rhs[col] ;
   }
}
/*
   -----------------------------------
   solve (U^T + D) * (D + U) sol = rhs
   -----------------------------------
*/
MARKTIME(t1) ;
DFrontMtx_MT_solve(frontmtx, solDA2, AtranRHS, manager, ownersIV,
                  cpus, msglvl, msgFile) ;
MARKTIME(t2) ;
cpusolve = t2 - t1 ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n CPU %8.3f : solveMT call", t2 - t1) ;
   fflush(msgFile) ;
}
/*
   -------------------------------------------
   permute solution back to the original order
   -------------------------------------------
*/
ivptr = IV_entries(newToOldIV) ;
DA2_copy( AtranRHS, solDA2 ) ;
DA2_scatter(solDA2, ivptr, NULL, AtranRHS) ;
 
#if MYDEBUG > 0
   fprintf(stdout, "\n\n solution in original form - solDA2 ") ;
   DA2_writeForHumanEye(solDA2, stdout)  ;
   fflush(stdout) ;
#endif
/*
   ---------------------
   free the working data
   ---------------------
*/
MARKTIME(t1) ;
DA2_free(AtranRHS) ;
DDenseMtxManager_free(manager) ;
MARKTIME(t2) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n CPU %8.3f : free working data", t2 - t1) ;
   fflush(msgFile) ;
}
MARKTIME(t3) ;
cputotal = t3 - t0 ;
cpumisc  = cputotal - cpusolve ;
cpus[6] += cpumisc ;
cpus[7] += cpumisc ;
return ; }
