/*  QRfactor.c  */

#include "../DFrontMtx.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------
   serial QR factorization
   compute the cholesky factor of A^T*A

   input --
 
      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      inpmtxA  -- pointer to the DInpMtx object that holds the
                  entries of A, cannot be NULL

      facops   -- QR factorization ops count

      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- assembled updates
         cpus[3] -- form staircase and lastrowIV
         cpus[4] -- factor the front
         cpus[5] -- store factor entries
         cpus[6] -- store update entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time

      msglvl   -- message level
      msgFile  -- message file

      created -- 97apr02, dkw
   ----------------------------------------------------
*/
void
DFrontMtx_QRfactor (
   DFrontMtx     *frontmtx,
   DInpMtx       *inpmtxA,
   double        *facops,
   double        cpus[],
   int           msglvl,
   FILE          *msgFile
) {
DA2           tmpDA2, *factormtx ;
DChv          *childi, *firstchild ;
DChvList      *updatelist ;
DChvManager   *manager ;
double        opscnt, t0, t1, t2, t3 ;
double        *entries ;
DStairMtx     *stairfrontJ ;
ETree         *frontETree ;
int           count, irow, jcol, J, K, ncolA, ncolJ, nD, nfront,
              nJ, nrowA, nrowJ, nupdrow, rc, size, tmprowJ, totsize ;
int           *frontSizes, *head, *indices, *link, *list,
              *parent, *vtxToFront ;
IVL           *rowsIVL, *symbfacIVL ;
Tree          *tree ;

MARKTIME(t0) ;
/*
   ---------------
   check the input
   ---------------
*/
if (   frontmtx == NULL 
   || (inpmtxA == NULL)
   || cpus == NULL
   || (msglvl > 0 && msgFile == NULL) ) {
   fprintf(stderr, 
       "\n fatal error in DFrontMtx_QRfactor(%p,%p,%p,%d,%p)"
           "\n bad input\n", frontmtx, inpmtxA,
           cpus, msglvl, msgFile) ;
   exit(-1) ;
}
/*
   -------------------------------
   extract pointers and dimensions
   -------------------------------
*/
nfront       = frontmtx->nfront ;
frontETree   = frontmtx->frontETree ;
frontSizes   = IV_entries(frontETree->nodwghtsIV) ;
vtxToFront   = IV_entries(frontETree->vtxToFrontIV) ;
tree         = frontETree->tree ;
parent       = tree->par ;
symbfacIVL   = frontmtx->symbfacIVL ;
DA2_setDefaultFields(&tmpDA2) ;

/*
   ----------------------------
   allocate the working storage
   ----------------------------
*/
manager = DChvManager_new() ;
updatelist = DChvList_new() ;
DChvList_init(updatelist, nfront+1, NULL, 0, NULL) ;

/*
   -----------------------------------------------------
   get a head[nfront] and link[nrowA] map pair,
   head[J]    = first row from A that goes into front J
   link[irow] = next row from A that goes into the same
                front as irow
   -----------------------------------------------------
*/
nrowA = 1 + IVmax(DInpMtx_nent(inpmtxA), DInpMtx_ivec1(inpmtxA), &rc) ;
ncolA = 1 + IVmax(DInpMtx_nent(inpmtxA), DInpMtx_ivec2(inpmtxA), &rc) ;

if ( msglvl > 1 ) {
   fprintf(msgFile, "\n inside DFrontMtx_QRfactor nrowA = %d  ncolA = %d \n",
           nrowA, ncolA);
   fflush(msgFile) ;
}
if ( msglvl > 3 ) {
   DInpMtx_writeForHumanEye(inpmtxA, msgFile) ;
   fflush(msgFile) ;
}

head = IVinit(nfront, -1) ;
link = IVinit(nrowA,  -1) ;
for ( irow = 0 ; irow < nrowA ; irow++ ) {
   DInpMtx_vector(inpmtxA, irow, &size, &indices, &entries) ;
   if ( size > 0 ) {
      jcol       = indices[0] ;
      J          = vtxToFront[jcol] ;
      link[irow] = head[J] ;
      head[J]    = irow ;
   }
}
/*
   ------------------------------------------
   get a IVL object to hold the list of rows
   in A that goes into each front
   ------------------------------------------
*/
rowsIVL = IVL_new() ;
IVL_init2(rowsIVL, IVL_CHUNKED, nfront, nrowA) ;
list = IVinit(ncolA, -1) ;
for ( J = 0 ; J < nfront ; J++ ) {
    count = 0 ;
    for ( irow = head[J] ; irow != -1 ; irow = link[irow] ) {
        list[count++] = irow ;
    }
    if ( count > 0 ) {
       IVqsortUp(count, list) ;
       IVL_setList(rowsIVL, J, count, list) ;
    }
}
if ( msglvl > 3 ) {
   fprintf(msgFile, " \n rowsIVL for all %d fronts \n",J);
   fflush(msgFile) ;
   rc = IVL_writeForHumanEye(rowsIVL, msgFile) ;
}
/*
   --------------------------------------------
   loop over the tree in a post-order traversal
   --------------------------------------------
*/
*facops = 0.0 ;
for ( J = Tree_postOTfirst(tree) ;
      J != -1 ;
      J = Tree_postOTnext(tree, J) ) {
      K = parent[J] ;
/*
   -------------------------
   initialize the stairfront
   -------------------------
*/
   MARKTIME(t1) ;
   nJ  = frontSizes[J] ;
   IVL_listAndSize(symbfacIVL, J, &totsize, &indices) ;
   if ( msglvl > 1 ) {
      fprintf(msgFile, 
              "\n\n ### visiting front %d, nJ = %d, nbndJ = %d", 
              J, nJ, totsize - nJ) ;
      fflush(msgFile) ;
   }

   nrowJ = rowsIVL->sizes[J] ;
   ncolJ = totsize ;
   tmprowJ = nrowJ ;

   firstchild = DChvList_getList(updatelist, J) ;
   for ( childi = firstchild ; childi != NULL ; childi = childi->next ) {
       nD = childi->nD ;
       nrowJ += nD ;
   }
   if ( msglvl > 1 ) {
      fprintf(msgFile, "\n initialize the stairmtx objects ") ;
      fprintf(msgFile, 
              "\n\n ### stairinit nrowJ %d, ncolJ = %d, totupdrows = %d", 
              nrowJ, ncolJ, nrowJ-tmprowJ) ;
      fflush(msgFile) ;
   }
   stairfrontJ = DStairMtx_new() ;
   DStairMtx_init(stairfrontJ, J, nrowJ, ncolJ, nJ) ;

   MARKTIME(t2) ;
   cpus[0] += t2 - t1 ;
/*
   -------------------------
   load the original entries
   -------------------------
*/
   MARKTIME(t1) ;
   DFrontMtx_QRloadEntries(stairfrontJ, J, inpmtxA, symbfacIVL,
                           rowsIVL, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[1] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, " \n after QRloadEntries for J = %d\n", J);
      fflush(msgFile) ;
   }
/*
   -------------------------------------
   assemble updates from childern fronts
   -------------------------------------
*/
   MARKTIME(t1) ;
   if ( firstchild != NULL ) {
      DFrontMtx_QRassembleUpdates(stairfrontJ, J, symbfacIVL, rowsIVL,
                                  manager, firstchild, msglvl, msgFile) ;
   }
   MARKTIME(t2) ;
   cpus[2] += t2 - t1 ;
   if ( firstchild != NULL && msglvl > 2 ) {
      fprintf(msgFile, " \n after QRassembleUpdates for J = %d\n", J);
      fflush(msgFile) ;
   }
/*
   -----------------------------
   form staircase and lastrowIVL
   -----------------------------
*/
   MARKTIME(t1) ;
   DStairMtx_formStaircase(stairfrontJ, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[3] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, " \n after formStaircase for J = %d\n", J);
      if ( msglvl > 4 ) DStairMtx_writeForHumanEye(stairfrontJ, msgFile) ;
      fflush(msgFile) ;
   }
/*
   ----------------
   factor the front
   ----------------
*/
   MARKTIME(t1) ;
   DFrontMtx_QRfactorFront(stairfrontJ, J, &opscnt, msglvl, msgFile) ;
   *facops += opscnt ;
   MARKTIME(t2) ;
   cpus[4] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n\n after QRfactorFront" ) ;
      fflush(msgFile) ;
   }
/*
   -----------------------
   store the update matrix
   -----------------------
*/
   MARKTIME(t1) ;
   factormtx = &tmpDA2 ;
   nupdrow = stairfrontJ->nupdrow ;
   DA2_subDA2(factormtx, &stairfrontJ->stairmtx, nJ, nJ+nupdrow-1,
              nJ, totsize-1) ;
   DFrontMtx_QRstoreUpdate(factormtx, updatelist, manager, J, K, nJ,
                           totsize, nupdrow, indices, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[5] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n finished storing update for front = %d\n", J) ;
      fflush(msgFile) ;
   }
/*
   -------------------
   store the QR factor
   -------------------
*/
   MARKTIME(t1) ;
   DA2_subDA2(factormtx, &stairfrontJ->stairmtx, 0, nJ-1, 0, totsize-1) ;
   DFrontMtx_QRstoreFactor(frontmtx, factormtx, J, msglvl, msgFile) ;
   MARKTIME(t2) ;
   cpus[6] += t2 - t1 ;
   if ( msglvl > 2 ) {
      fprintf(msgFile, "\n finished storing QR factor for front = %d\n", J) ;
      fflush(msgFile) ;
   }
/*
   -------------------
   free the stairfront
   -------------------
*/
   DStairMtx_free(stairfrontJ) ;
}

if ( msglvl > 2 ) {
   fprintf(msgFile, " \n total QR facops = %8.3f\n", *facops);
   fflush(msgFile) ;
}
if ( msglvl > 4 ) {
   fprintf(msgFile, "\n\n diagonal entries DVL object ") ;
   DVL_writeForHumanEye(frontmtx->diagDVL, msgFile) ;
   fprintf(msgFile, "\n\n upper    entries DVL object ") ;
   DVL_writeForHumanEye(frontmtx->upperDVL, msgFile) ;
   fflush(msgFile) ;
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
DA2_clearData(&tmpDA2) ;
DChvList_free(updatelist) ;
DChvManager_free(manager) ;
IVfree(head) ;
IVfree(link) ;
IVfree(list) ;
IVL_free(rowsIVL) ;

MARKTIME(t3) ;
cpus[8] = t3 - t0 ;
cpus[7] = cpus[8] - cpus[0] - cpus[1] - cpus[2] - cpus[3] 
        - cpus[4] - cpus[5] - cpus[6] ;

return ; }

/*--------------------------------------------------------------------*/
