/*  IO.c  */

#include "../DFrontMtx.h"

static const char *suffixb = ".dfrontmtxb" ;
static const char *suffixf = ".dfrontmtxf" ;

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   purpose -- to read in an DFrontMtx object from a file

   input --

      fn -- filename, must be *.frontmtxb or *.frontmtxf

   return value -- 1 if success, 0 if failure

   created -- 97jun18, cca
   -----------------------------------------------------
*/
int
DFrontMtx_readFromFile ( 
   DFrontMtx   *frontmtx, 
   char        *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fn == NULL ) {
   fprintf(stderr, "\n error in DFrontMtx_readFromFile(%p,%s)"
           "\n bad input\n", frontmtx, fn) ;
   return(0) ;
}
/*
   -------------
   read the file
   -------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "rb")) == NULL ) {
         fprintf(stderr, "\n error in DFrontMtx_readFromFile(%p,%s)"
                 "\n unable to open file %s", frontmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DFrontMtx_readFromBinaryFile(frontmtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "r")) == NULL ) {
         fprintf(stderr, "\n error in DFrontMtx_readFromFile(%p,%s)"
                 "\n unable to open file %s", frontmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DFrontMtx_readFromFormattedFile(frontmtx, fp) ;
         fclose(fp) ;
      }
   } else {
      fprintf(stderr, "\n error in DFrontMtx_readFromFile(%p,%s)"
              "\n bad DFrontMtx file name %s,"
              "\n must end in %s (binary) or %s (formatted)\n",
              frontmtx, fn, fn, suffixb, suffixf) ;
      rc = 0 ;
   }
} else {
   fprintf(stderr, "\n error in DFrontMtx_readFromFile(%p,%s)"
       "\n bad DFrontMtx file name %s,"
       "\n must end in %s (binary) or %s (formatted)\n",
       frontmtx, fn, fn, suffixb, suffixf) ;
   rc = 0 ;
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- to read an DFrontMtx object from a formatted file

   return value -- 1 if success, 0 if failure

   created -- 97jun18, cca
   ------------------------------------------------------------
*/
int
DFrontMtx_readFromFormattedFile ( 
   DFrontMtx   *frontmtx, 
   FILE        *fp 
) {
int    rc ;
int    itemp[6] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DFrontMtx_clearData(frontmtx) ;
/*
   -----------------------------
   read in the six scalar fields
   -----------------------------
*/
if ( (rc = IVfscanf(fp, 6, itemp)) != 6 ) {
   fprintf(stderr, "\n error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", frontmtx, fp, rc, 6) ;
   return(0) ;
}
frontmtx->nfront       = itemp[0] ;
frontmtx->neqns        = itemp[1] ;
frontmtx->symmetryflag = itemp[2] ;
frontmtx->sparsityflag = itemp[3] ;
frontmtx->pivotingflag = itemp[4] ;
#if MYDEBUG > 0
fprintf(stdout, 
        "\n\n nfront        = %d"
        "\n neqns         = %d"
        "\n symmetryflag  = %d"
        "\n sparsityflag  = %d"
        "\n pivotingflag  = %d",
        frontmtx->nfront,
        frontmtx->neqns,
        frontmtx->symmetryflag,
        frontmtx->sparsityflag,
        frontmtx->pivotingflag) ;
fflush(stdout) ;
#endif
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
frontmtx->frontETree = ETree_new() ;
rc = ETree_readFromFormattedFile(frontmtx->frontETree, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n error %d reading in frontETree object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n\n ETree object") ;
ETree_writeForHumanEye(frontmtx->frontETree, stdout) ;
fflush(stdout) ;
#endif
/*
   ---------------------------------------------
   read in the symbolic factorization IVL object
   ---------------------------------------------
*/
frontmtx->symbfacIVL = IVL_new() ;
frontmtx->symbfacIVL->type = IVL_CHUNKED ;
rc = IVL_readFromFormattedFile(frontmtx->symbfacIVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n error %d reading in symbfacIVL object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -------------------------------
   read in the frontsizesIV object
   -------------------------------
*/
frontmtx->frontsizesIV = IV_new() ;
rc = IV_readFromFormattedFile(frontmtx->frontsizesIV, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n error %d reading in frontsizesIV object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
if (  frontmtx->pivotingflag == 1 
   && frontmtx->symmetryflag == 0 ) {
/*
   ----------------------------
   read in the pivotsIVL object
   ----------------------------
*/
   frontmtx->pivotsIVL = IVL_new() ;
   frontmtx->pivotsIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromFormattedFile(frontmtx->pivotsIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
              "\n error %d reading in pivotsIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( frontmtx->pivotingflag == 1 ) {
/*
   ----------------------------
   read in the coladjIVL object
   ----------------------------
*/
   frontmtx->coladjIVL = IVL_new() ;
   frontmtx->coladjIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromFormattedFile(frontmtx->coladjIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
              "\n error %d reading coladjIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ----------------------------
      read in the rowadjIVL object
      ----------------------------
*/
      frontmtx->rowadjIVL = IVL_new() ;
      frontmtx->rowadjIVL->type = IVL_CHUNKED ;
      rc = IVL_readFromFormattedFile(frontmtx->rowadjIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
              "\n error %d reading rowadjIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( frontmtx->sparsityflag == 1 ) {
/*
   ---------------------------
   read in the upperIVL object
   ---------------------------
*/
   frontmtx->upperIVL = IVL_new() ;
   frontmtx->upperIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromFormattedFile(frontmtx->upperIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n error %d reading upperIVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ---------------------------
      read in the lowerIVL object
      ---------------------------
*/
      frontmtx->lowerIVL = IVL_new() ;
      frontmtx->lowerIVL->type = IVL_CHUNKED ;
      rc = IVL_readFromFormattedFile(frontmtx->lowerIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
              "\n error %d reading lowerIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
/*
   --------------------------
   read in the diagDVL object
   --------------------------
*/
frontmtx->diagDVL = DVL_new() ;
frontmtx->diagDVL->type = DVL_CHUNKED ;
rc = DVL_readFromFormattedFile(frontmtx->diagDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
        "\n error %d reading diagDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------------
   read in the upperDVL object
   ---------------------------
*/
frontmtx->upperDVL = DVL_new() ;
frontmtx->upperDVL->type = DVL_CHUNKED ;
rc = DVL_readFromFormattedFile(frontmtx->upperDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
        "\n error %d reading upperDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
if ( frontmtx->symmetryflag == 2 ) {
/*
   ---------------------------
   read in the lowerDVL object
   ---------------------------
*/
   frontmtx->lowerDVL = DVL_new() ;
   frontmtx->lowerDVL->type = DVL_CHUNKED ;
   rc = DVL_readFromFormattedFile(frontmtx->lowerDVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromFormattedFile(%p,%p)"
           "\n error %d reading lowerDVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( itemp[5] == 1 ) {
/*
   -----------------
   allocate the lock
   -----------------
*/
   frontmtx->lock = Lock_new() ;
   Lock_init(frontmtx->lock, itemp[5]) ;
/*
#if THREAD_TYPE == TT_SOLARIS
   ALLOCATE(frontmtx->lock, mutex_t, 1) ;
   if ( itemp[5] == 1 ) {
      mutex_init(frontmtx->lock, USYNC_THREAD, NULL) ;
   } else if ( itemp[5] == 2 ) {
      mutex_init(frontmtx->lock, USYNC_PROCESS, NULL) ;
   }
#endif
#if THREAD_TYPE == TT_POSIX
   ALLOCATE(frontmtx->lock, pthread_mutex_t, 1) ;
   pthread_mutex_init(frontmtx->lock, NULL) ;
#endif
*/
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- to read an DFrontMtx object from a binary file

   return value -- 1 if success, 0  if failure

   created -- 97jun18, cca
   ---------------------------------------------------------
*/
int
DFrontMtx_readFromBinaryFile ( 
   DFrontMtx   *frontmtx, 
   FILE        *fp 
) {
int    rc ;
int    itemp[6] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DFrontMtx_clearData(frontmtx) ;
/*
   -----------------------------
   read in the six scalar fields
   -----------------------------
*/
if ( (rc = fread((void *) itemp, sizeof(int), 6, fp)) != 6 ) {
   fprintf(stderr, "\n error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n %d items of %d read\n", frontmtx, fp, rc, 6) ;
   return(0) ;
}
frontmtx->nfront       = itemp[0] ;
frontmtx->neqns        = itemp[1] ;
frontmtx->symmetryflag = itemp[2] ;
frontmtx->sparsityflag = itemp[3] ;
frontmtx->pivotingflag = itemp[4] ;
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
frontmtx->frontETree = ETree_new() ;
rc = ETree_readFromBinaryFile(frontmtx->frontETree, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n error %d reading in frontETree object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------------------------------
   read in the symbolic factorization IVL object
   ---------------------------------------------
*/
frontmtx->symbfacIVL = IVL_new() ;
frontmtx->symbfacIVL->type = IVL_CHUNKED ;
rc = IVL_readFromBinaryFile(frontmtx->symbfacIVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n error %d reading in symbfacIVL object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -------------------------------
   read in the frontsizesIV object
   -------------------------------
*/
frontmtx->frontsizesIV = IV_new() ;
rc = IV_readFromBinaryFile(frontmtx->frontsizesIV, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n error %d reading in frontsizesIV object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
if (  frontmtx->pivotingflag == 1 
   && frontmtx->symmetryflag == 0 ) {
/*
   ----------------------------
   read in the pivotsIVL object
   ----------------------------
*/
   frontmtx->pivotsIVL = IVL_new() ;
   frontmtx->pivotsIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromBinaryFile(frontmtx->pivotsIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
              "\n error %d reading in pivotsIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( frontmtx->pivotingflag == 1 ) {
/*
   ----------------------------
   read in the coladjIVL object
   ----------------------------
*/
   frontmtx->coladjIVL = IVL_new() ;
   frontmtx->coladjIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromBinaryFile(frontmtx->coladjIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
              "\n error %d reading coladjIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ----------------------------
      read in the rowadjIVL object
      ----------------------------
*/
      frontmtx->rowadjIVL = IVL_new() ;
      frontmtx->rowadjIVL->type = IVL_CHUNKED ;
      rc = IVL_readFromBinaryFile(frontmtx->rowadjIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
              "\n error %d reading rowadjIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( frontmtx->sparsityflag == 1 ) {
/*
   ---------------------------
   read in the upperIVL object
   ---------------------------
*/
   frontmtx->upperIVL = IVL_new() ;
   frontmtx->upperIVL->type = IVL_CHUNKED ;
   rc = IVL_readFromBinaryFile(frontmtx->upperIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n error %d reading upperIVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ---------------------------
      read in the lowerIVL object
      ---------------------------
*/
      frontmtx->lowerIVL = IVL_new() ;
      frontmtx->lowerIVL->type = IVL_CHUNKED ;
      rc = IVL_readFromBinaryFile(frontmtx->lowerIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
              "\n error %d reading lowerIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
/*
   --------------------------
   read in the diagDVL object
   --------------------------
*/
frontmtx->diagDVL = DVL_new() ;
frontmtx->diagDVL->type = IVL_CHUNKED ;
rc = DVL_readFromBinaryFile(frontmtx->diagDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
        "\n error %d reading diagDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------------
   read in the upperDVL object
   ---------------------------
*/
frontmtx->upperDVL = DVL_new() ;
frontmtx->upperDVL->type = IVL_CHUNKED ;
rc = DVL_readFromBinaryFile(frontmtx->upperDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
        "\n error %d reading upperDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
if ( frontmtx->symmetryflag == 2 ) {
/*
   ---------------------------
   read in the lowerIVL object
   ---------------------------
*/
   frontmtx->lowerDVL = DVL_new() ;
   frontmtx->lowerDVL->type = IVL_CHUNKED ;
   rc = DVL_readFromBinaryFile(frontmtx->lowerDVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_readFromBinaryFile(%p,%p)"
           "\n error %d reading lowerDVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( itemp[5] == 1 ) {
/*
   -----------------
   allocate the lock
   -----------------
*/
   frontmtx->lock = Lock_new() ;
   Lock_init(frontmtx->lock, itemp[5]) ;
/*
#if THREAD_TYPE == TT_SOLARIS
   ALLOCATE(frontmtx->lock, mutex_t, 1) ;
   if ( itemp[5] == 1 ) {
      mutex_init(frontmtx->lock, USYNC_THREAD, NULL) ;
   } else if ( itemp[5] == 2 ) {
      mutex_init(frontmtx->lock, USYNC_PROCESS, NULL) ;
   }
#endif
#if THREAD_TYPE == TT_POSIX
   ALLOCATE(frontmtx->lock, pthread_mutex_t, 1) ;
   pthread_mutex_init(frontmtx->lock, NULL) ;
#endif
*/
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   purpose -- to write an DFrontMtx object to a file

   input --

      fn -- filename
        *.frontmtxb -- binary
        *.frontmtxf -- formatted
        anything else -- for human eye

   return value -- 1 if success, 0 otherwise

   created -- 97jun18, cca
   -------------------------------------------------
*/
int
DFrontMtx_writeToFile ( 
   DFrontMtx   *frontmtx, 
   char        *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fn == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_writeToFile(%p,%s)"
    "\n bad input\n", frontmtx, fn) ; 
}
/*
   ------------------
   write out the file
   ------------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "wb")) == NULL ) {
         fprintf(stderr, "\n error in DFrontMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", frontmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DFrontMtx_writeToBinaryFile(frontmtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "w")) == NULL ) {
         fprintf(stderr, "\n error in DFrontMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", frontmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DFrontMtx_writeToFormattedFile(frontmtx, fp) ;
         fclose(fp) ;
      }
   } else {
      if ( (fp = fopen(fn, "a")) == NULL ) {
         fprintf(stderr, "\n error in DFrontMtx_writeToFile(%p,%s)"
                 "\n unable to open file %s", frontmtx, fn, fn) ;
         rc = 0 ;
      } else {
         rc = DFrontMtx_writeForHumanEye(frontmtx, fp) ;
         fclose(fp) ;
      }
   }
} else {
   if ( (fp = fopen(fn, "a")) == NULL ) {
      fprintf(stderr, "\n error in DFrontMtx_writeToFile(%p,%s)"
              "\n unable to open file %s", frontmtx, fn, fn) ;
      rc = 0 ;
   } else {
      rc = DFrontMtx_writeForHumanEye(frontmtx, fp) ;
      fclose(fp) ;
   }
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------
   purpose -- to write an DFrontMtx object to a formatted file

   return value -- 1 if success, 0 otherwise

   created -- 97jun18, cca
   -----------------------------------------------------------
*/
int
DFrontMtx_writeToFormattedFile ( 
   DFrontMtx   *frontmtx, 
   FILE        *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   exit(-1) ;
}
/*
   ---------------------------
   write the six scalar fields
   ---------------------------
*/
rc = fprintf(fp, "\n %d %d %d %d %d %d", 
             frontmtx->nfront, frontmtx->neqns,
             frontmtx->symmetryflag, frontmtx->sparsityflag,
             frontmtx->pivotingflag, (frontmtx->lock != NULL) ? 1 : 0) ;
if ( rc < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n rc = %d, return from first fprintf\n", 
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   --------------------------
   write out the ETree object
   --------------------------
*/
rc = ETree_writeToFormattedFile(frontmtx->frontETree, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n error %d writing frontETree object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -----------------------------------------------
   write out the symbolic factorization IVL object
   -----------------------------------------------
*/
rc = IVL_writeToFormattedFile(frontmtx->symbfacIVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n error %d writing symbfacIVL object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------------------
   write out the frontsizesIV object
   ---------------------------------
*/
rc = IV_writeToFormattedFile(frontmtx->frontsizesIV, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n error %d writing frontsizesIV object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
if (  frontmtx->pivotingflag == 1 
   && frontmtx->symmetryflag == 0 ) {
/*
   ------------------------------
   write out the pivotsIVL object
   ------------------------------
*/
   rc = IVL_writeToFormattedFile(frontmtx->pivotsIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
              "\n error %d writing pivotsIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( frontmtx->pivotingflag == 1 ) {
/*
   ------------------------------
   write out the coladjIVL object
   ------------------------------
*/
   rc = IVL_writeToFormattedFile(frontmtx->coladjIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
              "\n error %d writing coladjIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ------------------------------
      write out the rowadjIVL object
      ------------------------------
*/
      rc = IVL_writeToFormattedFile(frontmtx->rowadjIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
              "\n error %d writing rowadjIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( frontmtx->sparsityflag == 1 ) {
/*
   -----------------------------
   write out the upperIVL object
   -----------------------------
*/
   rc = IVL_writeToFormattedFile(frontmtx->upperIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n error %d writing upperIVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      -----------------------------
      write out the lowerIVL object
      -----------------------------
*/
      rc = IVL_writeToFormattedFile(frontmtx->lowerIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
              "\n error %d writing lowerIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
/*
   ----------------------------
   write out the diagDVL object
   ----------------------------
*/
rc = DVL_writeToFormattedFile(frontmtx->diagDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
        "\n error %d writing diagDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -----------------------------
   write out the upperDVL object
   -----------------------------
*/
rc = DVL_writeToFormattedFile(frontmtx->upperDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
        "\n error %d writing upperDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
if ( frontmtx->symmetryflag == 2 ) {
/*
   -----------------------------
   write out the lowerDVL object
   -----------------------------
*/
   rc = DVL_writeToFormattedFile(frontmtx->lowerDVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToFormattedFile(%p,%p)"
           "\n error %d writing lowerDVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------
   purpose -- to write an DFrontMtx object to a binary file

   return value -- 1 if success, 0 otherwise

   created -- 97jun18, cca
   --------------------------------------------------------
*/
int
DFrontMtx_writeToBinaryFile ( 
   DFrontMtx    *frontmtx, 
   FILE         *fp 
) {
int   rc ;
int   itemp[6] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   exit(-1) ;
}
/*
   ---------------------------
   write the six scalar fields
   ---------------------------
*/
itemp[0] = frontmtx->nfront ;
itemp[1] = frontmtx->neqns ;
itemp[2] = frontmtx->symmetryflag ;
itemp[3] = frontmtx->sparsityflag ;
itemp[4] = frontmtx->pivotingflag ;
itemp[5] = (frontmtx->lock != NULL) ? 1 : 0 ;
rc = fwrite((void *) itemp, sizeof(int), 6, fp) ;
if ( rc != 6 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n rc = %d, return from first fprintf\n", 
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   --------------------------
   write out the ETree object
   --------------------------
*/
rc = ETree_writeToBinaryFile(frontmtx->frontETree, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n error %d writing frontETree object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -----------------------------------------------
   write out the symbolic factorization IVL object
   -----------------------------------------------
*/
rc = IVL_writeToBinaryFile(frontmtx->symbfacIVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n error %d writing symbfacIVL object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
/*
   ---------------------------------
   write out the frontsizesIV object
   ---------------------------------
*/
rc = IV_writeToBinaryFile(frontmtx->frontsizesIV, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n error %d writing frontsizesIV object\n",
           frontmtx, fp, rc) ;
   return(0) ;
}
if (  frontmtx->pivotingflag == 1 
   && frontmtx->symmetryflag == 0 ) {
/*
   ------------------------------
   write out the pivotsIVL object
   ------------------------------
*/
   rc = IVL_writeToBinaryFile(frontmtx->pivotsIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
              "\n error %d writing pivotsIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
}
if ( frontmtx->pivotingflag == 1 ) {
/*
   ------------------------------
   write out the coladjIVL object
   ------------------------------
*/
   rc = IVL_writeToBinaryFile(frontmtx->coladjIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
              "\n error %d writing coladjIVL object\n",
              frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      ------------------------------
      write out the rowadjIVL object
      ------------------------------
*/
      rc = IVL_writeToBinaryFile(frontmtx->rowadjIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
              "\n error %d writing rowadjIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
if ( frontmtx->sparsityflag == 1 ) {
/*
   -----------------------------
   write out the upperIVL object
   -----------------------------
*/
   rc = IVL_writeToBinaryFile(frontmtx->upperIVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n error %d writing upperIVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
   if ( frontmtx->symmetryflag == 2 ) {
/*
      -----------------------------
      write out the lowerIVL object
      -----------------------------
*/
      rc = IVL_writeToBinaryFile(frontmtx->lowerIVL, fp) ;
      if ( rc != 1 ) {
         fprintf(stderr, 
              "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
              "\n error %d writing lowerIVL object\n",
              frontmtx, fp, rc) ;
         return(0) ;
      }
   }
}
/*
   ----------------------------
   write out the diagDVL object
   ----------------------------
*/
rc = DVL_writeToBinaryFile(frontmtx->diagDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
        "\n error %d writing diagDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
/*
   -----------------------------
   write out the upperDVL object
   -----------------------------
*/
rc = DVL_writeToBinaryFile(frontmtx->upperDVL, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, 
        "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
        "\n error %d writing upperDVL object\n",
        frontmtx, fp, rc) ;
   return(0) ;
}
if ( frontmtx->symmetryflag == 2 ) {
/*
   -----------------------------
   write out the lowerDVL object
   -----------------------------
*/
   rc = DVL_writeToBinaryFile(frontmtx->lowerDVL, fp) ;
   if ( rc != 1 ) {
      fprintf(stderr, 
           "\n fatal error in DFrontMtx_writeToBinaryFile(%p,%p)"
           "\n error %d writing lowerDVL object\n",
           frontmtx, fp, rc) ;
      return(0) ;
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   purpose -- to write out the statistics for the DFrontMtx object

   return value -- 1 if success, 0 otherwise

   created -- 97jun18, cca
   ---------------------------------------------------------------
*/
int
DFrontMtx_writeStats ( 
   DFrontMtx   *frontmtx, 
   FILE        *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in DFrontMtx_writeStats(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   exit(-1) ;
}
fprintf(fp, "\n\n DFrontMtx object at address %p"
        "\n nfront = %d", frontmtx, frontmtx->nfront) ;
switch ( frontmtx->symmetryflag ) {
case 0 : 
   fprintf(fp, "\n symmetric structure, symmetric entries") ;
   break ;
case 1 : 
   fprintf(fp, "\n symmetric structure, nonsymmetric entries") ;
   break ;
case 2 : 
   fprintf(fp, "\n nonsymmetric structure, nonsymmetric entries") ;
   break ;
case 3 : 
   fprintf(fp, "\n QR factorization") ;
   break ;
default :
   break ;
}
switch ( frontmtx->sparsityflag ) {
case 0 : 
   fprintf(fp, "\n dense fronts") ;
   break ;
case 1 : 
   fprintf(fp, "\n sparse fronts") ;
   break ;
default :
   break ;
}
switch ( frontmtx->pivotingflag ) {
case 0 : 
   fprintf(fp, "\n pivoting disabled") ;
   break ;
case 1 : 
   fprintf(fp, "\n pivoting enabled") ;
   break ;
default :
   break ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   purpose -- to write the object to a file
              in human readable form

   created -- 97may16, cca
   ----------------------------------------
*/
int
DFrontMtx_writeForHumanEye (
   DFrontMtx   *frontmtx,
   FILE        *fp
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( frontmtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DFrontMtx_writeForHumanEye(%p,%p)"
           "\n bad input\n", frontmtx, fp) ;
   exit(-1) ;
}
DFrontMtx_writeStats(frontmtx, fp) ;
if ( frontmtx->frontETree != NULL ) {
   fprintf(fp, "\n\n front tree DFrontMtx object") ;
   ETree_writeForHumanEye(frontmtx->frontETree, fp) ;
}
if ( frontmtx->symbfacIVL != NULL ) {
   fprintf(fp, "\n\n symbolic factorization IVL object") ;
   IVL_writeForHumanEye(frontmtx->symbfacIVL, fp) ;
}
if ( frontmtx->frontsizesIV != NULL ) {
   fprintf(fp, "\n\n front sizes IV object") ;
   IV_writeForHumanEye(frontmtx->frontsizesIV, fp) ;
}
if ( frontmtx->pivotsIVL != NULL ) {
   fprintf(fp, "\n\n pivots IVL object") ;
   IVL_writeForHumanEye(frontmtx->pivotsIVL, fp) ;
}
if ( frontmtx->rowadjIVL != NULL ) {
   fprintf(fp, "\n\n row adjacency IVL object") ;
   IVL_writeForHumanEye(frontmtx->rowadjIVL, fp) ;
}
if ( frontmtx->coladjIVL != NULL ) {
   fprintf(fp, "\n\n column adjacency IVL object") ;
   IVL_writeForHumanEye(frontmtx->coladjIVL, fp) ;
}
if ( frontmtx->lowerIVL != NULL ) {
   fprintf(fp, "\n\n lower indices IVL object for sparse fronts") ;
   IVL_writeForHumanEye(frontmtx->lowerIVL, fp) ;
}
if ( frontmtx->upperIVL != NULL ) {
   fprintf(fp, "\n\n upper indices IVL object for sparse fronts") ;
   IVL_writeForHumanEye(frontmtx->upperIVL, fp) ;
}
if ( frontmtx->diagDVL != NULL ) {
   fprintf(fp, "\n\n diagonal entries DVL object ") ;
   DVL_writeForHumanEye(frontmtx->diagDVL, fp) ;
}
if ( frontmtx->lowerDVL != NULL ) {
   fprintf(fp, "\n\n lower entries DVL object ") ;
   DVL_writeForHumanEye(frontmtx->lowerDVL, fp) ;
}
if ( frontmtx->upperDVL != NULL ) {
   fprintf(fp, "\n\n upper entries DVL object ") ;
   DVL_writeForHumanEye(frontmtx->upperDVL, fp) ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
