/*  testQRgrid.c  */

#include "../DFrontMtx.h"
#include "../../EGraph.h"
#include "../../SymbFac.h"
#include "../../Drand.h"
#include "../../misc.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------
   test the QR factor method for a DFrontMtx object
   on an n1 x n2 x n3 grid
   (1)  read in a Etree object
   (2)  get the oldToNew permutation from the ETree
   (3)  read in a DInpMtx object
   (4)  create a solution DA2 object X
   (5)  multiply the solution with the matrix
        to get a right hand side DA2 object Y
   (6)  permute the DInpMtx object
   (7)  read in A^TA Graph object
   (8)  get the symbolic factorization of A^TA and
        then permute the IVL
   (9)  permute the ETree object
   (10) factor the matrix 
   (11) solve the systems

   created -- 97apr11, dkw
   ---------------------------------------------------
*/
{
DA2           *X, *Y, *Z ;
DFrontMtx     *frontmtx ;
DInpMtx       *inpmtx ;
double        cputotal, factorops ;
double        cpus[9] ;
Drand         *drand ;
DV            *xDV, *yDV ;
double        nops, t1, t2 ;
double        *entries ;
EGraph        *egraph ;
ETree         *etree, *frontETree ;
FILE          *msgFile ;
Graph         *graph ;
int           ielem, ii, irow, jrhs, jrow, msglvl, nelem, nfront, 
              nrhs, nrowA, nvtx, n1, n2, n3, rc, seed, size,
              symmetryflag ;
int           *indices, *newToOld, *oldToNew ;
IV            *oldToNewIV, *newToOldIV ;
IVL           *adjIVL, *symbfacIVL ;

if ( argc != 8 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile n1 n2 n3 seed nrhs "
      "\n    msglvl  -- message level"
      "\n    msgFile -- message file"
      "\n    n1      -- # of points in the first direction"
      "\n    n2      -- # of points in the second direction"
      "\n    n3      -- # of points in the third direction"
      "\n    seed    -- random number seed"
      "\n    nrhs    -- # of right hand sides"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
n1   = atoi(argv[3]) ;
n2   = atoi(argv[4]) ;
n3   = atoi(argv[5]) ;
seed = atoi(argv[6]) ;
nrhs = atoi(argv[7]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl  -- %d" 
        "\n msgFile -- %s" 
        "\n n1      -- %d" 
        "\n n2      -- %d" 
        "\n n3      -- %d" 
        "\n seed    -- %d" 
        "\n nrhs    -- %d" 
        "\n",
        argv[0], msglvl, argv[2], n1, n2, n3, seed, nrhs) ;
fflush(msgFile) ;
nvtx = n1*n2*n3 ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_setDefaultFields(drand) ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
Drand_setNormal(drand, 0.0, 1.0) ;
/*
   -----------------------------------------------------
   get the grid adjacency structure and set up the graph
   -----------------------------------------------------
*/
MARKTIME(t1) ;
if ( n3 == 1 ) {
   adjIVL = IVL_make9P(n1, n2, 1) ;
} else {
   adjIVL = IVL_make27P(n1, n2, n3, 1) ;
}
graph = Graph_new() ;
Graph_init2(graph, 0, nvtx, 0, adjIVL->tsize, nvtx,
            adjIVL->tsize, adjIVL, NULL, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : create the grid graph",
        t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n grid graph") ;
   Graph_writeForHumanEye(graph, msgFile) ;
}
/*
   ---------------------------------------------
   make the nested dissection permutation vector
   ---------------------------------------------
*/
MARKTIME(t1) ;
newToOld = IVinit(nvtx, -1) ;
oldToNew = IVinit(nvtx, -1) ;
mkNDperm(n1, n2, n3, newToOld, 0, n1-1, 0, n2-1, 0, n3-1) ;
for ( ii = 0 ; ii < nvtx ; ii++ ) {
   oldToNew[newToOld[ii]] = ii ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : make the nested dissection ordering",
        t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n oldToNew") ;
   IVfprintf(msgFile, nvtx, oldToNew) ;
}
/*
   ----------------------------------
   create the elimination tree object
   ----------------------------------
*/
MARKTIME(t1) ;
etree = ETree_new() ;
ETree_initFromGraphWithPerms(etree, graph, newToOld, oldToNew) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n elimination tree") ;
   ETree_writeForHumanEye(etree, msgFile) ;
}
frontETree = ETree_transform(etree, NULL, 0, nvtx, seed) ;
nfront = frontETree->nfront ;
ETree_free(etree) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : create the front tree",
        t2 - t1) ;
/*
   --------------------------------
   create the natural factor matrix
   --------------------------------
*/
MARKTIME(t1) ;
if ( n3 == 1 ) {
   egraph = EGraph_make9P(n1, n2, 1) ;
   entries = DVinit(4, 0.0) ;
} else {
   egraph = EGraph_make27P(n1, n2, n3, 1) ;
   entries = DVinit(8, 0.0) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : create egraph ", t2 - t1) ;
if ( msglvl > 2 ) {
   EGraph_writeForHumanEye(egraph, msgFile) ;
   fflush(msgFile) ;
}
nvtx  = egraph->nvtx  ;
nelem = egraph->nelem ;
MARKTIME(t1) ;
inpmtx = DInpMtx_new() ;
DInpMtx_init(inpmtx, 1, 2, 0, 0) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %9.5f : initialize the DInpMtx object",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n DInpMtx after initialization") ;
   DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
}
for ( ielem = 0, jrow = 0 ; ielem < nelem ; ielem++ ) {
   IVL_listAndSize(egraph->adjIVL, ielem, &size, &indices) ;
   if ( n3 == 1 ) {
      for ( irow = 0 ; irow < 4 ; irow++, jrow++ ) {
         Drand_fillDvector(drand, size, entries) ;
         DInpMtx_inputRow(inpmtx, jrow, size, indices, entries) ;
      }
   } else {
      for ( irow = 0 ; irow < 8 ; irow++, jrow++ ) {
         Drand_fillDvector(drand, size, entries) ;
         DInpMtx_inputRow(inpmtx, jrow, size, indices, entries) ;
      }
   }
}
EGraph_free(egraph) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n natural factor matrix") ;
   if ( msglvl == 2 ) {
      DInpMtx_writeStats(inpmtx, msgFile) ;
   } else {
      DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   -----------------------------
   get the permutation IV object
   -----------------------------
*/
MARKTIME(t1) ;
oldToNewIV = ETree_oldToNewVtxPerm(frontETree) ;
newToOldIV = ETree_newToOldVtxPerm(frontETree) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : get permutation", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n vertex old-to-new IV object") ;
   if ( msglvl == 2 ) {
      IV_writeStats(oldToNewIV, msgFile) ;
   } else {
      IV_writeForHumanEye(oldToNewIV, msgFile) ;
   }
   fprintf(msgFile, "\n\n vertex new-to-old IV object") ;
   if ( msglvl == 2 ) {
      IV_writeStats(newToOldIV, msgFile) ;
   } else {
      IV_writeForHumanEye(newToOldIV, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   ----------------------------------
*/
DInpMtx_changeCoordType(inpmtx, 1) ;
DInpMtx_changeStorageMode(inpmtx, 3) ;
nrowA = 1 + IVmax(DInpMtx_nent(inpmtx), DInpMtx_ivec1(inpmtx), &rc) ;

X = DA2_new() ;
DA2_init(X, nvtx, nrhs, 1, nvtx, NULL) ;
Drand_fillDvector(drand, nvtx*nrhs, X->entries) ;
DVramp(nvtx*nrhs, X->entries, 1.0, 1.0 ) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n rhs matrices X ") ;
   DA2_writeForHumanEye(X, msgFile) ;
   fflush(msgFile) ;
}
Y = DA2_new() ;
DA2_init(Y, nrowA, nrhs, 1, nrowA, NULL) ;
DVzero(nrowA*nrhs, Y->entries) ;
xDV = DV_new() ;
yDV = DV_new() ;
MARKTIME(t1) ;
for ( jrhs = 0 ; jrhs < nrhs ; jrhs++ ) {
   DV_init(xDV,  nvtx, X->entries + jrhs*nvtx) ;
   DV_init(yDV, nrowA, Y->entries + jrhs*nrowA) ;
   DInpMtx_mvm(inpmtx, yDV, 1.0, xDV) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : DInpMtx mvm, %8.3f mflops",
        t2 - t1, 2.e-6*inpmtx->nent/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Y ") ;
   DA2_writeForHumanEye(Y, msgFile) ;
   fflush(msgFile) ;
}
Z = DA2_new() ;
DA2_init(Z, nvtx, nrhs, 1, nvtx, NULL) ;
DVzero(nvtx*nrhs, Z->entries) ;
/*
   --------------------------
   permute the DInpMtx object
   --------------------------
*/
MARKTIME(t1) ;
DInpMtx_permute(inpmtx,  NULL, IV_entries(oldToNewIV)) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute inpmtx ", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after permute DInpMtx object ") ;
   if ( msglvl == 2 ) {
      DInpMtx_writeStats(inpmtx, msgFile) ;
   } else {
      DInpMtx_writeForHumanEye(inpmtx, msgFile) ;
   }
}
/*
   --------------------------------------------
   create the symbolic factorization IVL object
   --------------------------------------------
*/
MARKTIME(t1) ;
symbfacIVL = SymbFac_initFromGraph(frontETree, graph) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : compute the symbolic factorization",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n symbolic factorization IVL object") ;
   if ( msglvl == 2 ) {
      IVL_writeStats(symbfacIVL, msgFile) ;
   } else {
      IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   }
   fflush(msgFile) ;
}
IVL_overwrite(symbfacIVL, oldToNewIV) ;
IVL_sortUp(symbfacIVL) ;
/*
   ----------------------------------------
   permute the vertices in the ETree object
   ----------------------------------------
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n before permuting the vertex map") ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
   fflush(msgFile) ;
}
MARKTIME(t1) ;
ETree_permuteVertices(frontETree, oldToNewIV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute ETree", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after permuting the vertex map") ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   -------------------------------
   initialize the DFrontMtx object
   -------------------------------
*/
fprintf(msgFile, "\n\n nfront = %d, nrows = %d, ncols = %d",
        nfront, nrowA, nvtx) ;
symmetryflag = 3 ;
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL,
               symmetryflag, 0, 0, 0, 0, NULL) ;
/*
   -----------------
   factor the matrix
   -----------------
*/
DVzero(9, cpus) ;
DInpMtx_changeCoordType(inpmtx, 1) ;
DInpMtx_changeStorageMode(inpmtx, 3) ;
MARKTIME(t1) ;
DFrontMtx_QRfactor(frontmtx, inpmtx, &factorops, cpus, msglvl, msgFile) ;
MARKTIME(t2) ;

fprintf(msgFile, "\n after QRfactor call facops = %8.2f",factorops) ;
fprintf(msgFile, "\n CPU %8.3f : DFrontMtx_QRfactor, %8.3f mflops",
        t2 - t1, 1.e-6*factorops/(t2-t1)) ;
cputotal = cpus[8] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile, "\n"
   "\n    initialize fronts       %8.3f %6.2f"
   "\n    load fronts             %8.3f %6.2f"
   "\n    assemble updates        %8.3f %6.2f"
   "\n    form staircase          %8.3f %6.2f"
   "\n    factor fronts           %8.3f %6.2f"
   "\n    store update            %8.3f %6.2f"
   "\n    store factor            %8.3f %6.2f"
   "\n    miscellaneous           %8.3f %6.2f"
   "\n    total time              %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal, 
   cpus[5], 100.*cpus[5]/cputotal, 
   cpus[6], 100.*cpus[6]/cputotal,
   cpus[7], 100.*cpus[7]/cputotal, cputotal) ;
}
/*
   ----------------------------
   set local indicies for solve
   ----------------------------
*/
DFrontMtx_postProcess(frontmtx, 0, 1, msglvl, msgFile) ;
/*
   -----------------------
   solve the linear system
   -----------------------
*/
nops = 2*(frontmtx->upperDVL->tsize
        + frontmtx->diagDVL->tsize
        + frontmtx->upperDVL->tsize) ;
MARKTIME(t1) ;
DFrontMtx_QRsolve(frontmtx, inpmtx, newToOldIV, Z, Y, cpus) ;
MARKTIME(t2) ;

fprintf(msgFile, "\n\n after QRsolve call") ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n solution Z ") ;
   DA2_writeForHumanEye(Z, msgFile) ;
   fflush(msgFile) ;
}

DA2_sub(Z, X) ;
fprintf(msgFile, 
     "\n CPU %8.3f : DFrontMtx_QRsolve, %8.3f mflops, %12.4e max error",
     t2 - t1, 1.e-6*nops/(t2 - t1), DA2_maxabs(Z)) ;

cputotal = cpus[7] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile, "\n"
   "\n    initialize rhs matrices      %8.3f %6.2f"
   "\n    load rhs matrices            %8.3f %6.2f"
   "\n    assemble children and parent %8.3f %6.2f"
   "\n    solve and update             %8.3f %6.2f"
   "\n    store entries                %8.3f %6.2f"
   "\n    link and free matrices       %8.3f %6.2f"
   "\n    miscellaneous                %8.3f %6.2f"
   "\n    total time                   %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal,
   cpus[5], 100.*cpus[5]/cputotal,
   cpus[6], 100.*cpus[6]/cputotal, cputotal) ;
}
/*
   ------------------------------
   unset local indicies for solve
   ------------------------------
*/
DFrontMtx_postProcess(frontmtx, 0, 2, msglvl, msgFile) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
ETree_free(frontETree) ;
DInpMtx_free(inpmtx) ;
DA2_free(X) ;
DA2_free(Y) ;
DA2_free(Z) ;
DV_free(xDV) ;
DV_free(yDV) ;
Graph_free(graph) ;
IV_free(oldToNewIV) ;
IV_free(newToOldIV) ;
IVL_free(symbfacIVL) ;
DFrontMtx_free(frontmtx) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
