/*  testQRFactor.c  */

#include "../DFrontMtx.h"
#include "../../SymbFac.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------
   test the factor method for a DFrontMtx object
   (1)  read in a Etree object
   (2)  get the oldToNew permutation from the ETree
   (3)  read in a DInpMtx object
   (4)  create a solution DA2 object X
   (5)  multiply the solution with the matrix
        to get a right hand side DA2 object Y
   (6)  permute the DInpMtx object
   (7)  read in A^TA Graph object
   (8)  get the symbolic factorization of A^TA and
        then permute the IVL
   (9)  permute the ETree object
   (10) factor the matrix 
   (11) solve the systems

   created -- 97apr11, dkw
   ---------------------------------------------------
*/
{
char          *inDInpMtxFile, *inETreeFile, *inGraphFile ;
DA2           *X, *Y, *Z ;
DFrontMtx     *frontmtx ;
DInpMtx       *inpmtxA ;
double        cputotal, factorops ;
double        cpus[9] ;
double        *dvec ;
Drand         drand ;
DV            *xDV, *yDV ;
double        nops, t1, t2   ;
ETree         *frontETree   ;
FILE          *msgFile ;
Graph         *graph ;
int           jrhs, msglvl, n, nfront, nrhs, nrowA,
              randomFlag, rc, seed, symmetryflag ;
IV            *oldToNewIV, *newToOldIV ;
IVL           *symbfacIVL ;

if ( argc != 9 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile inETreeFile inDInpMtxFile inGraphFile"
      "\n         seed randomFlag nrhs "
      "\n    msglvl           -- message level"
      "\n    msgFile          -- message file"
      "\n    inETreeFile      -- message file"
      "\n    inDInpMtxFile    -- message file"
      "\n    inGraphFile      -- input file, must be *.graphf or *.graphb"
      "\n    seed             -- random number seed"
      "\n    randomFlag -- random number flag"
      "\n       0 --> use matrix entries from file"
      "\n       1 --> use random matrix entries"
      "\n    nrhs             -- # of right hand sides"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "w")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inETreeFile   = argv[3] ;
inDInpMtxFile = argv[4] ;
inGraphFile   = argv[5] ;
seed          = atoi(argv[6]) ;
randomFlag    = atoi(argv[7]) ;
nrhs          = atoi(argv[8]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl           -- %d" 
        "\n msgFile          -- %s" 
        "\n inETreeFile      -- %s"
        "\n inDInpMtxFile    -- %s"
        "\n inGraphFile      -- %s"
        "\n seed             -- %d" 
        "\n randomFlag       -- %d" 
        "\n nrhs             -- %d" 
        "\n",
        argv[0], msglvl, argv[2], inETreeFile, inDInpMtxFile,
        inGraphFile, seed, randomFlag, nrhs) ;
fflush(msgFile) ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setNormal(&drand, 0.0, 1.0) ;
/*
   ------------------------
   read in the ETree object
   ------------------------
*/
if ( strcmp(inETreeFile, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
frontETree = ETree_new() ;
MARKTIME(t1) ;
rc = ETree_readFromFile(frontETree, inETreeFile) ;
MARKTIME(t2) ;
n      = frontETree->nvtx   ;
nfront = frontETree->nfront ;
fprintf(msgFile, "\n CPU %8.3f : read in frontETree from file %s",
        t2 - t1, inETreeFile) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from ETree_readFromFile(%p,%s)",
           rc, frontETree, inETreeFile) ;
   exit(-1) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after reading ETree object from file %s",
           inETreeFile) ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   -----------------------------
   get the permutation IV object
   -----------------------------
*/
MARKTIME(t1) ;
oldToNewIV = ETree_oldToNewVtxPerm(frontETree) ;
newToOldIV = ETree_newToOldVtxPerm(frontETree) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : get permutation", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n vertex old-to-new IV object") ;
   if ( msglvl == 2 ) {
      IV_writeStats(oldToNewIV, msgFile) ;
   } else {
      IV_writeForHumanEye(oldToNewIV, msgFile) ;
   }
   fprintf(msgFile, "\n\n vertex new-to-old IV object") ;
   if ( msglvl == 2 ) {
      IV_writeStats(newToOldIV, msgFile) ;
   } else {
      IV_writeForHumanEye(newToOldIV, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   --------------------------
   read in the DInpMtx object
   --------------------------
*/
if ( strcmp(inDInpMtxFile, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
inpmtxA = DInpMtx_new() ;
MARKTIME(t1) ;
rc = DInpMtx_readFromFile(inpmtxA, inDInpMtxFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : read in inpmtxA from file %s",
        t2 - t1, inDInpMtxFile) ;
if ( rc != 1 ) {
   fprintf(msgFile,
           "\n return value %d from DInpMtx_readFromFile(%p,%s)",
           rc, inpmtxA, inDInpMtxFile) ;
   exit(-1) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after reading DInpMtx object from file %s",
           inDInpMtxFile) ;
   if ( msglvl == 2 ) {
      DInpMtx_writeStats(inpmtxA, msgFile) ;
   } else {
      DInpMtx_writeForHumanEye(inpmtxA, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   -----------------------------------
   fill the matrix with random numbers
   -----------------------------------
*/
if ( randomFlag == 1 ) {
   MARKTIME(t1) ;
   inpmtxA->inputMode = 2 ;
   dvec = DVinit(inpmtxA->nent, 0.0) ;
   DV_init(&inpmtxA->dvecDV, inpmtxA->nent, dvec);
   Drand_fillDvector(&drand, inpmtxA->nent,
                     DV_entries(&inpmtxA->dvecDV)) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n CPU %8.3f : fill matrix with random numbers ",
           t2 - t1) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   ----------------------------------
*/
DInpMtx_changeCoordType(inpmtxA, 1) ;
DInpMtx_changeStorageMode(inpmtxA, 3) ;
nrowA = 1 + IVmax(DInpMtx_nent(inpmtxA), DInpMtx_ivec1(inpmtxA), &rc) ;

X = DA2_new() ;
DA2_init(X, n, nrhs, 1, n, NULL) ;
Drand_fillDvector(&drand, n*nrhs, X->entries) ;
DVramp(n*nrhs, X->entries, 1.0, 1.0 ) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n rhs matrices X ") ;
   DA2_writeForHumanEye(X, msgFile) ;
   fflush(msgFile) ;
}
Y = DA2_new() ;
DA2_init(Y, nrowA, nrhs, 1, nrowA, NULL) ;
DVzero(nrowA*nrhs, Y->entries) ;
xDV = DV_new() ;
yDV = DV_new() ;
MARKTIME(t1) ;
for ( jrhs = 0 ; jrhs < nrhs ; jrhs++ ) {
   DV_init(xDV,     n, X->entries + jrhs*n) ;
   DV_init(yDV, nrowA, Y->entries + jrhs*nrowA) ;
   DInpMtx_mvm(inpmtxA, yDV, 1.0, xDV) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : DInpMtx mvm, %8.3f mflops",
        t2 - t1, 2.e-6*inpmtxA->nent/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Y ") ;
   DA2_writeForHumanEye(Y, msgFile) ;
   fflush(msgFile) ;
}
Z = DA2_new() ;
DA2_init(Z, n, nrhs, 1, n, NULL) ;
DVzero(n*nrhs, Z->entries) ;
/*
   --------------------------
   permute the DInpMtx object
   --------------------------
*/
MARKTIME(t1) ;
DInpMtx_permute(inpmtxA,  NULL, IV_entries(oldToNewIV)) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute inpmtxA ", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after permute DInpMtx object from file %s",
           inDInpMtxFile) ;
   if ( msglvl == 2 ) {
      DInpMtx_writeStats(inpmtxA, msgFile) ;
   } else {
      DInpMtx_writeForHumanEye(inpmtxA, msgFile) ;
   }
}
/*
   -----------------------------
   read in the A^TA Graph object
   -----------------------------
*/
if ( strcmp(inGraphFile, "none") == 0 ) {
   fprintf(msgFile, "\n no file to read from") ;
   exit(0) ;
}
graph = Graph_new() ;
MARKTIME(t1) ;
rc = Graph_readFromFile(graph, inGraphFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : read in graph from file %s",
        t2 - t1, inGraphFile) ;
if ( rc != 1 ) {
   fprintf(msgFile, "\n return value %d from Graph_readFromFile(%p,%s)",
           rc, graph, inGraphFile) ;
   exit(-1) ;
}
rc = Graph_isSymmetric(graph) ;
   fprintf(msgFile, "\n return value %d from Graph_isSymmetric(%p,%s)",
           rc, graph, inGraphFile) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after reading Graph object from file %s",
           inGraphFile) ;
   if ( msglvl == 2 ) {
      Graph_writeStats(graph, msgFile) ;
   } else {
      Graph_writeForHumanEye(graph, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   create the symbolic factorization IVL object
   --------------------------------------------
*/
MARKTIME(t1) ;
symbfacIVL = SymbFac_initFromGraph(frontETree, graph) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : compute the symbolic factorization",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n symbolic factorization IVL object") ;
   if ( msglvl == 2 ) {
      IVL_writeStats(symbfacIVL, msgFile) ;
   } else {
      IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   }
   fflush(msgFile) ;
}
IVL_overwrite(symbfacIVL, oldToNewIV) ;
IVL_sortUp(symbfacIVL) ;
/*
   ----------------------------------------
   permute the vertices in the ETree object
   ----------------------------------------
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n before permuting the vertex map") ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
   fflush(msgFile) ;
}
MARKTIME(t1) ;
ETree_permuteVertices(frontETree, oldToNewIV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute ETree", t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n after permuting the vertex map") ;
   if ( msglvl == 2 ) {
      ETree_writeStats(frontETree, msgFile) ;
   } else {
      ETree_writeForHumanEye(frontETree, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   -------------------------------
   initialize the DFrontMtx object
   -------------------------------
*/
fprintf(msgFile, "\n\n nfront = %d, nrows = %d, ncols = %d",
        nfront, nrowA, n) ;
symmetryflag = 3 ;
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL,
               symmetryflag, 0, 0, 0, 0, NULL) ;
/*
   -----------------
   factor the matrix
   -----------------
*/
DVzero(9, cpus) ;
DInpMtx_changeCoordType(inpmtxA, 1) ;
DInpMtx_changeStorageMode(inpmtxA, 3) ;
MARKTIME(t1) ;
DFrontMtx_QRfactor(frontmtx, inpmtxA, &factorops, cpus, msglvl, msgFile) ;
MARKTIME(t2) ;

fprintf(msgFile, "\n after QRfactor call facops = %8.2f",factorops) ;
fprintf(msgFile, "\n CPU %8.3f : DFrontMtx_QRfactor, %8.3f mflops",
        t2 - t1, 1.e-6*factorops/(t2-t1)) ;
cputotal = cpus[8] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile, "\n"
   "\n    initialize fronts       %8.3f %6.2f"
   "\n    load fronts             %8.3f %6.2f"
   "\n    assemble updates        %8.3f %6.2f"
   "\n    form staircase          %8.3f %6.2f"
   "\n    factor fronts           %8.3f %6.2f"
   "\n    store update            %8.3f %6.2f"
   "\n    store factor            %8.3f %6.2f"
   "\n    miscellaneous           %8.3f %6.2f"
   "\n    total time              %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal, 
   cpus[5], 100.*cpus[5]/cputotal, 
   cpus[6], 100.*cpus[6]/cputotal,
   cpus[7], 100.*cpus[7]/cputotal, cputotal) ;
}
/*
   ----------------------------
   set local indicies for solve
   ----------------------------
*/
/*
DFrontMtx_setLocalIndices(frontmtx) ;
*/
DFrontMtx_postProcess(frontmtx, 0, 1, msglvl, msgFile) ;
/*
   -----------------------
   solve the linear system
   -----------------------
*/
nops = 2*(frontmtx->upperDVL->tsize
        + frontmtx->diagDVL->tsize
        + frontmtx->upperDVL->tsize) ;
MARKTIME(t1) ;
DFrontMtx_QRsolve(frontmtx, inpmtxA, newToOldIV, Z, Y, cpus) ;
MARKTIME(t2) ;

fprintf(msgFile, "\n\n after QRsolve call") ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n solution Z ") ;
   DA2_writeForHumanEye(Z, msgFile) ;
   fflush(msgFile) ;
}

DA2_sub(Z, X) ;
fprintf(msgFile, 
     "\n CPU %8.3f : DFrontMtx_QRsolve, %8.3f mflops, %12.4e max error",
     t2 - t1, 1.e-6*nops/(t2 - t1), DA2_maxabs(Z)) ;

cputotal = cpus[7] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile, "\n"
   "\n    initialize rhs matrices      %8.3f %6.2f"
   "\n    load rhs matrices            %8.3f %6.2f"
   "\n    assemble children and parent %8.3f %6.2f"
   "\n    solve and update             %8.3f %6.2f"
   "\n    store entries                %8.3f %6.2f"
   "\n    link and free matrices       %8.3f %6.2f"
   "\n    miscellaneous                %8.3f %6.2f"
   "\n    total time                   %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal,
   cpus[5], 100.*cpus[5]/cputotal,
   cpus[6], 100.*cpus[6]/cputotal, cputotal) ;
}
/*
   ------------------------------
   unset local indicies for solve
   ------------------------------
*/
/*
DFrontMtx_unsetLocalIndices(frontmtx) ;
*/
DFrontMtx_postProcess(frontmtx, 0, 2, msglvl, msgFile) ;
/*
   ------------------------
   free the working storage
   ------------------------
*/
if ( randomFlag == 1 ) {
   DVfree(dvec) ;
}
ETree_free(frontETree) ;
DInpMtx_free(inpmtxA) ;
DA2_free(X) ;
DA2_free(Y) ;
DA2_free(Z) ;
DV_free(xDV) ;
DV_free(yDV) ;
Graph_free(graph) ;
IV_free(oldToNewIV) ;
IV_free(newToOldIV) ;
IVL_free(symbfacIVL) ;
DFrontMtx_free(frontmtx) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
