/*  testGrid.c  */

#include "../DFrontMtx.h"
#include "../../DLinSystem.h"
#include "../../Drand.h"
#include "../../SymbFac.h"
#include "../../timings.h"
#include "../../misc.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------
   test the factor method for a grid matrix
   (1) construct a grid adjacency structure
   (2) get the nested dissection ordering
   (3) get the nested dissection elimination tree
   (4) transform into a front tree
   (3) get the oldToNew permutation from the ETree
   (4) permute the ETree object
   (5) permute the DInpMtx object
   (6) get the symbolic factorization
   (6) create a solution DA2 object
   (7) multiply the solution with the matrix
       to get a right hand side DA2 object
   (8) factor the matrix 
   (9) solve the systems

   created -- 97apr04, cca
   ---------------------------------------------------
*/
{
DA2                *mtxB, *mtxBkeep, *mtxX, *mtxXkeep ;
DChv               *chv, *rootchv ;
DDenseMtx          *Bkeep, *rhsmtx, *solmtx, *Xkeep ;
DDenseMtxManager   *manager ;
DFrontMtx          *frontmtx ;
DInpMtx            *inpmtxA ;
DLinSystem         *linsys ;
DPencil            *pencil ;
double             cputotal, droptol, factorops ;
double             cpus[9], ops[4] ;
double             *b, *dvec, *x ;
Drand              drand ;
double             nops, tau, t1, t2   ;
DV                 *bDV, *xDV ;
ETree              *etree, *frontETree   ;
FILE               *msgFile ;
Graph              *graph ;
int                ient, ii, irhs, lockflag, maxsize, maxzeros, 
                   msglvl, ncol, nent, neqns, n1, n2, n3, nrhs, 
                   nrow, nzf, pivotingflag, profile,
                   randomflag, seed, sparsityflag, symmetryflag,
                   v, vsize ;
int                *colind, *ivec1, *ivec2, *newToOld, *oldToNew, 
                   *rowind, *vadj ;
int                stats[6] ;
IV                 *oldToNewIV ;
IVL                *adjIVL, *symbfacIVL ;

if ( argc != 18 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile n1 n2 n3 maxzeros maxsize" 
"\n         seed symmetryflag sparsityflag "
"\n         pivotingflag tau droptol randomflag lockflag nrhs"
"\n    msglvl       -- message level"
"\n    msgFile      -- message file"
"\n    n1           -- number of grid points in the first direction"
"\n    n2           -- number of grid points in the second direction"
"\n    n3           -- number of grid points in the third direction"
"\n    maxzeros     -- max number of zeroes in a front"
"\n    maxsize      -- max number of internal nodes in a front"
"\n    seed         -- random number seed"
"\n    symmetryflag -- symmetry flag"
"\n       0 --> symmetric structure, symmetric entries"
"\n       1 --> symmetric structure, nonsymmetric entries"
"\n       2 --> nonsymmetric structure, nonsymmetric entries"
"\n       3 --> matrix from QR factorization"
"\n    sparsityflag -- sparsity flag"
"\n       0 --> store dense fronts"
"\n       1 --> store sparse fronts, use droptol to drop entries"
"\n    pivotingflag -- pivoting flag"
"\n       0 --> do not pivot"
"\n       1 --> enable pivoting"
"\n    tau     -- upper bound on factor entries"
"\n               used only with pivoting"
"\n    droptol -- lower bound on factor entries"
"\n               used only with sparse fronts"
"\n    randomflag -- random number flag"
"\n       0 --> use matrix entries from file"
"\n       1 --> use random matrix entries"
"\n    lockflag -- flag to specify lock status"
"\n       0 --> mutex lock is not allocated or initialized"
"\n       1 --> mutex lock is allocated and it can synchronize"
"\n             only threads in this process."
"\n       2 --> mutex lock is allocated and it can synchronize"
"\n             only threads in this and other processes."
"\n    nrhs     -- # of right hand sides"
"\n    profile  -- profile flag"
"\n       0 --> do not display a profile of the upper entries"
"\n       1 --> display a profile of the upper entries"
"\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
n1            = atoi(argv[3]) ;
n2            = atoi(argv[4]) ;
n3            = atoi(argv[5]) ;
maxzeros      = atoi(argv[6]) ;
maxsize       = atoi(argv[7]) ;
seed          = atoi(argv[8]) ;
symmetryflag  = atoi(argv[9]) ;
sparsityflag  = atoi(argv[10]) ;
pivotingflag  = atoi(argv[11]) ;
tau           = atof(argv[12]) ;
droptol       = atof(argv[13]) ;
randomflag    = atoi(argv[14]) ;
lockflag      = atoi(argv[15]) ;
nrhs          = atoi(argv[16]) ;
profile       = atoi(argv[17]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl        -- %d" 
        "\n msgFile       -- %s" 
        "\n n1            -- %d" 
        "\n n2            -- %d" 
        "\n n3            -- %d" 
        "\n maxzeros      -- %d" 
        "\n maxsize       -- %d" 
        "\n seed          -- %d" 
        "\n symmetryflag  -- %d" 
        "\n sparsityflag  -- %d" 
        "\n pivotingflag  -- %d" 
        "\n tau           -- %e" 
        "\n droptol       -- %e" 
        "\n randomflag    -- %d" 
        "\n lockflag      -- %d" 
        "\n nrhs          -- %d" 
        "\n profile       -- %d" 
        "\n",
        argv[0], msglvl, argv[2], n1, n2, n3, maxzeros, maxsize,
        seed, symmetryflag, sparsityflag, pivotingflag, 
        tau, droptol, randomflag, lockflag, nrhs, profile) ;
fflush(msgFile) ;
neqns = n1 * n2 * n3 ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
/*
Drand_setUniform(&drand, 0.0, 1.0) ;
*/
Drand_setNormal(&drand, 0.0, 1.0) ;
/*
   --------------------------------
   get the grid adjacency structure
   --------------------------------
*/
MARKTIME(t1) ;
if ( n3 == 1 ) {
   adjIVL = IVL_make9P(n1, n2, 1) ;
} else {
   adjIVL = IVL_make27P(n1, n2, n3, 1) ;
}
graph = Graph_new() ;
Graph_init2(graph, 0, neqns, 0, adjIVL->tsize, neqns,
            adjIVL->tsize, adjIVL, NULL, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : create the grid graph",
        t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n grid graph") ;
   Graph_writeForHumanEye(graph, msgFile) ;
}
/*
   ---------------------------------------------
   make the nested dissection permutation vector
   ---------------------------------------------
*/
MARKTIME(t1) ;
newToOld = IVinit(neqns, -1) ;
oldToNew = IVinit(neqns, -1) ;
mkNDperm(n1, n2, n3, newToOld, 0, n1-1, 0, n2-1, 0, n3-1) ;
for ( ii = 0 ; ii < neqns ; ii++ ) {
   oldToNew[newToOld[ii]] = ii ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : make the nested dissection ordering",
        t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n oldToNew") ;
   IVfprintf(msgFile, neqns, oldToNew) ;
}
/*
   ----------------------------------
   create the elimination tree object
   ----------------------------------
*/
MARKTIME(t1) ;
etree = ETree_new() ;
ETree_initFromGraphWithPerms(etree, graph, newToOld, oldToNew) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n elimination tree") ;
   ETree_writeForHumanEye(etree, msgFile) ;
}
frontETree = ETree_transform(etree, NULL, maxzeros, maxsize, seed) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n front tree") ;
   ETree_writeForHumanEye(frontETree, msgFile) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : create the front tree",
        t2 - t1) ;
ETree_nFactorOps2(frontETree, ops) ;
fprintf(msgFile, 
       "\n %.2f per cent intra-front ops, %.2f per cent interfront ops",
100.*(ops[0] + ops[1] + ops[2])/(ops[0] + ops[1] + ops[2] + ops[3]),
 100.*(ops[3])/(ops[0] + ops[1] + ops[2] + ops[3])) ;
/*
{
int      loc, nfront = frontETree->nfront ;
double   *dvec = DVinit(nfront, 0.0) ;

ETree_forwSolveProfile(frontETree, dvec) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n forward solve stack") ;
   DVfprintf(msgFile, nfront, dvec) ;
}
fprintf(msgFile, "\n max forward solve stack = %.0f", 
        DVmax(nfront, dvec, &loc)) ;
ETree_backSolveProfile(frontETree, dvec) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n backward solve stack") ;
   DVfprintf(msgFile, nfront, dvec) ;
}
fprintf(msgFile, "\n max backward solve stack = %.0f", 
        DVmax(nfront, dvec, &loc)) ;
}
*/
/*
   --------------------------------------
   permute the vertices in the front tree
   --------------------------------------
*/
MARKTIME(t1) ;
oldToNewIV = ETree_oldToNewVtxPerm(frontETree) ;
ETree_permuteVertices(frontETree, oldToNewIV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute the front tree",
        t2 - t1) ;
/*
   -------------------------
   set up the DInpMtx object
   -------------------------
*/
MARKTIME(t1) ;
inpmtxA = DInpMtx_new() ;
nent = adjIVL->tsize ;
DInpMtx_init(inpmtxA, 1, 2, nent, 0) ;
ivec1 = DInpMtx_ivec1(inpmtxA) ;
ivec2 = DInpMtx_ivec2(inpmtxA) ;
dvec  = DInpMtx_dvec(inpmtxA) ;
for ( v = 0, ient = 0 ; v < neqns ; v++ ) {
   IVL_listAndSize(adjIVL, v, &vsize, &vadj) ;
   for ( ii = 0 ; ii < vsize ; ii++, ient++ ) {
      ivec1[ient] = v ;
      ivec2[ient] = vadj[ii] ;
   }
}
Drand_fillDvector(&drand, nent, dvec) ;
DInpMtx_setNent(inpmtxA, nent) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n\n matrix object") ;
   DInpMtx_writeForHumanEye(inpmtxA, msgFile) ;
}
if ( symmetryflag == 0 ) {
/*
   ----------------------------------------------------
   symmetric matrix, drop entries in the lower triangle
   ----------------------------------------------------
*/
   DInpMtx_dropLowerTriangle(inpmtxA) ;
   DInpMtx_changeStorageMode(inpmtxA, 1) ;
   DInpMtx_sortAndCompress(inpmtxA) ;
}
DInpMtx_changeCoordType(inpmtxA, 3) ;
DInpMtx_changeStorageMode(inpmtxA, 2) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set up the DInpMtxA object",
        t2 - t1) ;
/*
   -------------------
   generate the pencil
   -------------------
*/
MARKTIME(t1) ;
pencil = DPencil_new() ;
DPencil_init(pencil, inpmtxA, 0.0, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set up the DPencil object",
        t2 - t1) ;
/*
   --------------------------
   generate the linear system
   --------------------------
*/
MARKTIME(t1) ;
rhsmtx = DDenseMtx_new() ;
solmtx = DDenseMtx_new() ;
DDenseMtx_init(solmtx, 0, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_rowIndices(solmtx, &nrow, &rowind) ;
IVramp(nrow, rowind, 0, 1) ;
DDenseMtx_columnIndices(solmtx, &ncol, &colind) ;
IVramp(ncol, colind, 0, 1) ;
DDenseMtx_init(rhsmtx, 1, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_rowIndices(rhsmtx, &nrow, &rowind) ;
IVramp(nrow, rowind, 0, 1) ;
DDenseMtx_columnIndices(rhsmtx, &ncol, &colind) ;
IVramp(ncol, colind, 0, 1) ;
DVfill(neqns*nrhs, DDenseMtx_entries(solmtx), 1.0) ;
Drand_fillDvector(&drand, neqns*nrhs, DDenseMtx_entries(solmtx)) ;
DVfill(neqns*nrhs, DDenseMtx_entries(rhsmtx), 0.0) ;
bDV = DV_new() ;
xDV = DV_new() ;
x   = DDenseMtx_entries(solmtx) ;
b   = DDenseMtx_entries(rhsmtx) ;
for ( irhs = 0 ; irhs < nrhs ; irhs++ ) {
   DV_init(bDV, neqns, b) ;
   DV_init(xDV, neqns, x) ;
   if ( symmetryflag == 0 ) {
      DPencil_symmetricMVM(pencil, bDV, xDV) ;
   } else {
      DPencil_mvm(pencil, bDV, xDV) ;
   }
   x += neqns ;
   b += neqns ;
}
DV_free(bDV) ;
DV_free(xDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : set up the solution and rhs ",
        t2 - t1) ;
/*
   ------------------------------------
   initialize the linear systems object
   ------------------------------------
*/
MARKTIME(t1) ;
linsys = DLinSystem_new() ;
DLinSystem_init(linsys, neqns, nrhs, symmetryflag,
                pencil, solmtx, rhsmtx) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : initialize the linear system",
        t2 - t1) ;
/*
   -------------------------
   permute the linear system
   -------------------------
*/
MARKTIME(t1) ;
DLinSystem_permute(linsys, oldToNewIV, msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : permute the linear system",
        t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n permuted matrix pencil") ;
   DPencil_writeForHumanEye(linsys->pencil, msgFile) ;
   fprintf(msgFile, "\n\n permuted solution") ;
   DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
   fprintf(msgFile, "\n\n permuted right hand side") ;
   DDenseMtx_writeForHumanEye(linsys->rhsmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   create the symbolic factorization IVL object
   --------------------------------------------
*/
MARKTIME(t1) ;
symbfacIVL = SymbFac_initFromDPencil(frontETree, linsys->pencil) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : compute the symbolic factorization",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n symbolic factorization IVL object") ;
   if ( msglvl == 2 ) {
      IVL_writeStats(symbfacIVL, msgFile) ;
   } else {
      IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   ---------------------------------------
   convert the DPencil storage to chevrons
   ---------------------------------------
*/
MARKTIME(t1) ;
DPencil_changeCoordType(linsys->pencil, 3) ;
DPencil_changeStorageMode(linsys->pencil, 3) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : convert to chevron vectors ",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n DPencil object ") ;
   if ( msglvl == 2 ) {
      DPencil_writeStats(linsys->pencil, msgFile) ;
   } else if ( msglvl > 3 ) {
      DPencil_writeForHumanEye(linsys->pencil, msgFile) ;
   }
}
/*
   -------------------------------
   initialize the DFrontMtx object
   -------------------------------
*/
MARKTIME(t1) ;
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL,
               symmetryflag, sparsityflag, pivotingflag,
               lockflag, 0, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : initialize the front matrix",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile,
        "\n diagDVL->tsize  = %d"
        "\n lowerDVL->tsize = %d"
        "\n upperDVL->tsize = %d",
        (frontmtx->diagDVL  != NULL) ? frontmtx->diagDVL->tsize  : 0,
        (frontmtx->lowerDVL != NULL) ? frontmtx->lowerDVL->tsize : 0,
        (frontmtx->upperDVL != NULL) ? frontmtx->upperDVL->tsize : 0) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n front matrix initialized") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------
   factor the matrix
   -----------------
*/
DVzero(9, cpus) ;
MARKTIME(t1) ;
rootchv = DFrontMtx_factor(frontmtx, linsys->pencil, tau, droptol, 
                           cpus, stats, msglvl, msgFile) ;
MARKTIME(t2) ;
if ( rootchv != NULL ) {
   fprintf(msgFile, "\n\n factorization did not complete") ;
   for ( chv = rootchv ; chv != NULL ; chv = chv->next ) {
      fprintf(stdout, "\n chv %d, nD = %d, nL = %d, nU = %d",
              chv->id, chv->nD, chv->nL, chv->nU) ;
/*
      DChv_writeForHumanEye(chv, msgFile) ;
*/
   }
}
if ( symmetryflag == 0 ) {
   nzf       = ETree_nFactorEntries(frontETree, 1) ;
   factorops = ETree_nFactorOps(frontETree, 1) ;
} else {
   nzf       = ETree_nFactorEntries(frontETree, 2) ;
   factorops = ETree_nFactorOps(frontETree, 2) ;
}
fprintf(msgFile, 
        "\n %d factor entries, %.0f factor ops, %8.3f ratio",
        nzf, factorops, factorops/nzf) ;
fprintf(msgFile, "\n CPU %8.3f : factor matrix, %8.3f mflops",
        t2 - t1, 1.e-6*factorops/(t2-t1)) ;
fprintf(msgFile,
        "\n %8d pivots, %8d pivot tests, %8d delayed rows and columns",
        stats[0], stats[1], stats[2]) ;
if ( frontmtx->rowadjIVL != NULL ) {
   fprintf(msgFile,
           "\n %d entries in rowadjIVL", frontmtx->rowadjIVL->tsize) ;
}
if ( frontmtx->coladjIVL != NULL ) {
   fprintf(msgFile,
           ", %d entries in coladjIVL", frontmtx->coladjIVL->tsize) ;
}
fprintf(msgFile,
        "\n %d entries in D, %d entries in L, %d entries in U",
        stats[3], stats[4], stats[5]) ;
fprintf(msgFile, "\n %d locks", frontmtx->nlocks) ;
cputotal = cpus[8] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile,
   "\n    initialize fronts       %8.3f %6.2f"
   "\n    load original entries   %8.3f %6.2f"
   "\n    update fronts           %8.3f %6.2f"
   "\n    assemble postponed data %8.3f %6.2f"
   "\n    factor fronts           %8.3f %6.2f"
   "\n    extract postponed data  %8.3f %6.2f"
   "\n    store factor entries    %8.3f %6.2f"
   "\n    miscellaneous           %8.3f %6.2f"
   "\n    total time              %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal,
   cpus[5], 100.*cpus[5]/cputotal,
   cpus[6], 100.*cpus[6]/cputotal,
   cpus[7], 100.*cpus[7]/cputotal, cputotal) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n front factor matrix") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
}
if ( profile != 0 ) {
   double   tausmall, taubig ;
   double   *x, *y ;
   DV       *xDV, *yDV ;
   int      ii, nbig, npts, nsmall, ntotal ;
   
   tausmall = 1.e-20 ;
   taubig   = 1.e+20 ;
   xDV = DV_new() ;
   yDV = DV_new() ;
   DVL_log10profile(frontmtx->upperDVL, 100, xDV, yDV, tausmall, taubig,
               &nsmall, &nbig, &ntotal) ;
   fprintf(msgFile, "\n\n %d small entries, %d big entries, %d total",
           nsmall, nbig, ntotal) ;
   fprintf(msgFile, "\n profile of U") ;
   DV_sizeAndEntries(xDV, &npts, &x) ;
   DV_sizeAndEntries(yDV, &npts, &y) ;
   for ( ii = 0 ; ii < npts ; ii++ ) {
      fprintf(msgFile, "\n %20.12e %20.12e", x[ii], y[ii]) ;
   }
}
/*
   ------------------------
   convert to local indices
   ------------------------
*/
/*
DFrontMtx_setLocalIndices(frontmtx) ;
*/
DFrontMtx_postProcess(frontmtx, 0, 1, msglvl, msgFile) ;
if ( msglvl > 2 ) {
   if ( pivotingflag == 0 ) {
      fprintf(msgFile, "\n local symbfacIVL") ;
      IVL_writeForHumanEye(frontmtx->symbfacIVL, msgFile) ;
   } else {
      if ( symmetryflag == 2 ) {
         fprintf(msgFile, "\n local rowadjIVL") ;
         IVL_writeForHumanEye(frontmtx->rowadjIVL, msgFile) ;
      }
      fprintf(msgFile, "\n local coladjIVL") ;
      IVL_writeForHumanEye(frontmtx->coladjIVL, msgFile) ;
   }
}
/*
   --------------------------------------------------------------
   solve the linear system via multifrontal solve on multiple rhs
   --------------------------------------------------------------
*/
if ( symmetryflag == 0 ) {
   nops = 2*(frontmtx->upperDVL->tsize
           + frontmtx->diagDVL->tsize
           + frontmtx->upperDVL->tsize) ;
} else {
   nops = 2*(frontmtx->lowerDVL->tsize
           + frontmtx->diagDVL->tsize
           + frontmtx->upperDVL->tsize) ;
}
Bkeep    = DDenseMtx_new() ;
mtxB     = DA2_new() ;
mtxBkeep = DA2_new() ;
DDenseMtx_init(Bkeep, 2, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_setDA2(linsys->rhsmtx, mtxB) ;
DDenseMtx_setDA2(Bkeep, mtxBkeep) ;
DA2_copy(mtxBkeep, mtxB) ;

Xkeep    = DDenseMtx_new() ;
mtxX     = DA2_new() ;
mtxXkeep = DA2_new() ;
DDenseMtx_init(Xkeep, 2, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_setDA2(linsys->solmtx, mtxX) ;
DDenseMtx_setDA2(Xkeep, mtxXkeep) ;
DA2_copy(mtxXkeep, mtxX) ;

MARKTIME(t1) ;
manager = DDenseMtxManager_new() ;
DDenseMtxManager_init(manager, 0) ;
DFrontMtx_solveDA2(frontmtx, mtxX, mtxB, manager, cpus) ;
MARKTIME(t2) ;
DDenseMtxManager_writeForHumanEye(manager, msgFile) ;
DDenseMtxManager_free(manager) ;
DA2_sub(mtxX, mtxXkeep) ;
fprintf(msgFile,
     "\n CPU %8.3f : DFrontMtx solve, %8.3f mflops, %12.4e max error",
     t2 - t1, 1.e-6*nops*nrhs/(t2 - t1), DA2_maxabs(mtxX)) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n error ") ;
   DA2_writeForHumanEye(mtxX, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   convert from local indices to global indices
   --------------------------------------------
*/
/*
DFrontMtx_unsetLocalIndices(frontmtx) ;
*/
DFrontMtx_postProcess(frontmtx, 0, 2, msglvl, msgFile) ;
if ( msglvl > 2 ) {
   if ( pivotingflag == 0 ) {
      fprintf(msgFile, "\n re-set symbfacIVL") ;
      IVL_writeForHumanEye(frontmtx->symbfacIVL, msgFile) ;
   } else {
      if ( symmetryflag == 2 ) {
         fprintf(msgFile, "\n re-set rowadjIVL") ;
         IVL_writeForHumanEye(frontmtx->rowadjIVL, msgFile) ;
      }
      fprintf(msgFile, "\n re-set coladjIVL") ;
      IVL_writeForHumanEye(frontmtx->coladjIVL, msgFile) ;
   }
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
ETree_free(frontETree) ;
DLinSystem_free(linsys) ;
DDenseMtx_free(Xkeep) ;
DDenseMtx_free(Bkeep) ;
DA2_free(mtxB) ;
DA2_free(mtxBkeep) ;
DA2_free(mtxX) ;
DA2_free(mtxXkeep) ;
IV_free(oldToNewIV) ;
IVL_free(symbfacIVL) ;
DFrontMtx_free(frontmtx) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
