/*  testFactor.c  */

#include "../DFrontMtx.h"
#include "../../SymbFac.h"
#include "../../DLinSystem.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------------------
   test the factor method for a DFrontMtx object
   (1) read in a DInpMtx object
   (2) read in a Etree object
   (3) get the oldToNew permutation from the ETree
   (4) permute the ETree object
   (5) permute the DInpMtx object
   (6) get the symbolic factorization
   (6) create a solution DA2 object
   (7) multiply the solution with the matrix
       to get a right hand side DA2 object
   (8) factor the matrix 
   (9) solve the systems

   created -- 97apr04, cca
   ---------------------------------------------------
*/
{
char               *inETreeFile, *inpmtxAfile, *inpmtxBfile ;
DA2                *mtxB, *mtxBkeep, *mtxX, *mtxXkeep ;
DChv               *chv, *rootchv ;
DDenseMtx          *Bkeep, *Xkeep ;
DDenseMtxManager   *manager ;
DFrontMtx          *frontmtx ;
DLinSystem         *linsys ;
double             cputotal, droptol, factorops, sigma ;
double             cpus[9] ;
Drand              drand ;
double             nops, tau, t1, t2   ;
ETree              *frontETree   ;
FILE               *msgFile ;
int                lockflag, msglvl, neqns, nrhs, pivotingflag, profile,
                   randomflag, seed, sparsityflag, symmetryflag ;
int                stats[6] ;
IV                 *oldToNewIV ;
IVL                *symbfacIVL ;

if ( argc != 17 ) {
   fprintf(stdout, 
      "\n\n usage : %s msglvl msgFile inETreeFile inpmtxAfile"
      "\n         sigma inpmtxBfile seed symmetryflag sparsityflag "
      "\n         pivotingflag tau droptol randomflag "
      "\n    msglvl       -- message level"
      "\n    msgFile      -- message file"
      "\n    inETreeFile  -- front ETree file"
      "\n    inpmtxAfile  -- file that contains A"
      "\n    sigma        -- shift value"
      "\n    inpmtxBfile  -- file that contains B"
      "\n    seed         -- random number seed"
      "\n    symmetryflag -- symmetry flag"
      "\n       0 --> symmetric structure, symmetric entries"
      "\n       1 --> symmetric structure, nonsymmetric entries"
      "\n       2 --> nonsymmetric structure, nonsymmetric entries"
      "\n       3 --> matrix from QR factorization"
      "\n    sparsityflag -- sparsity flag"
      "\n       0 --> store dense fronts"
      "\n       1 --> store sparse fronts, use droptol to drop entries"
      "\n    pivotingflag -- pivoting flag"
      "\n       0 --> do not pivot"
      "\n       1 --> enable pivoting"
      "\n    tau     -- upper bound on factor entries"
      "\n               used only with pivoting"
      "\n    droptol -- lower bound on factor entries"
      "\n               used only with sparse fronts"
      "\n    randomflag -- random number flag"
      "\n       0 --> use matrix entries from file"
      "\n       1 --> use random matrix entries"
      "\n    lockflag -- flag to specify lock status"
      "\n       0 --> mutex lock is not allocated or initialized"
      "\n       1 --> mutex lock is allocated and it can synchronize"
      "\n             only threads in this process."
      "\n       2 --> mutex lock is allocated and it can synchronize"
      "\n             only threads in this and other processes."
      "\n    nrhs     -- # of right hand sides"
      "\n    profile  -- profile flag"
      "\n       0 --> do not display a profile of the upper entries"
      "\n       1 --> display a profile of the upper entries"
      "\n", argv[0]) ;
   return ;
}
msglvl = atoi(argv[1]) ;
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n fatal error in %s"
           "\n unable to open file %s\n",
           argv[0], argv[2]) ;
   return ;
}
inETreeFile   = argv[3] ;
inpmtxAfile   = argv[4] ;
sigma         = atof(argv[5]) ;
inpmtxBfile   = argv[6] ;
seed          = atoi(argv[7]) ;
symmetryflag  = atoi(argv[8]) ;
sparsityflag  = atoi(argv[9]) ;
pivotingflag  = atoi(argv[10]) ;
tau           = atof(argv[11]) ;
droptol       = atof(argv[12]) ;
randomflag    = atoi(argv[13]) ;
lockflag      = atoi(argv[14]) ;
nrhs          = atoi(argv[15]) ;
profile       = atoi(argv[16]) ;
fprintf(msgFile, 
        "\n %s "
        "\n msglvl        -- %d" 
        "\n msgFile       -- %s" 
        "\n inETreeFile   -- %s"
        "\n inpmtxAfile   -- %s"
        "\n sigma         -- %f"
        "\n inpmtxBfile   -- %s"
        "\n seed          -- %d" 
        "\n symmetryflag  -- %d" 
        "\n sparsityflag  -- %d" 
        "\n pivotingflag  -- %d" 
        "\n tau           -- %e" 
        "\n droptol       -- %e" 
        "\n randomflag    -- %d" 
        "\n lockflag      -- %d" 
        "\n nrhs          -- %d" 
        "\n profile       -- %d" 
        "\n",
        argv[0], msglvl, argv[2], inETreeFile, inpmtxAfile, sigma, 
        inpmtxBfile, seed, symmetryflag, sparsityflag, pivotingflag, 
        tau, droptol, randomflag, lockflag, nrhs, profile) ;
fflush(msgFile) ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
/*
Drand_setUniform(&drand, 0.0, 1.0) ;
*/
Drand_setNormal(&drand, 0.0, 1.0) ;
/*
   -------------------------------------------
   initialize the ETree and oldToNewIV objects
   -------------------------------------------
*/
frontETree = ETree_new() ;
oldToNewIV = ETree_initFromFile(frontETree, inETreeFile, 
                                msglvl, msgFile) ;
neqns = IV_size(oldToNewIV) ;
/*
ETree_writeForHumanEye(frontETree, msgFile) ;
*/
/*
   ------------------------
   set up the linear system
   ------------------------
*/
linsys = DLinSystem_new() ;
Drand_setSeed(&drand, seed) ;
DLinSystem_setup(linsys, 0, neqns, nrhs, inpmtxAfile, sigma,
                 inpmtxBfile, symmetryflag, randomflag, &drand,
                 msglvl, msgFile) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n matrix pencil") ;
   DPencil_writeForHumanEye(linsys->pencil, msgFile) ;
   fprintf(msgFile, "\n\n solution") ;
   DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
   fprintf(msgFile, "\n\n right hand side") ;
   DDenseMtx_writeForHumanEye(linsys->rhsmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   -------------------------
   permute the linear system
   -------------------------
*/
DLinSystem_permute(linsys, oldToNewIV, msglvl, msgFile) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n permuted matrix pencil") ;
   DPencil_writeForHumanEye(linsys->pencil, msgFile) ;
   fprintf(msgFile, "\n\n permuted solution") ;
   DDenseMtx_writeForHumanEye(linsys->solmtx, msgFile) ;
   fprintf(msgFile, "\n\n permuted right hand side") ;
   DDenseMtx_writeForHumanEye(linsys->rhsmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   --------------------------------------------
   create the symbolic factorization IVL object
   --------------------------------------------
*/
MARKTIME(t1) ;
symbfacIVL = SymbFac_initFromDPencil(frontETree, linsys->pencil) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : compute the symbolic factorization",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n symbolic factorization IVL object") ;
   if ( msglvl == 2 ) {
      IVL_writeStats(symbfacIVL, msgFile) ;
   } else {
      IVL_writeForHumanEye(symbfacIVL, msgFile) ;
   }
   fflush(msgFile) ;
}
/*
   ---------------------------------------
   convert the DPencil storage to chevrons
   ---------------------------------------
*/
MARKTIME(t1) ;
DPencil_changeCoordType(linsys->pencil, 3) ;
DPencil_changeStorageMode(linsys->pencil, 3) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : convert to chevron vectors ",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n DPencil object ") ;
   if ( msglvl == 2 ) {
      DPencil_writeStats(linsys->pencil, msgFile) ;
   } else if ( msglvl > 3 ) {
      DPencil_writeForHumanEye(linsys->pencil, msgFile) ;
   }
}
/*
   -------------------------------
   initialize the DFrontMtx object
   -------------------------------
*/
MARKTIME(t1) ;
frontmtx = DFrontMtx_new() ;
DFrontMtx_init(frontmtx, frontETree, symbfacIVL,
               symmetryflag, sparsityflag, pivotingflag,
               lockflag, 0, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU %8.3f : initialize the front matrix",
        t2 - t1) ;
if ( msglvl > 1 ) {
   fprintf(msgFile,
        "\n diagDVL->tsize  = %d"
        "\n lowerDVL->tsize = %d"
        "\n upperDVL->tsize = %d",
        (frontmtx->diagDVL  != NULL) ? frontmtx->diagDVL->tsize  : 0,
        (frontmtx->lowerDVL != NULL) ? frontmtx->lowerDVL->tsize : 0,
        (frontmtx->upperDVL != NULL) ? frontmtx->upperDVL->tsize : 0) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n front matrix initialized") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
   fflush(msgFile) ;
}
/*
   -----------------
   factor the matrix
   -----------------
*/
DVzero(9, cpus) ;
MARKTIME(t1) ;
rootchv = DFrontMtx_factor(frontmtx, linsys->pencil, tau, droptol, 
                           cpus, stats, msglvl, msgFile) ;
MARKTIME(t2) ;
if ( rootchv != NULL ) {
   fprintf(msgFile, "\n\n factorization did not complete") ;
   for ( chv = rootchv ; chv != NULL ; chv = chv->next ) {
      fprintf(stdout, "\n chv %d, nD = %d, nL = %d, nU = %d",
              chv->id, chv->nD, chv->nL, chv->nU) ;
/*
      DChv_writeForHumanEye(chv, msgFile) ;
*/
   }
}
/*
fprintf(msgFile, "\n\n diag DVL") ;
DVL_writeForHumanEye(frontmtx->diagDVL, msgFile) ;
*/
/*
{
double   *diagent ;
int      nent ;

DFrontMtx_diagEntries(frontmtx, frontmtx->nfront - 1,
                      &nent, &diagent) ;
fprintf(msgFile, "\n\n diagonal entries for the top front") ;
DVfprintf(msgFile, nent, diagent) ;
}
*/
if ( symmetryflag == 0 ) {
   factorops = ETree_nFactorOps(frontETree, 1) ;
} else {
   factorops = ETree_nFactorOps(frontETree, 2) ;
}
fprintf(msgFile, "\n\n CPU %8.3f : factor matrix, %8.3f mflops",
        t2 - t1, 1.e-6*factorops/(t2-t1)) ;
fprintf(msgFile,
        "\n %8d pivots, %8d pivot tests, %8d delayed rows and columns",
        stats[0], stats[1], stats[2]) ;
if ( frontmtx->rowadjIVL != NULL ) {
   fprintf(msgFile,
           "\n %d entries in rowadjIVL", frontmtx->rowadjIVL->tsize) ;
}
if ( frontmtx->coladjIVL != NULL ) {
   fprintf(msgFile,
           ", %d entries in coladjIVL", frontmtx->coladjIVL->tsize) ;
}
fprintf(msgFile,
        "\n %d entries in D, %d entries in L, %d entries in U",
        stats[3], stats[4], stats[5]) ;
fprintf(msgFile, "\n %d locks", frontmtx->nlocks) ;
cputotal = cpus[8] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile,
   "\n    initialize fronts       %8.3f %6.2f"
   "\n    load original entries   %8.3f %6.2f"
   "\n    update fronts           %8.3f %6.2f"
   "\n    assemble postponed data %8.3f %6.2f"
   "\n    factor fronts           %8.3f %6.2f"
   "\n    extract postponed data  %8.3f %6.2f"
   "\n    store factor entries    %8.3f %6.2f"
   "\n    miscellaneous           %8.3f %6.2f"
   "\n    total time              %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal,
   cpus[5], 100.*cpus[5]/cputotal,
   cpus[6], 100.*cpus[6]/cputotal,
   cpus[7], 100.*cpus[7]/cputotal, cputotal) ;
}
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n\n front factor matrix") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
}
if ( profile != 0 ) {
   double   tausmall, taubig ;
   double   *x, *y ;
   DV       *xDV, *yDV ;
   int      ii, nbig, npts, nsmall, ntotal ;
   
   tausmall = 1.e-20 ;
   taubig   = 1.e+20 ;
   xDV = DV_new() ;
   yDV = DV_new() ;
   DVL_log10profile(frontmtx->upperDVL, 100, xDV, yDV, tausmall, taubig,
               &nsmall, &nbig, &ntotal) ;
   fprintf(msgFile, "\n\n %d small entries, %d big entries, %d total",
           nsmall, nbig, ntotal) ;
   fprintf(msgFile, "\n profile of U") ;
   DV_sizeAndEntries(xDV, &npts, &x) ;
   DV_sizeAndEntries(yDV, &npts, &y) ;
   for ( ii = 0 ; ii < npts ; ii++ ) {
      fprintf(msgFile, "\n %20.12e %20.12e", x[ii], y[ii]) ;
   }
/*
   { double   *entries, *maxvalues ;
     double   maxval, val ;
     int      ilast, irow, J, jcol, k, ncolJ, nJ, size ;
     int      *indJ ;

      J = frontmtx->nfront - 2 ;
      J = 1483 ;
      DVL_listAndSize(frontmtx->upperDVL, J, &size, &entries) ;
      nJ = DFrontMtx_frontSize(frontmtx, J) ;
      DFrontMtx_columnIndices(frontmtx, J, &ncolJ, &indJ) ;
      fprintf(msgFile, "\n J = %d, nJ = %d, ncolJ = %d, size = %d", 
              J, nJ, ncolJ, size) ;
      fprintf(msgFile, "\n upper entries for second level separator") ;
      DVfprintf(msgFile, size, entries) ;
      maxvalues = DVinit(ncolJ, 0.0) ;
      for ( jcol = 1, k = 0 ; jcol < ncolJ ; jcol++ ) {
         ilast = (jcol < nJ) ? jcol : nJ ;
         for ( irow = 0, maxval = 0.0 ; irow < ilast ; irow++ ) {
            if ( (val = fabs(entries[k++])) > maxval ) {
               maxval = val ;
            }
         }
         maxvalues[jcol] = maxval ;
      }
      DVqsortUp(ncolJ, maxvalues) ;
      fprintf(msgFile, 
              "\n maxvalues of the columns in ascending order") ;
      DVfprintf(msgFile, ncolJ, maxvalues) ;
      DVfree(maxvalues) ;
   }
*/
}
/*
   -----------
   postprocess
   -----------
*/
MARKTIME(t1) ;
DFrontMtx_postProcess(frontmtx, 1, 1, msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n\n CPU %8.3f : post process the matrix", t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n fronts sorted and local indices set") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
}
/*
   --------------------------------------------------------------
   solve the linear system via multifrontal solve on multiple rhs
   --------------------------------------------------------------
*/
if ( symmetryflag == 0 ) {
   nops = 2*(frontmtx->upperDVL->tsize
           + frontmtx->diagDVL->tsize
           + frontmtx->upperDVL->tsize) ;
} else {
   nops = 2*(frontmtx->lowerDVL->tsize
           + frontmtx->diagDVL->tsize
           + frontmtx->upperDVL->tsize) ;
}
Bkeep    = DDenseMtx_new() ;
mtxB     = DA2_new() ;
mtxBkeep = DA2_new() ;
DDenseMtx_init(Bkeep, 2, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_setDA2(linsys->rhsmtx, mtxB) ;
DDenseMtx_setDA2(Bkeep, mtxBkeep) ;
DA2_copy(mtxBkeep, mtxB) ;

Xkeep    = DDenseMtx_new() ;
mtxX     = DA2_new() ;
mtxXkeep = DA2_new() ;
DDenseMtx_init(Xkeep, 2, -1, neqns, nrhs, 1, neqns) ;
DDenseMtx_setDA2(linsys->solmtx, mtxX) ;
DDenseMtx_setDA2(Xkeep, mtxXkeep) ;
DA2_copy(mtxXkeep, mtxX) ;

MARKTIME(t1) ;
manager = DDenseMtxManager_new() ;
DDenseMtxManager_init(manager, 0) ;
DFrontMtx_solveDA2(frontmtx, mtxX, mtxB, manager, cpus) ;
MARKTIME(t2) ;
DDenseMtxManager_writeForHumanEye(manager, msgFile) ;
DDenseMtxManager_free(manager) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n solution ") ;
   DA2_writeForHumanEye(mtxX, msgFile) ;
   fflush(msgFile) ;
}
DA2_sub(mtxX, mtxXkeep) ;
fprintf(msgFile,
     "\n CPU %8.3f : DFrontMtx solve, %8.3f mflops, %12.4e max error",
     t2 - t1, 1.e-6*nops*nrhs/(t2 - t1), DA2_maxabs(mtxX)) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n error ") ;
   DA2_writeForHumanEye(mtxX, msgFile) ;
   fflush(msgFile) ;
}
cputotal = cpus[7] ;
if ( cputotal > 0.0 ) {
   fprintf(msgFile,
   "\n    initialize rhs matrices      %8.3f %6.2f"
   "\n    load rhs matrices            %8.3f %6.2f"
   "\n    assemble children and parent %8.3f %6.2f"
   "\n    solve and update             %8.3f %6.2f"
   "\n    store entries                %8.3f %6.2f"
   "\n    link and free matrices       %8.3f %6.2f"
   "\n    miscellaneous                %8.3f %6.2f"
   "\n    total time                   %8.3f",
   cpus[0], 100.*cpus[0]/cputotal,
   cpus[1], 100.*cpus[1]/cputotal,
   cpus[2], 100.*cpus[2]/cputotal,
   cpus[3], 100.*cpus[3]/cputotal,
   cpus[4], 100.*cpus[4]/cputotal,
   cpus[5], 100.*cpus[5]/cputotal,
   cpus[6], 100.*cpus[6]/cputotal, cputotal) ;
}
/*
   --------------------------------------------
   convert from local indices to global indices
   --------------------------------------------
*/
MARKTIME(t1) ;
DFrontMtx_postProcess(frontmtx, 0, 2, msglvl, msgFile) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n\n CPU %8.3f : post process the matrix", t2 - t1) ;
if ( msglvl > 2 ) {
   fprintf(msgFile, "\n fronts sorted and local indices set") ;
   DFrontMtx_writeForHumanEye(frontmtx, msgFile) ;
}
/*
   ------------------------
   free the working storage
   ------------------------
*/
ETree_free(frontETree) ;
DLinSystem_free(linsys) ;
DDenseMtx_free(Xkeep) ;
DDenseMtx_free(Bkeep) ;
DA2_free(mtxB) ;
DA2_free(mtxBkeep) ;
DA2_free(mtxX) ;
DA2_free(mtxXkeep) ;
IV_free(oldToNewIV) ;
IVL_free(symbfacIVL) ;
DFrontMtx_free(frontmtx) ;

fprintf(msgFile, "\n") ;
fclose(msgFile) ;

return ; }

/*--------------------------------------------------------------------*/
