/*  DFrontMtx.h  */

#include "../DPencil.h"
#include "../ETree.h"
#include "../IVL.h"
#include "../DVL.h"
#include "../DStairMtx.h"
#include "../DA2.h"
#include "../DChvManager.h"
#include "../DChvList.h"
#include "../DDenseMtxManager.h"
#include "../DDenseMtxList.h"
#include "../DChv.h"
#include "../Ideq.h"
#include "../Lock.h"

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   nfront -- number of fronts in the matrix
   neqns  -- number of rows and columns in the matrix
   symmetryflag -- flag to specify symmetry of the matrix
      0 --> symmetric structure, symmetric entries
      1 --> symmetric structure, nonsymmetric entries
      2 --> nonsymmetric structure, nonsymmetric entries
      3 --> matrix from QR factorization
   sparsityflag -- flag to specify dense or sparse fronts
      0 --> dense fronts
      1 --> sparse fronts
   pivotingflag -- flag to specify pivoting enabled
      0 --> pivoting not used
      1 --> pivoting used
   frontETree -- pointer to an ETree object that holds the front tree
   symbfacIVL -- pointer to an IVL object that holds 
      the symbolic factorization
   frontsizesIV -- pointer to an IV object that holds the number of
      internal rows and columns in each front
   pivotsIVL  -- pointer to an IVL object that holds pivot sizes,
      used only for symmetric matrices with pivoting
   rowadjIVL  -- pointer to an IVL object that holds the row ids
      of the fronts, used only for nonsymmetric matrices with pivoting
   coladjIVL  -- pointer to an IVL object that holds the column ids
      of the fronts, used only with pivoting
   lowerIVL   -- pointer to an IVL object that holds the sparsity
      structure of the L matrix, used only for nonsymmetric matrices 
      with pivoting and sparse fronts
   upperIVL   -- pointer to an IVL object that holds the sparsity
      structure of the U matrix, used only with pivoting and sparse 
      fronts
   diagDVL  -- pointer to an DVL object that holds the entries in D
   lowerDVL -- pointer to an DVL object that holds the entries in L
   upperDVL -- pointer to an DVL object that holds the entries in U
   lock     -- mutex object that controls access to allocating
               storage in IVL and DVL objects, can be NULL
   nlocks   -- number of times the lock was locked

   created -- 97may12, cca
   -------------------------------------------------------------------
*/
typedef struct _DFrontMtx   DFrontMtx ;
struct _DFrontMtx {
   int       nfront        ;
   int       neqns         ;
   int       symmetryflag  ;
   int       sparsityflag  ;
   int       pivotingflag  ;
   ETree     *frontETree   ;
   IVL       *symbfacIVL   ;
   IV        *frontsizesIV ;
   IVL       *pivotsIVL    ;
   IVL       *rowadjIVL    ;
   IVL       *coladjIVL    ;
   IVL       *lowerIVL     ;
   IVL       *upperIVL     ;
   DVL       *diagDVL      ;
   DVL       *lowerDVL     ;
   DVL       *upperDVL     ;
   Lock      *lock         ;
   int       nlocks        ;
} ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   this object is used during a threaded factorization

   pencil    -- matrix pencil A + sigma * B
   tau       -- upper bound on factor entries when pivoting enabled
   droptol   -- drop tolerance used for sparse fronts
   ownersIV  -- map from fronts to threads
   lookahead -- parameter used to control computation lookahead
      0   --> no lookahead
      > 0 --> look up a number of levels up the tree

   frontmtx      -- object used to store factorization
   manager       -- object used to manage working DChv objects
   aggregatelist -- object used to store aggregate data
   postponedlist -- object used to store postponed data

   myid    -- thread id
   fronts  -- vector of pointers to active fronts
   stats   -- int statistics vector
   cpus    -- double vector to store breakdown of cpu times
   msglvl  -- message level
   msgFile -- message file

   created -- 97may30, cca
   ----------------------------------------------------------------
*/
typedef struct _DFactorData   DFactorData ;
struct _DFactorData {
/*
   -------------------------
   global data, not modified
   -------------------------
*/
   DPencil       *pencil   ;
   double        tau       ;
   double        droptol   ;
   int           *parent   ;
   IV            *ownersIV ;
   int           lookahead ;
/*
   ---------------------
   shared data, modified
   ---------------------
*/
   DFrontMtx     *frontmtx      ;
   DChvManager   *manager       ;
   DChvList      *aggregatelist ;
   DChvList      *postponedlist ;
/*
   ----------
   local data
   ----------
*/
   int           myid          ;
   DChv          **fronts      ;
   int           stats[10]     ;
   double        cpus[20]      ;
   int           msglvl        ;
   FILE          *msgFile      ;
} ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
  this object is used during a threaded solve A X = B
 
   frontmtx   -- object that stores the factors of A
   solDA2     -- object that stores X
   rhsDA2     -- object that stores B
   ownersIV   -- map from fronts to owning threads
   manager    -- object that manages the working DDenseMtx objects
   matrixList -- object that manages the lists of DDenseMtx objects
   p_mtxJ     -- vector of pointers to working DDenseMtx objects
   p_mtxBJ    -- vector of pointers to working DDenseMtx objects
   ndescLeft  -- vector that manages synchronization 
                 in the forward solve
   myid       -- logical id of local thread
   msglvl     -- message level
   msgFile    -- message file
   stats      -- statistics array
   cpus       -- cpu array
 
   created -- 97jun27, cca
   ----------------------------------------------------------------
*/
typedef struct _DSolveData   DSolveData ;
struct _DSolveData {
   DFrontMtx        *frontmtx   ;
   DA2              *solDA2     ;
   DA2              *rhsDA2     ;
   IV               *ownersIV   ;
   DDenseMtxManager *manager    ;
   DDenseMtxList    *matrixList ;
   DDenseMtx        **p_mtxBJ   ;
   DDenseMtx        **p_mtxJ    ;
   int              *ndescLeft  ;
   int              myid        ;
   int              msglvl      ;
   FILE             *msgFile    ;
   int              stats[10]   ;
   double           cpus[10]    ;
} ;
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
  this object is used during a threaded QR factorization
 
   inpmtxA    -- input matrix A
   rowsIVL    -- list of rows in A that goes into each front
   ownersIV   -- map from fronts to threads
   frontmtx   -- object used to store factorization
   manager    -- object used to manage working DChv objects
   updatelist -- object used to store update data
   facops     -- QR factorization ops. count
 
   myid    -- thread id
   fronts  -- vector of pointers to active fronts
   stats   -- int statistics vector
   cpus    -- double vector to store breakdown of cpu times
   msglvl  -- message level
   msgFile -- message file
 
   created -- 97aug25, dkw
   ----------------------------------------------------------------
*/
typedef struct _DQRFactorData   DQRFactorData ;
struct _DQRFactorData {
/*
   -------------------------
   global data, not modified
   -------------------------
*/
   DInpMtx       *inpmtxA  ;
   int           *parent   ;
   IVL           *rowsIVL  ;
   IV            *ownersIV ;
/*
   ---------------------
   shared data, modified
   ---------------------
*/
   DFrontMtx     *frontmtx   ;
   DChvManager   *manager    ;
   DChvList      *updatelist ;
   double        *facops     ;
/*
   ----------
   local data
   ----------
*/
   int           myid          ;
   int           stats[3]      ;
   double        cpus[10]      ;
   int           msglvl        ;
   FILE          *msgFile      ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97may12, cca
   -----------------------
*/
DFrontMtx *
DFrontMtx_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97may12, cca
   -----------------------
*/
void
DFrontMtx_setDefaultFields (
   DFrontMtx   *frontmtx
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97may12, cca
   --------------------------------------------------
*/
void
DFrontMtx_clearData (
   DFrontMtx   *frontmtx
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97may12, cca
   ------------------------------------------
*/
void
DFrontMtx_free (
   DFrontMtx   *frontmtx
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in instance.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------------
  simple method to return the dimensions of front J and the number
  of bytes necessary for the DChv object to hold the front.
 
   created -- 97nov13, cca
   ----------------------------------------------------------------
*/
void
DFrontMtx_initialFrontDimensions (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnD,
   int         *pnL,
   int         *pnU,
   int         *pnbytes
) ;
/*
   ---------------------------------------------------------
   return the number of internal rows and columns in front J
 
   created -- 97may12, cca
   ---------------------------------------------------------
*/
int
DFrontMtx_frontSize (
   DFrontMtx   *frontmtx,
   int         J
) ;
/*
   ------------------------------------------------------
   set the number of internal rows and columns in front J
 
   created -- 97may12, cca
   ------------------------------------------------------
*/
void
DFrontMtx_setFrontSize (
   DFrontMtx   *frontmtx,
   int         J,
   int         size
) ;
/*
   -----------------------------------------------------------------
   return the maximum number of internal rows and columns in a front
 
   created -- 97may12, cca
   -----------------------------------------------------------------
*/
int
DFrontMtx_maxFrontSize (
   DFrontMtx   *frontmtx
) ;
/*
   ------------------------------------
   return the number of rows in front J
 
   created -- 97may12, cca
   ------------------------------------
*/
int
DFrontMtx_nrowInFront (
   DFrontMtx   *frontmtx,
   int         J
) ;
/*
   --------------------------------------------
   return the maximum number of rows in a front
 
   created -- 97may12, cca
   --------------------------------------------
*/
int
DFrontMtx_maxNrowInFront (
   DFrontMtx   *frontmtx
) ;
/*
   ---------------------------------------
   return the number of columns in front J
 
   created -- 97may12, cca
   ---------------------------------------
*/
int
DFrontMtx_ncolInFront (
   DFrontMtx   *frontmtx,
   int         J
) ;
/*
   -----------------------------------------------
   return the maximum number of columns in a front
 
   created -- 97may12, cca
   -----------------------------------------------
*/
int
DFrontMtx_maxNcolInFront (
   DFrontMtx   *frontmtx
) ;
/*
   ---------------------------------------------
   fill *pncol with the number of columns and
   *pcolind with a pointer to the column indices
 
   created -- 97may12, cca
   ---------------------------------------------
*/
void
DFrontMtx_columnIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pncol,
   int         **pcolind
) ;
/*
   -------------------------------------------
   fill *pnrow with the number of rows and
   *prowind with a pointer to the rows indices
 
   created -- 97may12, cca
   -------------------------------------------
*/
void
DFrontMtx_rowIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnrow,
   int         **prowind
) ;
/*
   ----------------------------------------------
   fill *pnpivot with the number of npivots and
   *ppivotsizes with a pointer to the pivot sizes
 
   created -- 97may12, cca
   ----------------------------------------------
*/
void
DFrontMtx_pivotsizes (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnpivot,
   int         **ppivotsizes
) ;
/*
   ----------------------------------------------
   fill *pnent with the number of npivots and
   *pdiagent with a pointer to the pivot sizes
 
   created -- 97may12, cca
   ----------------------------------------------
*/
void
DFrontMtx_diagEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **pdiagent
) ;
/*
   ----------------------------------------------
   fill *pnent with the number of entries and
   *plowerent with a pointer to the lower entries
 
   created -- 97may12, cca
   ----------------------------------------------
*/
void
DFrontMtx_lowerEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **plowerent
) ;
/*
   ----------------------------------------------
   fill *pnent with the number of entries and
   *pupperent with a pointer to the upper entries
 
   created -- 97may12, cca
   ----------------------------------------------
*/
void
DFrontMtx_upperEntries (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnent,
   double      **pupperent
) ;
/*
   ----------------------------------------------------------
   fill *pnrow with the number of rows in J,
   *psizes with the sizes vector for the sparse rows of L
   *pindices with the column indices for the sparse rows of L
 
   created -- 97may12, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_lowerSparsityInfo (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pnrow,
   int         **psizes,
   int         **plowerind
) ;
/*
   ----------------------------------------------------------
   fill *pncol with the number of columns in J,
   *psizes with the sizes vector for the sparse columns of U
   *pindices with the row indices for the sparse columns of U
 
   created -- 97may12, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_upperSparsityInfo (
   DFrontMtx   *frontmtx,
   int         J,
   int         *pncol,
   int         **psizes,
   int         **pupperind
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- basic initializer
 
   frontETree   -- ETree object that stores the front tree
   symbfacIVL   -- IVL object that stores the symbolic factorization
   symmetryflag -- symmetry flag,
      0 --> matrix has symmetric structure, symmetric entries
      1 --> matrix has symmetric structure, nonsymmetric entries
      2 --> matrix has nonsymmetric structure, nonsymmetric entries
      3 --> matrix from QR factorization
   sparsityflag -- flag to specify dense or sparse fronts
      0 --> dense fronts
      1 --> sparse fronts
   pivotingflag -- flag to specify pivoting enabled
      0 --> pivoting not used
      1 --> pivoting used
 
   in a multithreaded environment, we need to protect the critical
   section where data is allocated. we use a lockflag to do this.
   in a serial or distributed environment, use lockflag = 0.
 
   lockflag -- flag to specify lock status
      0 --> mutex lock is not allocated or initialized
      1 --> mutex lock is allocated and it can synchronize
            only threads in this process.
      2 --> mutex lock is allocated and it can synchronize
            only threads in this and other processes.
 
   in a distributed environment we need to specify which process
   owns each front. when we can preallocate data structures
   (when there is no pivoting and dense fronts are stored) we
   need each process to determine what parts of the data it
   can allocate and set up. in a serial or multithreaded 
   environment, use ownersIV = NULL.
 
      ownersIV -- map from fronts to owning processes
      myid     -- id of this process.
 
   created  -- 97apr23, cca
   modified -- 97jul11, cca
      ownersIV and myid added
   ------------------------------------------------------------------
*/
void
DFrontMtx_init (
   DFrontMtx   *frontmtx,
   ETree       *frontETree,
   IVL         *symbfacIVL,
   int         symmetryflag,
   int         sparsityflag,
   int         pivotingflag,
   int         lockflag,
   int         myid,
   IV          *ownersIV
) ;
/*
   -----------------------------
   purpose -- initialize a front
 
   created -- 97may23, cca
   -----------------------------
*/
void
DFrontMtx_initializeFront (
   DFrontMtx   *frontmtx,
   DChv        *frontJ,
   int         J
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in loadEntries.c ----------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   load entries from sigma*A
 
   chv     -- pointer to the DChv object that holds the front
   pencil  -- pointer to a DPencil that holds the matrix entries 
   msglvl  -- message level
   msgFile -- message file
 
   created  -- 97jul18, cca
   ------------------------------------------------------------
*/
void
DFrontMtx_loadEntries (
   DChv      *chv,
   DPencil   *pencil,
   int       msglvl,
   FILE      *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in update.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   compute the updates to front J, store them in the DChv object
 
   created -- 97may13, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_update (
   DFrontMtx   *frontmtx,
   DChv        *frontJ,
   int         head[],
   int         link[],
   int         offsets[],
   DV          *tmpDV,
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in postponed.c ------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- to assemble any postponed data into frontJ
 
   frontJ  -- pointer to DChv objec that contains current front
   chvlist -- pointer to a DChvList object that handles the
              lists of postponed DChv objects
   manager -- pointer to a DChvManager object for the list
              of free DChv objects
   pndelay -- pointer to address to contain the # of delayed rows 
              and columns that were assembled into the front
 
   return value -- pointer to DChv object that contains the new front
 
   created -- 97may23, cca
   ------------------------------------------------------------------
*/
DChv *
DFrontMtx_assemblePostponedData (
   DFrontMtx     *frontmtx,
   DChv          *frontJ,
   DChvList      *chvlist,
   DChvManager   *manager,
   int           *pndelay
) ;
/*
   ---------------------------------------------------------
   purpose -- extract and store the postponed data
 
   frontJ  -- pointer to present front object
   npost   -- # of postponed rows and columns in frontJ
   K       -- parent of J
   chvlist -- pointer to a DChvList object that handles the
              lists of postponed DChv objects
              a singly linked list to assemble
   manager -- pointer to a DChvManager object for the list
              of free DChv objects
 
   created -- 97may23, cca
   ---------------------------------------------------------
*/
void
DFrontMtx_storePostponedData (
   DFrontMtx     *frontmtx,
   DChv          *frontJ,
   int           npost,
   int           K,
   DChvList      *chvlist,
   DChvManager   *manager
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in factor.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
   compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization of A.
   this is a wrapper method around DFrontMtx_factor().
 
   input --
 
      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file
 
   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factorDInpMtx (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtx,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------------------------
   compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization
   of A + sigma*B. this is a wrapper method around DFrontMtx_factor().
 
   input --
 
      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file
 
   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factorDPencil (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*

-------------------------------------------------------------------
  compute an (U^T + I)D(I + U) or (L + I)D(I + L) factorization
   of A - sigma*B.
 
   input --
 
      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      pencil   -- pointer to the DPencil object that holds A + sigma*B
      tau      -- upper bound on entries in L and U,
                  used only when pivoting enabled
      droptol  -- lower bound on entries in L and U,
                  used only when sparsity enabled
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- get updates from descendents
         cpus[3] -- assembled postponed data
         cpus[4] -- factor the front
         cpus[5] -- extract postponed data
         cpus[6] -- store factor entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
      stats[] -- statistics array
         stats[0] -- # of pivots
         stats[1] -- # of pivot tests
         stats[2] -- # of delayed rows and columns
         stats[3] -- # of entries in D
         stats[4] -- # of entries in L
         stats[5] -- # of entries in U
      msglvl   -- message level
      msgFile  -- message file
 
   created -- 97jul18, cca

-------------------------------------------------------------------
*/
DChv *
DFrontMtx_factor (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in storeFront.c -----------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------------
   purpose -- to store the factor entries from frontJ and
              link the front into the update list for its parent
 
   pivotsizesIV -- object that holds the number and sizes of the pivots
   droptol      -- drop tolerance for sparse factorizations
 
   created -- 97may23, cca
   --------------------------------------------------------------------
*/
void
DFrontMtx_storeFront (
   DFrontMtx   *frontmtx,
   DChv        *frontJ,
   IV          *pivotsizesIV,
   double      droptol
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in solve.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------------
  this method is used in the multithreaded and MPI parallel solves
 
   frontmtx -- front matrix object
   rhsDA2   -- right hand side matrix object,
      note, for multithreaded factorization, rhsDA2 is global
      note, for MPI factorization, rhsDA2 is local
   J       -- present front to solve
   manager -- manager of working dense matrices
   mtxList -- list object for storing descendent matrices
   p_mtxJ  -- array of pointers to internal matrix objects
   p_mtxBJ -- array of pointers to boundary matrix objects
   status  -- status vector for the fronts
      status[J] = 'W' --> initialize the data structures
      status[J] = 'R' --> load update, solve and update
      status[J] = 'F' --> front is finished
   cpus -- vector of CPU breakdowns
      cpus[0] --- initialize the fronts
      cpus[1] --- load rhs and solution
      cpus[2] --- assemble children and parents
      cpus[3] --- solve and update
      cpus[4] --- store rhs and solution
      cpus[5] --- store updates
   msglvl  -- message level
   msgFile -- message filek
 
   created -- 97nov15, cca
   ----------------------------------------------------------------
*/
void
DFrontMtx_parallelForwardSolve (
   DFrontMtx          *frontmtx,
   DA2                *rhsDA2,
   int                J,
   DDenseMtxManager   *manager,
   DDenseMtxList      *mtxList,
   DDenseMtx          *p_mtxJ[],
   DDenseMtx          *p_mtxBJ[],
   char               status[],
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) ;
/*
   ----------------------------------------------------------------
  this method is used in the multithreaded and MPI parallel solves
 
   frontmtx -- front matrix object
   rhsDA2   -- right hand side matrix object,
      note, for multithreaded factorization, rhsDA2 is global
      note, for MPI factorization, rhsDA2 is local
   solDA2   -- solution matrix object,
      note, for multithreaded factorization, solDA2 is global
      note, for MPI factorization, solDA2 is local
   J       -- present front to solve
   manager -- manager of working dense matrices
   mtxList -- list object for storing descendent matrices
   p_mtxJ  -- array of pointers to internal matrix objects
   status  -- status vector for the fronts
      status[J] = 'W' --> initialize the data structures
      status[J] = 'R' --> load update, solve and update
      status[J] = 'F' --> front is finished
   cpus -- vector of CPU breakdowns
      cpus[0] --- initialize the fronts
      cpus[1] --- load rhs and solution
      cpus[2] --- assemble children and parents
      cpus[3] --- solve and update
      cpus[4] --- store rhs and solution
      cpus[5] --- store updates
   msglvl  -- message level
   msgFile -- message filek
 
   created -- 97nov15, cca
   ----------------------------------------------------------------
*/
void
DFrontMtx_parallelBackwardSolve (
   DFrontMtx          *frontmtx,
   DA2                *rhsDA2,
   DA2                *solDA2,
   int                J,
   DDenseMtxManager   *manager,
   DDenseMtxList      *mtxList,
   DDenseMtx          *p_mtxJ[],
   char               status[],
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) ;
/*
   ----------------------------------------------------------
   initialize the two DDenseMtx objects for the forward solve
 
   created -- 97jun26, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_forwInit (
   DFrontMtx          *frontmtx,
   int                J,
   int                nrhs,
   DDenseMtxManager   *manager,
   DDenseMtx          **pmtxJ,
   DDenseMtx          **pmtxBJ
) ;
/*
   --------------------------------
   load the right hand side entries
 
   created -- 97jun26, cca
   --------------------------------
*/
void
DFrontMtx_forwLoadRHS (
   DDenseMtx          *mtxJ,
   DA2                *rhsDA2
) ;
/*

-------------------------------------------------------------------
  assemble updates from the children and release their matrix
objects
 
   created -- 97jun26, cca

-------------------------------------------------------------------
*/
void
DFrontMtx_forwLoadFromChildren (
   DDenseMtx          *mtxJ,
   DDenseMtx          *mtxBJ,
   DDenseMtx          *firstI
) ;
/*
   ---------------------------------------------------------
   solve (L_{J,J} + I) sol_J = rhs_J
   update  rhs_{bnd{J}} := rhs_{bnd{J}} - L_{bnd{J},J} sol_J
 
   created -- 97jun27, cca
   ---------------------------------------------------------
*/
void
DFrontMtx_forwSolveAndUpdate (
   DFrontMtx          *frontmtx,
   DDenseMtx          *mtxJ,
   DDenseMtx          *mtxBJ
) ;
/*
   ----------------------------------------
   store the solution entries and link the
   update object in the list for the parent
 
   created -- 97jun27, cca
   ----------------------------------------
*/
void
DFrontMtx_forwStore (
   DA2                *rhsDA2,
   DDenseMtx          *mtxJ
) ;
/*
   ------------------------------------------
   initialize the solution object for front J
 
   created -- 97jun27, cca
   ------------------------------------------
*/
DDenseMtx *
DFrontMtx_backInit (
   DFrontMtx          *frontmtx,
   int                J,
   int                nrhs,
   DDenseMtxManager   *manager
) ;
/*
   -------------------------------------------------------------
   initialize the solution object for front J
   1. get object from manager
   2. load rhs_J
   3. store column indices of J in the row indices of the object
 
   created -- 97jun27, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_backLoadSolution (
   DDenseMtx   *mtxJ,
   int         nDJ,
   int         rowindJ[],
   DA2         *rhsDA2
) ;
/*
   -----------------------------------------------
   if the parent exists, load the solution entries
   from the parent and release parent's object
 
   created -- 97jun27, cca
   -----------------------------------------------
*/
void
DFrontMtx_backLoadFromParent (
   DDenseMtx          *mtxJ,
   int                nDJ,
   DDenseMtx          *mtxK
) ;
/*
   --------------------------------------------------------------
   perform the update rhs_J := rhsJ - U_{J,nnd{J}} * sol_{bnd{J}}
   and solve (I + U_{J,J}) sol_J = rhs_J
 
   created -- 97jun27, cca
   --------------------------------------------------------------
*/
void
DFrontMtx_backSolveAndUpdate (
   DFrontMtx   *frontmtx,
   DDenseMtx   *mtxJ
) ;
/*
   ------------------------------------------------
   store the solution entries and place a copy of
   the solution matrix into each list for the child
 
   created -- 97jun27, cca
   ------------------------------------------------
*/
void
DFrontMtx_backStore (
   DDenseMtx          *mtxJ,
   int                nDJ,
   DA2                *solDA2
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in solveDA2.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------
   solve one of three
 
   symmetryflag = 0 
      (U^T + I) * D * (I + U) sol = rhs
      D has 1x1 and/or 2x2 pivots
   symmetryflag = 2 
      (L + I) * D * (I + U) sol = rhs
      D is diagonal
   symmetryflag = 3 
      (U^T + D) * (D + U) sol = rhs
      D is diagonal
   cpus     -- vector of cpu time breakdowns
      cpus[0] -- initialize rhs matrices
      cpus[1] -- load rhs matrices with rhs 
      cpus[2] -- assemble from children and parent
      cpus[3] -- solve and update
      cpus[4] -- store entries
      cpus[5] -- link and free matrices
      cpus[6] -- miscellaneous
      cpus[7] -- total time

   created -- 97jun05, cca
   ---------------------------------------
*/
void
DFrontMtx_solveDA2 (
   DFrontMtx          *frontmtx,
   DA2                *solDA2,
   DA2                *rhsDA2,
   DDenseMtxManager   *manager,
   double             cpus[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in solveMT.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   a multithreaded solve of a linear system A X = B 
 
   frontmtx -- object that stores the factors of A
   solDA2   -- object that stores the solution X
   rhsDA2   -- object that stores the solution B
   manager  -- object that manages working data
   ownersIV -- object that stores the map from fronts to threads
   cpus     -- vector of cpu time breakdowns
   stats    -- vector of statistics
   msglvl   -- message level
   msgFile  -- message file
 
   created -- 97jun27, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_MT_solve (
   DFrontMtx          *frontmtx,
   DA2                *solDA2,
   DA2                *rhsDA2,
   DDenseMtxManager   *manager,
   IV                 *ownersIV,
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
------ methods found in util.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
  create and return the willUpdate vector
   willUpdate[J] == 'Y' --> this thread or process myid owns J
                            or a descendent with support with J
   willUpdate[J] == 'N' --> this thread or process myid does not own J
                            or any descendent with support with J
                          
   created -- 97jul03, cca
   -------------------------------------------------------------------
*/
char *
DFrontMtx_willUpdate (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         myid
) ;
/*
   -------------------------------------------------------------------
   create and return the status vector
   status[J] == 'F' --> J is not on any active path for this thread
   status[J] == 'W' --> J is on any active path for this thread 
      
   created -- 97jul03, cca
   -------------------------------------------------------------------
*/
char *
DFrontMtx_status (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         myid
) ;
/*
   -----------------------------------------------
   create and return the nactiveChild vector.
   nactiveChild[J] contains the number of children 
   of J that belong to an active path
                         
   created -- 97jul03, cca
   -----------------------------------------------
*/
int *
DFrontMtx_nactiveChild (
   DFrontMtx   *frontmtx,
   char        *status,
   int         myid
) ;
/*
   -------------------------------------------
   create, initialize and return a Ideq object
   loaded with the leaves of the active paths.
 
   created -- 97jul03, cca
   -------------------------------------------
*/
Ideq *
DFrontMtx_setUpDequeue (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   char        *status,
   int         myid
) ;
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with postponed chevrons
 
   created -- 97jul03, cca
   -----------------------------------------------
*/
DChvList *
DFrontMtx_postList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) ;
/*
   -----------------------------------------------
   create, initialize and return a DChvList object
   to deal with aggregate chevrons
      
   created -- 97jul03, cca
   -----------------------------------------------
*/
DChvList *
DFrontMtx_aggregateList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) ;
/*
   ----------------------------------------------------
   create, initialize and return a DDenseMtxList object
   to deal with the dense matrices in a solve
      
   created -- 97jul28, cca
   ----------------------------------------------------
*/
DDenseMtxList *
DFrontMtx_solveList (
   DFrontMtx   *frontmtx,
   IV          *frontOwnersIV,
   int         lockflag
) ;
/*
   -----------------------------------------------
   return a DChv object that will hold the biggest 
   front for a factorization without pivoting
 
   created -- 97aug21, cca
   -----------------------------------------------
*/
DChv *
DFrontMtx_biggestFront (
   DFrontMtx   *frontmtx
) ;
/*
   -------------------------------------------------------------
   compute the inertia of a symmetric matrix
 
   fill *pnnegative with the number of negative eigenvalues of A
   fill *pnzero     with the number of zero eigenvalues of A
   fill *pnpositive with the number of positive eigenvalues of A
 
   created -- 97aug22, cca
   -------------------------------------------------------------
*/
void
DFrontMtx_inertia (
   DFrontMtx   *frontmtx,
   int         *pnnegative,
   int         *pnzero,
   int         *pnpositive
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------
   purpose -- to read in an DFrontMtx object from a file
 
   input --
 
      fn -- filename, must be *.frontmtxb or *.frontmtxf
 
   return value -- 1 if success, 0 if failure
 
   created -- 97jun18, cca
   -----------------------------------------------------
*/
int
DFrontMtx_readFromFile (
   DFrontMtx   *frontmtx,
   char        *fn
) ;
/*
   ------------------------------------------------------------
   purpose -- to read an DFrontMtx object from a formatted file
 
   return value -- 1 if success, 0 if failure
 
   created -- 97jun18, cca
   ------------------------------------------------------------
*/
int
DFrontMtx_readFromFormattedFile (
   DFrontMtx   *frontmtx,
   FILE        *fp
) ;
/*
   ---------------------------------------------------------
   purpose -- to read an DFrontMtx object from a binary file
 
   return value -- 1 if success, 0  if failure
 
   created -- 97jun18, cca
   ---------------------------------------------------------
*/
int
DFrontMtx_readFromBinaryFile (
   DFrontMtx   *frontmtx,
   FILE        *fp
);
/*
   -------------------------------------------------
   purpose -- to write an DFrontMtx object to a file
 
   input --
 
      fn -- filename
        *.frontmtxb -- binary
        *.frontmtxf -- formatted
        anything else -- for human eye
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jun18, cca
   -------------------------------------------------
*/
int
DFrontMtx_writeToFile (
   DFrontMtx   *frontmtx,
   char        *fn
) ;
/*
   -----------------------------------------------------------
   purpose -- to write an DFrontMtx object to a formatted file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jun18, cca
   -----------------------------------------------------------
*/
int
DFrontMtx_writeToFormattedFile (
   DFrontMtx   *frontmtx,
   FILE        *fp
) ;
/*
   --------------------------------------------------------
   purpose -- to write an DFrontMtx object to a binary file
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jun18, cca
   --------------------------------------------------------
*/
int
DFrontMtx_writeToBinaryFile (
   DFrontMtx    *frontmtx,
   FILE         *fp
) ;
/*
   ---------------------------------------------------------------
   purpose -- to write out the statistics for the DFrontMtx object
 
   return value -- 1 if success, 0 otherwise
 
   created -- 97jun18, cca
   ---------------------------------------------------------------
*/
int
DFrontMtx_writeStats (
   DFrontMtx   *frontmtx,
   FILE        *fp
) ;
/*
   ----------------------------------------
   purpose -- to write the object to a file
              in human readable form
 
   created -- 97may16, cca
   ----------------------------------------
*/
int
DFrontMtx_writeForHumanEye (
   DFrontMtx   *frontmtx,
   FILE     *fp
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in factorMT.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
   parallel factorization method for A.
   all but two input parameters are the same as the serial method.
   this is a wrapper method around DFrontMtx_factorMT().
 
   ownersIV  -- pointer to IV object that holds map from fronts
                to threads
   lookahead -- lookahead parameter that allows computation at
                higher levels of the front tree to proceed when
                lower fronts are not yet finish. use lookahead = 0
                to turn off this option. otherwise lookahead ancestors
                of an active unfinished front can be active.
 
   return value -- pointer to the first DChv object in a list
                   that contains postponed data
 
   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MT_factorDInpMtx (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtx,
   double      tau,
   double      droptol,
   IV          *ownersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------------------------
   parallel factorization method for A + sigma*B.
   all but two input parameters are the same as the serial method.
   this is a wrapper method around DFrontMtx_factorMT().
 
   ownersIV  -- pointer to IV object that holds map from fronts 
                to threads
   lookahead -- lookahead parameter that allows computation at
                higher levels of the front tree to proceed when
                lower fronts are not yet finish. use lookahead = 0
                to turn off this option. otherwise lookahead ancestors
                of an active unfinished front can be active.
 
   return value -- pointer to the first DChv object in a list
                   that contains postponed data
 
   created -- 97nov08, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_MT_factorDPencil (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   IV          *ownersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------------------------
   parallel factorization method.
   all but two input parameters are the same as the serial method.

   ownersIV  -- pointer to IV object that holds map from fronts 
                to threads
   lookahead -- lookahead parameter that allows computation at
                higher levels of the front tree to proceed when
                lower fronts are not yet finish. use lookahead = 0
                to turn off this option. otherwise lookahead ancestors
                of an active unfinished front can be active.

   return value -- pointer to the first DChv object in a list
                   that contains postponed data
 
   created -- 97may30, cca
   -------------------------------------------------------------------
*/
DChv *
DFrontMtx_factorMT (
   DFrontMtx   *frontmtx,
   DPencil     *pencil,
   double      tau,
   double      droptol,
   IV          *ownersIV,
   int         lookahead,
   double      cpus[],
   int         stats[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in checkFront.c ------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------
   purpose -- check out and take action on a front
 
   created -- 97may29, cca
   -----------------------------------------------
*/
char
DFrontMtx_checkFront (
   DFrontMtx     *frontmtx,
   DFactorData   *data,
   int           J,
   int           lookahead,
   char          status[],
   char          willUpdate[],
   int           nactiveChild[],
   int           head[],
   int           link[],
   int           offsets[],
   IV            *pivotsizesIV,
   IV            *markIV,
   DV            *tmpDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in postProcess.c -----------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------
   purpose -- postprocess the DFrontMtx object
 
   permuteflag == 1 --> used when pivoting enabled
     (1) permute the columns of U_{J,bnd{J}} so they are in
         ascending order w.r.t. to global column numbering
     (2) permute the rows of L_{bnd{J},J} so they are in
         ascending order w.r.t. to global row numbering
   localflag == 0 --> do nothing
   localflag == 1 --> global-to-local,
      overwrite row and column indices bnd{J}
      with indices that are local w.r.t. J's parent
   localflag == 2 --> local-to-global,
      overwrite local row and column indices bnd{J}
      with global indices 
 
   created -- 97nov20, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_postProcess (
   DFrontMtx   *frontmtx,
   int         permuteflag,
   int         localflag,
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in permuteFactor.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------
   purpose -- 
     (1) permute the columns of U_{J,bnd{J}} so they are in
         ascending order w.r.t. to global column numbering
     (2) permute the rows of L_{bnd{J},J} so they are in
         ascending order w.r.t. to global row numbering
 
   created -- 97nov20, cca
   ----------------------------------------------------------
*/
void
DFrontMtx_permuteFactor (
   DFrontMtx   *frontmtx,
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -----------------------------------------------------
   permute the columns in U_{J,bnd{J}} so that the 
   columns are in a compatible ordering with its parent.
 
   created -- 97nov21, cca
   -----------------------------------------------------
*/
void
DFrontMtx_permuteColumnsOfU (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         ncolK,
   int         colindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------
   permute the rows in L_{bnd{J},J} so that the rows 
   are in a compatible ordering with its parent.
 
   created -- 97nov21, cca
   -------------------------------------------------
*/
void
DFrontMtx_permuteRowsOfL (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         nrowK,
   int         rowindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in localInd.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------------
   purpose -- 
 
   overwrite row and column indices bnd{J}
   with indices that are local w.r.t. J's parent
 
   created -- 97nov20, cca
   ---------------------------------------------
*/
void
DFrontMtx_globalToLocalInd (
   DFrontMtx   *frontmtx,
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------------
   convert global column indices into local column indices
 
   created -- 97nov21, cca
   -------------------------------------------------------
*/
void
DFrontMtx_setLocalColumnIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         ncolK,
   int         colindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------
   convert global row indices into local row indices
 
   created -- 97nov21, cca
   -------------------------------------------------
*/
void
DFrontMtx_setLocalRowIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         nrowK,
   int         rowindK[],
   int         map[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in globalInd.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------
   purpose -- 
 
      overwrite local row and column indices bnd{J}
      with global indices 
 
   created -- 97nov20, cca
   ------------------------------------------------
*/
void
DFrontMtx_localToGlobalInd (
   DFrontMtx   *frontmtx,
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------------
   convert local column indices into global column indices
 
   created -- 97nov21, cca
   -------------------------------------------------------
*/
void
DFrontMtx_setGlobalColumnIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         ncolK,
   int         colindK[],
   int         msglvl,
   FILE        *msgFile
) ;
/*
   -------------------------------------------------
   convert local row indices into global row indices
 
   created -- 97nov21, cca
   -------------------------------------------------
*/
void
DFrontMtx_setGlobalRowIndices (
   DFrontMtx   *frontmtx,
   int         J,
   int         K,
   int         nrowK,
   int         rowindK[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in DFactorData.c -----------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97may30, cca
   -----------------------
*/
DFactorData *
DFactorData_new (
   void
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97may30, cca
   -----------------------
*/
void
DFactorData_setDefaultFields (
   DFactorData   *data
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97may30, cca
   --------------------------------------------------
*/
void
DFactorData_clearData (
   DFactorData   *data
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97may30, cca
   ------------------------------------------
*/
void
DFactorData_free (
   DFactorData   *data
) ;
/*
   -----------------------
   basic initializer
 
   created -- 97may30, cca
   -----------------------
*/
void
DFactorData_init (
   DFactorData   *data,
   DFrontMtx     *frontmtx,
   DPencil       *pencil,
   double        tau,
   double        droptol,
   IV            *ownersIV,
   int           lookahead,
   DChvManager   *manager,
   DChvList      *aggregatelist,
   DChvList      *postponedlist
) ;
/*
   -----------------------------------------------------
   set the thread id, the message level and message file
 
   created -- 97may30
   -----------------------------------------------------
*/
void
DFactorData_setInfo (
   DFactorData   *data,
   int           myid,
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in DSolveData.c ------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97jun27, cca
   -----------------------
*/
DSolveData *
DSolveData_new (
   void
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97jun27, cca
   -----------------------
*/
void
DSolveData_setDefaultFields (
   DSolveData   *data
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97jun27, cca
   --------------------------------------------------
*/
void
DSolveData_clearData (
   DSolveData   *data
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97jun27, cca
   ------------------------------------------
*/
void
DSolveData_free (
   DSolveData   *data
) ;
/*
   -----------------------
   basic initializer
 
   created -- 97jun27, cca
   -----------------------
*/
void
DSolveData_init (
   DSolveData         *data,
   DFrontMtx          *frontmtx,
   DA2                *solDA2,
   DA2                *rhsDA2,
   IV                 *ownersIV,
   DDenseMtxManager   *manager,
   DDenseMtxList      *matrixList,
   DDenseMtx          **p_mtxJ,
   DDenseMtx          **p_mtxBJ,
   int                *ndescLeft
) ;
/*
   -----------------------------------------------------
   set the thread id, the message level and message file
 
   created -- 97jun27
   -----------------------------------------------------
*/
void
DSolveData_setInfo (
   DSolveData   *data,
   int           myid,
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in DQRFactorData.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97aug25, dkw
   -----------------------
*/
DQRFactorData *
DQRFactorData_new (
   void
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97aug25, dkw
   -----------------------
*/
void
DQRFactorData_setDefaultFields (
   DQRFactorData   *data
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97aug25, dkw
   --------------------------------------------------
*/
void
DQRFactorData_clearData (
   DQRFactorData   *data
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97aug25, dkw
   ------------------------------------------
*/
void
DQRFactorData_free (
   DQRFactorData   *data
) ;
/*
   -----------------------
   basic initializer
 
   created -- 97aug25, dkw
   -----------------------
*/
void
DQRFactorData_init (
   DQRFactorData  *data,
   DFrontMtx     *frontmtx,
   DInpMtx       *inpmtxA,
   IVL           *rowsIVL,
   IV            *ownersIV,
   DChvManager   *manager,
   DChvList      *updatelist,
   double        *facops
) ;
/*
   -----------------------------------------------------
   set the thread id, the message level and message file
 
   created -- 97may30
   -----------------------------------------------------
*/
void
DQRFactorData_setInfo (
   DQRFactorData  *data,
   int           myid,
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRassembleUpdates.c -----------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   purpose -- given a stairfront that has had its original
      entries assembled, incorporate any updates from its
      children fronts.
 
   created -- 97apr16, dkw
   -------------------------------------------------------------
*/
void
DFrontMtx_QRassembleUpdates (
   DStairMtx     *stairfrontJ,
   int           J,
   IVL           *symbfacIVL,
   IVL           *rowsIVL,
   DChvManager   *manager,
   DChv          *firstchild,
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRcheckFront.c ----------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------
   purpose -- check out and take action on a front
 
   created -- 97aug22, dkw
   -----------------------------------------------
*/
void
DFrontMtx_QRcheckFront (
   DFrontMtx      *frontmtx,
   DQRFactorData  *data,
   int            J,
   char           status[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRfactor.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------
   serial QR factorization
   compute the cholesky factor of A^T*A
 
   input --
 
      frontmtx -- pointer to the DFrontMtx object that will hold
                  the factorization
      inpmtxA  -- pointer to the DInpMtx object that holds the
                  entries of A, cannot be NULL
 
      facops   -- QR factorization ops count
 
      cpus[]   -- timing array
         cpus[0] -- initialize fronts
         cpus[1] -- load original entries
         cpus[2] -- assembled updates
         cpus[3] -- form staircase and lastrowIV
         cpus[4] -- factor the front
         cpus[5] -- store factor entries
         cpus[6] -- store update entries
         cpus[7] -- miscellaneous time
         cpus[8] -- total time
 
      msglvl   -- message level
      msgFile  -- message file
 
      created -- 97apr02, dkw
   ----------------------------------------------------
*/
void
DFrontMtx_QRfactor (
   DFrontMtx     *frontmtx,
   DInpMtx       *inpmtxA,
   double        *facops,
   double        cpus[],
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRfactorFront.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------------
   procedure to factor a front
 
   stairfront      -- front DStairMtx object
   J               -- front ID
   facops          -- QR factorization ops. count
   msglvl          -- message level
   msgFile         -- message file
 
   created -- 97apr17, dkw
   --------------------------------------------------------------------
*/
void
DFrontMtx_QRfactorFront (
   DStairMtx  *stairfront,
   int        J,
   double     *facops,
   int        msglvl,
   FILE       *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRfactorMT.c ------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------------
   parallel QR factorization method.
   all but two input parameters are the same as the serial method.
 
   ownersIV  -- pointer to IV object that holds map from fronts
                to threads
 
   created -- 97aug20, dkw
   -------------------------------------------------------------------
*/
void
DFrontMtx_MT_QRfactor (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtxA,
   IV          *ownersIV,
   double      *facops,
   double      cpus[],
   int         msglvl,
   FILE        *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRloadEntries.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   load original entries from A
 
   stairfront -- pointer to the StairMtx object 
                 that holds the stairfront
   J          -- front ID
   inpmtxA    -- pointer to a DInpMtx that holds the entries of A
   symbfacIVL -- symbolic factorization IVL
   rowsIVL    -- list of rows in A that goes into each front
   msglvl     -- message level
   msgFile    -- message file
 
   created -- 97apr09, dkw
   ------------------------------------------------------------
*/
void
DFrontMtx_QRloadEntries (
   DStairMtx *stairfront,
   int       J,
   DInpMtx   *inpmtxA,
   IVL       *symbfacIVL,
   IVL       *rowsIVL,
   int       msglvl,
   FILE      *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRsolve.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------
   solve (U^T + D) * (D + U) sol = rhs
      D is diagonal
   by calling DFrontMtx_solveDA2 providing
   sparsityflag = 0 and symmetryflag = 3
   being initialize earlier in the test driver
 
   cpus     -- vector of cpu time breakdowns
      cpus[0] -- initialize rhs matrices
      cpus[1] -- load rhs matrices with rhs
      cpus[2] -- assemble from children and parent
      cpus[3] -- solve and update
      cpus[4] -- store entries
      cpus[5] -- link and free matrices
      cpus[6] -- miscellaneous
      cpus[7] -- total time
 
   created -- 97may27, dkw
   --------------------------------------------
*/
void
DFrontMtx_QRsolve (
   DFrontMtx   *frontmtx,
   DInpMtx     *inpmtxA,
   IV          *newToOldIV,
   DA2         *solDA2,
   DA2         *rhsDA2,
   double      cpus[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRsolveMT.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------------
   a multithreaded solve of a linear system A X = B
 
   solve (U^T + D) * (D + U) sol = rhs
      D is diagonal
 
   frontmtx   -- object that stores the factors of A
   inpmtxA    -- object that stores the input matix of A
   newToOldIV -- object that stores the new-to-old permutation of A
   solDA2     -- object that stores the solution X
   rhsDA2     -- object that stores the solution B
   ownersIV   -- object that stores the map from fronts to threads
   cpus       -- vector of cpu time breakdowns
      cpus[0] -- initialize rhs matrices
      cpus[1] -- load rhs matrices with rhs
      cpus[2] -- assemble from children and parent
      cpus[3] -- solve and update
      cpus[4] -- store entries
      cpus[5] -- link and free matrices
      cpus[6] -- miscellaneous
      cpus[7] -- total time
   msglvl   -- message level
   msgFile  -- message file
 
   created -- 97jun27, dkw
   -------------------------------------------------------------
*/
void
DFrontMtx_MT_QRsolve (
   DFrontMtx          *frontmtx,
   DInpMtx            *inpmtxA,
   IV                 *newToOldIV,
   DA2                *solDA2,
   DA2                *rhsDA2,
   IV                 *ownersIV,
   double             cpus[],
   int                msglvl,
   FILE               *msgFile
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRstoreFactor.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------
   extract the QR factor entries into the frontmtx
 
   created -- 97may21, dkw
   -------------------------------------------------
*/
void
DFrontMtx_QRstoreFactor (
   DFrontMtx   *frontmtx,
   DA2         *factormtx,
   int         J,
   int         msglvl,
   FILE        *msgFile
 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRstoreUpdate.c ---------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------------
   procedure to store the update entries into a Chveron object
 
   factormtx       -- DA2 object that contains the update matrix
   updatelist      -- DChvList object
   manager         -- DChvManager object
   J               -- front ID
   K               -- parent of J
   nJ              -- number of interior node
   nDU             -- number of interior and exterior nodes
   nupdrow         -- number of update rows
   colsJ           -- column index
   msglvl          -- message level
   msgFile         -- message file
 
   created -- 97apr17, dkw
   --------------------------------------------------------------------
*/
void
DFrontMtx_QRstoreUpdate (
   DA2           *factormtx,
   DChvList      *updatelist,
   DChvManager   *manager,
   int           J,
   int           K,
   int           nJ,
   int           nDU,
   int           nupdrow,
   int           *colsJ,
   int           msglvl,
   FILE          *msgFile
) ;
/*--------------------------------------------------------------------*/
