/*  init.c  */

#include "../DDenseMtx.h"

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   return the number of bytes needed for an object of this size
 
   created -- 97aug15, cca
   ------------------------------------------------------------
*/
int
DDenseMtx_nbytesNeeded (
   int   nrow,
   int   ncol
) {
int   nbytes, ndouble, nint ;
/*
   ---------------
   check the input
   ---------------
*/
if (  nrow < 0 || ncol < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DDenseMtx_nbytesNeeded(%d,%d)"
           "\n bad input\n", nrow, ncol) ;
   exit(-1) ;
}
nint    = 6 + nrow + ncol ;
ndouble = nrow*ncol ;
if ( sizeof(int) == sizeof(double) ) {
   nbytes = nint*sizeof(int) + ndouble*sizeof(double) ;
} else if ( 2*sizeof(int) == sizeof(double) ) {
   nbytes = ((nint + 1)/2 + ndouble)*sizeof(double) ;
} else {
   fprintf(stderr, "\n error in DDenseMtx_nbytesNeeded(%d,%d)"
           "\n sizeof(int) = %d, sizeof(double) = %d",
           nrow, ncol, sizeof(int), sizeof(double)) ;
   exit(-1) ;
}
return(nbytes) ; }
 
/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------------
   return the number of bytes in the workspace owned by this object
 
   created -- 97jun08, cca
   ----------------------------------------------------------------
*/
int
DDenseMtx_nbytesInWorkspace (
   DDenseMtx   *mtx
) {
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DDenseMtx_nbytesInWorkspace(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
return(sizeof(double)*DV_maxsize(&mtx->wrkDV)) ; }
 
/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   set the number of bytes in the workspace owned by this object
 
   created -- 97aug23, cca
   -------------------------------------------------------------
*/
void
DDenseMtx_setNbytesInWorkspace (
   DDenseMtx   *mtx,
   int         nbytes
) {
if ( mtx == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DDenseMtx_setNbytesInWorkspace(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
DV_setSize(&mtx->wrkDV, nbytes/sizeof(double)) ;

return ; }
 
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   purpose -- set the fields of the object

   created -- 97aug23, cca
   ---------------------------------------
*/
void
DDenseMtx_setFields (
   DDenseMtx   *mtx,
   int         rowid,
   int         colid,
   int         nrow,
   int         ncol,
   int         inc1,
   int         inc2
) {
double   *dbuffer ;
int      *ibuffer ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || nrow < 0 || ncol < 0 
   || !((inc1 == ncol && inc2 == 1) || (inc1 == 1 && inc2 == nrow)) ) {
   fprintf(stderr, 
           "\n fatal error in DDenseMtx_setFields(%p,%d,%d,%d,%d,%d,%d)"
           "\n bad input\n", 
           mtx, rowid, colid, nrow, ncol, inc1, inc2) ;
   exit(-1) ;
}
dbuffer = DV_entries(&mtx->wrkDV) ;
ibuffer = (int *) dbuffer ;
/*
   ---------------------
   set the scalar fields
   ---------------------
*/
mtx->rowid = ibuffer[0] = rowid ;
mtx->colid = ibuffer[1] = colid ;
mtx->nrow  = ibuffer[2] = nrow  ;
mtx->ncol  = ibuffer[3] = ncol  ;
mtx->inc1  = ibuffer[4] = inc1  ;
mtx->inc2  = ibuffer[5] = inc2  ;
/*
   -------------------
   set up the pointers
   -------------------
*/
mtx->rowind = ibuffer + 6 ;
mtx->colind = mtx->rowind + nrow ;
if ( sizeof(int) == sizeof(double) ) {
   mtx->entries = dbuffer + 6 + nrow + ncol ;
} else if ( 2*sizeof(int) == sizeof(double) ) {
   mtx->entries = dbuffer + (7 + nrow + ncol)/2 ;
}
/*
fprintf(stdout, 
        "\n rowind - ibuffer = %d" 
        "\n colind - rowind  = %d" 
        "\n entries - dbuffer = %d", 
        mtx->rowind - ibuffer,
        mtx->colind - mtx->rowind,
        mtx->entries - dbuffer) ;
*/

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   purpose -- basic initializer

   created -- 97jun08, cca
   ----------------------------
*/
void
DDenseMtx_init (
   DDenseMtx     *mtx,
   int           rowid,
   int           colid,
   int           nrow,
   int           ncol,
   int           inc1,
   int           inc2
) {
int   nbytes ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || nrow < 0 || ncol < 0 
   || !((inc1 == ncol && inc2 == 1) || (inc1 == 1 && inc2 == nrow)) ) {
   fprintf(stderr, 
           "\n fatal error in DDenseMtx_init(%p,%d,%d,%d,%d,%d,%d)"
           "\n bad input\n", 
           mtx, rowid, colid, nrow, ncol, inc1, inc2) ;
   exit(-1) ;
}
/*
   -------------------------------------------------------
   get and set the number of bytes needed in the workspace
   -------------------------------------------------------
*/
nbytes = DDenseMtx_nbytesNeeded(nrow, ncol) ;
DDenseMtx_setNbytesInWorkspace(mtx, nbytes) ;
/*
   --------------
   set the fields
   --------------
*/
DDenseMtx_setFields(mtx, rowid, colid, nrow, ncol, inc1, inc2) ;
IVramp(nrow, mtx->rowind, 0, 1) ;
IVramp(ncol, mtx->colind, 0, 1) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- initialize the object from its working storage
              used when the object is a MPI message

   created -- 97jul17, cca
   ---------------------------------------------------------
*/
void
DDenseMtx_initFromBuffer (
   DDenseMtx   *mtx
) {
int   *ibuffer ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DDenseMtx_initFromBuffer(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
ibuffer   = (int *) DV_entries(&mtx->wrkDV) ;
DDenseMtx_setFields(mtx, ibuffer[0], ibuffer[1], 
                    ibuffer[2], ibuffer[3], ibuffer[4], ibuffer[5]) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------
   purpose -- initializer with pointers

   created -- 97jun08, cca
   ------------------------------------
*/
void
DDenseMtx_initWithPointers (
   DDenseMtx   *mtx,
   int         rowid,
   int         colid,
   int         nrow,
   int         ncol,
   int         inc1,
   int         inc2,
   int         *rowind,
   int         *colind,
   double      *entries
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || nrow <= 0 || ncol <= 0 || inc1 < 0 || inc2 < 0
   || (inc1 != 1 && inc2 != 1) 
   || entries == NULL || colind == NULL || rowind == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DDenseMtx_initWithPointers()"
           "\n mtx = %p, rowid = %d, colid = %d"
           "\n nrow = %d, ncol = %d, inc1 = %d, inc2 = %d"
           "\n rowind = %p, colind = %p, entries = %p "
           "\n bad input\n",
           mtx, rowid, colid, nrow, ncol, inc1, inc2, 
           rowind, colind, entries) ;
   exit(-1) ;
}
/*
   ---------------------
   set the scalar fields
   ---------------------
*/
mtx->rowid = rowid ;
mtx->colid = colid ;
mtx->nrow  = nrow  ;
mtx->ncol  = ncol  ;
mtx->inc1  = inc1  ;
mtx->inc2  = inc2  ;
/*
   --------------------------
   set up the working storage
   --------------------------
*/
mtx->rowind  = rowind  ;
mtx->colind  = colind  ;
mtx->entries = entries ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------
   this method initializes a DA2 object 
   to point into the entries

   created -- 97jun08, cca
   ------------------------------------
*/
void
DDenseMtx_setDA2 (
   DDenseMtx   *mtx,
   DA2         *da2
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || da2 == NULL ) {
   fprintf(stderr, "\n fatal error in DDenseMtx_setDA2(%p,%p)"
           "\n bad input\n", mtx, da2) ;
   exit(-1) ;
}
DA2_init(da2, mtx->nrow, mtx->ncol, mtx->inc1, mtx->inc2,
         mtx->entries) ;

return ; }

/*--------------------------------------------------------------------*/
