/*  DChvList.h  */

#include "../DChv.h"
#include "../Lock.h"

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------------
   this object handles a list of lists of DChv objects

   nlist  -- # of lists
   heads  -- heads[ilist] contains a pointer 
             to the first DChv object in list ilist
   counts -- when not-NULL, counts[ilist] contains the remaining number 
             of objects to be added to list ilist before it is complete
   lock   -- mutex object, can be NULL
   flags  -- when not NULL, a vector to specify when a list needs 
             to be locked before adding an object to it.
      flags[ilist] = 'N' --> no need to lock
      flags[ilist] = 'Y' --> must lock
   nlocks -- number of times the list was locked
   --------------------------------------------------------------------
*/
typedef struct _DChvList   DChvList ;
struct _DChvList {
   int       nlist   ;
   DChv      **heads ;
   int       *counts ;
   Lock      *lock   ;
   char      *flags  ;
   int       nlocks  ;
} ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 97may24, cca
   -----------------------
*/
DChvList *
DChvList_new (
   void
) ;
/*
   -----------------------
   set the default fields
 
   created -- 97may24, cca
   -----------------------
*/
void
DChvList_setDefaultFields (
   DChvList   *chvlist
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 97may24, cca
   --------------------------------------------------
*/
void
DChvList_clearData (
   DChvList   *chvlist
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 97may24, cca
   ------------------------------------------
*/
void
DChvList_free (
   DChvList   *chvlist
) ;

/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- basic initializer
 
   nlist  -- number of lists to be held by this object
   counts -- vector that contains number of items expected
             for each list.
      counts == NULL --> unknown number of items expected
      counts != NULL --> known number of items expected
   lockflag -- flag to specify lock status
      lockflag = 0 --> mutex lock is not allocated or initialized
      lockflag = 1 --> mutex lock is allocated and it can synchronize
                       only threads in this process.
      lockflag = 2 --> mutex lock is allocated and it can synchronize
                       threads in this and other processes.
   flags -- vector to specify whether to lock individual lists
      flags == NULL --> none or all lists must be locked,
                        use lockflag to determine
      flags[ilist] = 'N' --> no need to lock list ilist
      flags[ilist] = 'Y' --> must lock list ilist
 
   created -- 97may24, cca
   ------------------------------------------------------------------
*/
void
DChvList_init (
   DChvList   *chvlist,
   int        nlist,
   int        counts[],
   int        lockflag,
   char       flags[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------
   purpose -- to write the object to a file
              in human readable form
 
   created -- 97may24, cca
   ----------------------------------------
*/
void
DChvList_writeForHumanEye (
   DChvList   *chvlist,
   FILE       *fp
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------
   return 1 if list ilist is not empty
   return 0 if list ilist is empty
 
   created -- 97may24, cca
   -----------------------------------
*/
int
DChvList_isListNonempty (
   DChvList   *chvlist,
   int        ilist
) ;
/*
   ---------------------------------------------------------
   return 1 if the count for list ilist is zero
   return 0 if the count for list ilist is greater than zero
 
   created -- 97may24, cca
   ---------------------------------------------------------
*/
int
DChvList_isCountZero (
   DChvList   *chvlist,
   int        ilist
) ;
/*
   -----------------------
   add chv to list ilist
 
   created -- 97may24, cca
   -----------------------
*/
void
DChvList_addObjectToList (
   DChvList   *chvlist,
   DChv       *chv,
   int        ilist
) ;
/*
   ------------------------------------
   return pointer to head of list ilist
   and set head to NULL
 
   created -- 97may24, cca
   ------------------------------------
*/
DChv *
DChvList_getList (
   DChvList   *chvlist,
   int        ilist
) ;
/*--------------------------------------------------------------------*/
