/*  util.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   shift the indices, entries and adjust the nD dimension.
   note: shift can be positive or negative

   created -- 97apr26, cca
   -------------------------------------------------------
*/
void
DChv_shift (
   DChv   *chv,
   int    shift
) {
int   ii, stride ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_shift(%p,%d)"
           "\n bad input\n", chv, shift) ;
   exit(-1) ;
}
if ( shift == 0 ) {
   return ;
}
if ( chv->symflag == 0 ) {
/*
   --------------------
   chevron is symmetric
   --------------------
*/
   chv->colind += shift ;
   if ( shift < 0 ) {
      stride = chv->nD + chv->nU + 1 ;
      for ( ii = shift ; ii < 0 ; ii++ ) {
         chv->entries -= stride ;
         stride++ ;
      }
   } else {
      stride = chv->nD + chv->nU ;
      for ( ii = 0 ; ii < shift ; ii++ ) {
         chv->entries += stride ;
         stride-- ;
      }
   }
   chv->nD -= shift ;
} else {
/*
   -----------------------
   chevron is nonsymmetric
   -----------------------
*/
   chv->rowind += shift ;
   chv->colind += shift ;
   if ( shift < 0 ) {
      stride = 2*chv->nD + chv->nL + chv->nU + 1 ;
      for ( ii = shift ; ii < 0 ; ii++ ) {
         chv->entries -= stride ;
         stride += 2 ;
      }
   } else {
      stride = 2*chv->nD + chv->nL + chv->nU - 1 ;
      for ( ii = 0 ; ii < shift ; ii++ ) {
         chv->entries += stride ;
         stride -= 2 ;
      }
   }
   chv->nD -= shift ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------
   return the value of entry (irow, jcol)

   created -- 97apr26, cca
   --------------------------------------
*/
double
DChv_entry (
   DChv   *chv,
   int    irow,
   int    jcol
) {
double   value ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_entry(%p,%d,%d)"
           "\n bad input\n", chv, irow, jcol) ;
   exit(-1) ;
}
if ( chv->symflag == 0 ) {
   int   nD, nU ;
/*
   --------------------
   chevron is symmetric
   --------------------
*/
   nD = chv->nD ;
   nU = chv->nU ;
   if ( jcol >= nD + nU ) {
      fprintf(stderr, "\n fatal error in DChv_entry(%p,%d,%d)"
              "\n jcol = %d, nD + nU = %d\n", 
              chv, irow, jcol, jcol, nD + nU) ;
      exit(-1) ;
   }
   if ( irow > nD ) {
      value = 0.0 ;
   } else {
      value = chv->entries[irow*(nD + nU) 
                           - (irow*(irow-1))/2 + jcol - irow] ;
   }
} else {
   int   ii, nD, nL, nU, off ;
/*
   -----------------------
   chevron is nonsymmetric
   -----------------------
*/
   nD  = chv->nD ;
   nL  = chv->nL ;
   nU  = chv->nU ;
   ii  = (irow <= jcol) ? irow : jcol ;
   off = jcol - irow ;
   value = chv->entries[(2*ii+1)*nD + (ii+1)*nL + ii*nU 
                        - ii*ii - ii - 1 + jcol - irow] ;
}
return(value) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------
   return a pointer to the entry (irow, jcol)

   created -- 97apr30, cca
   ------------------------------------------
*/
double *
DChv_pointerToEntry (
   DChv   *chv,
   int    irow,
   int    jcol
) {
double   *entries ;
int      nD, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_pointerToEntry(%p,%d,%d)"
           "\n bad input\n", chv, irow, jcol) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
if ( irow >= nD && jcol >= nD ) {
   fprintf(stderr, "\n fatal error in DChv_pointerToEntry(%p,%d,%d)"
           "\n irow = %d, jcol = %d, nD = %d\n", 
           chv, irow, jcol, irow, jcol, nD) ;
   exit(-1) ;
}
entries = DChv_entries(chv) ;
if ( chv->symflag == 0 ) {
/*
   --------------------
   chevron is symmetric
   --------------------
*/
   if ( jcol >= nD + nU ) {
      fprintf(stderr, "\n fatal error in DChv_pointerToEntry(%p,%d,%d)"
              "\n jcol = %d, nD + nU = %d\n", 
              chv, irow, jcol, jcol, nD + nU) ;
      exit(-1) ;
   }
   if ( irow > nD ) {
      return(NULL) ;
   } else {
      return(&chv->entries[irow*(nD + nU) 
                           - (irow*(irow-1))/2 + jcol - irow]) ;
   }
} else {
   int   ii, off ;
/*
   -----------------------
   chevron is nonsymmetric
   -----------------------
*/
   ii  = (irow <= jcol) ? irow : jcol ;
   off = jcol - irow ;
   return(&chv->entries[(2*ii+1)*nD + (ii+1)*nL + ii*nU 
                        - ii*ii - ii - 1 + jcol - irow]) ;
}
return(NULL) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   return the maximum magnitude of the entries in the chevron

   created -- 97may15, cca
   ----------------------------------------------------------
*/
double
DChv_maxabs (
   DChv   *chv 
) {
int   loc, nD, nent, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_maxabs(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
if ( chv->symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
return(DVmaxabs(nent, DChv_entries(chv), &loc)) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------
   return the frobenius norm of the entries in the chevron

   created -- 97may15, cca
   -------------------------------------------------------
*/
double
DChv_frobNorm (
   DChv   *chv 
) {
int   nD, nent, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_frobNorm(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
if ( chv->symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
return(sqrt(DVdot(nent, DChv_entries(chv), DChv_entries(chv)))) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   subtract chvI from chvJ

   created -- 97may15, cca
   -----------------------
*/
void
DChv_sub (
   DChv   *chvJ,
   DChv   *chvI 
) {
double   *entriesI, *entriesJ ;
int      ii, nDI, nDJ, nent, nLI, nLJ, nUI, nUJ ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvI == NULL || chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_sub(%p,%p)"
           "\n bad input\n", chvI, chvJ) ;
   exit(-1) ;
}
DChv_dimensions(chvJ, &nDJ, &nLJ, &nUJ) ;
DChv_dimensions(chvI, &nDI, &nLI, &nUI) ;
if ( nDJ != nDI || nLJ != nLI || nUJ != nUI ) {
   fprintf(stderr, "\n fatal error in DChv_sub(%p,%p)"
           "\n dimensions do not match\n", chvJ, chvI) ;
   exit(-1) ;
}
entriesJ = DChv_entries(chvJ) ;
entriesI = DChv_entries(chvI) ;
if ( entriesJ == NULL || entriesI == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_sub(%p,%p)"
           "\n entriesJ = %p, entriesI = %p\n", 
           chvJ, chvI, entriesJ, entriesI) ;
   exit(-1) ;
}
if ( chvJ->symflag == 0 ) {
   nent = (nDJ*(nDJ+1))/2 + nDJ*nUJ ;
} else {
   nent = nDJ*(nDJ + nLJ + nUJ) ;
}
for ( ii = 0 ; ii < nent ; ii++ ) {
   entriesJ[ii] -= entriesI[ii] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------
   zero the entries in the chevron

   created -- 97may15, cca
   -------------------------------
*/
void
DChv_zero (
   DChv   *chv
) {
double   *entries ;
int      ii, nD, nent, nL, nU ;

DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
if ( entries == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_zero(%p)"
           "\n entries = %p",
           chv, entries) ;
   exit(-1) ;
}
if ( chv->symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
for ( ii = 0 ; ii < nent ; ii++ ) {
   entries[ii] = 0.0 ;
}
return ; }

/*--------------------------------------------------------------------*/
